<?php
/**
 * Dashboard widget for WPO
 *
 * @package MUSTWP
 */

declare(strict_types=1);

namespace MUSTWP;

/**
 * Dashboard widget class for performance overview
 */
class Dashboard {
    
    /**
     * Verdict severity levels for comparison (lower = worse)
     */
    private const VERDICT_SEVERITY = [
        'no_good' => 1,                    // Worst - No content performs well
        'majority_poor' => 2,              // Most content performs poorly
        'majority_needs_improvement' => 3, // Most content needs improvement
        'mixed' => 4,                      // Mixed performance
        'majority_good' => 5               // Best - Most content performs well
    ];
    
    /**
     * Device status levels for overall verdict comparison (lower = worse)
     * 
     * Strong: >= 70% Good AND Poor <= 5%
     * Mixed: 30-69% Good AND Poor <= 20%
     * Weak: < 30% Good OR Poor > 20%
     */
    private const DEVICE_STATUS = [
        'weak' => 1,   // Worst
        'mixed' => 2,
        'strong' => 3  // Best
    ];
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        add_action('wp_dashboard_setup', [$this, 'register_dashboard_widget']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_dashboard_assets']);
    }
    
    /**
     * Register dashboard widget
     */
    public function register_dashboard_widget(): void {
        wp_add_dashboard_widget(
            'mustwp_overview',
            __('Mustang WPO Overview', 'mustang-wpo'),
            [$this, 'render_widget']
        );
    }
    
    /**
     * Enqueue dashboard assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_dashboard_assets(string $hook): void {
        if ($hook !== 'index.php') {
            return;
        }
        
        wp_enqueue_style(
            'mustwp-dashboard',
            MUSTWP_PLUGIN_URL . 'build/css/dashboard.css',
            [],
            mustwp_get_asset_version(MUSTWP_PLUGIN_DIR . 'build/css/dashboard.css')
        );
        
        wp_enqueue_script(
            'mustwp-dashboard',
            MUSTWP_PLUGIN_URL . 'build/js/dashboard.js',
            [],
            mustwp_get_asset_version(MUSTWP_PLUGIN_DIR . 'build/js/dashboard.js'),
            true
        );
    }
    
    /**
     * Render dashboard widget
     */
    public function render_widget(): void {
        // State C: Check API key first
        $api_key = $this->settings['api_key'] ?? '';
        if (empty($api_key)) {
            $this->render_empty_state('no_api_key');
            return;
        }
        
        // State D: Check post types selection
        $post_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        $post_types = array_intersect($post_types, $allowed_types);
        
        if (empty($post_types)) {
            $this->render_empty_state('no_post_types');
            return;
        }
        
        // State B: Check for published content
        if (!$this->has_published_content($post_types)) {
            $this->render_empty_state('no_content');
            return;
        }
        
        // Get aggregated performance data
        $data = $this->get_performance_data($post_types);
        
        // State A: Check for audit data
        if ($data['total_posts'] === 0) {
            $this->render_empty_state('no_audits');
            return;
        }
        
        // Get post type labels
        $post_type_labels = $this->get_content_type_labels($post_types);
        
        // Get last audit timestamp
        $last_audit_timestamp = $this->get_last_audit_timestamp($post_types);
        
        // Render widget HTML
        $this->render_widget_html($data, $post_type_labels, $post_types, $last_audit_timestamp);
    }
    
    /**
     * Check if there is any published content for the given post types
     *
     * @param array $post_types Post types to check
     * @return bool True if published content exists
     */
    private function has_published_content(array $post_types): bool {
        foreach ($post_types as $type) {
            $counts = wp_count_posts($type);
            if ($counts && isset($counts->publish) && $counts->publish > 0) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Render empty state message
     *
     * @param string $state The empty state type
     */
    private function render_empty_state(string $state): void {
        echo '<div class="mustwp-dashboard-empty-state">';
        
        switch ($state) {
            case 'no_api_key':
                printf(
                    // translators: %s is a link to the integrations page
                    __('Connect your Google PageSpeed Insights API key to start measuring performance. %s to get started.', 'mustang-wpo'),
                    '<a href="' . esc_url(admin_url('admin.php?page=mustwp-integrations')) . '">' . 
                    esc_html__('Set up your API key', 'mustang-wpo') . '</a>'
                );
                break;
                
            case 'no_post_types':
                printf(
                    // translators: %s is a link to the settings page
                    __('No post types selected for auditing. %s to get started.', 'mustang-wpo'),
                    '<a href="' . esc_url(admin_url('admin.php?page=mustwp-settings')) . '">' . 
                    esc_html__('Configure your settings', 'mustang-wpo') . '</a>'
                );
                break;
                
            case 'no_content':
                esc_html_e("You don't have any published posts yet. Your performance scores will appear here after you publish your first post or page!", 'mustang-wpo');
                break;
                
            case 'no_audits':
                printf(
                    // translators: %s is a link to the global audit page
                    __("You haven't audited any posts yet. Your performance scores will appear here after you run your first %s!", 'mustang-wpo'),
                    '<a href="' . esc_url(admin_url('admin.php?page=mustwp-audit')) . '">' . 
                    esc_html__('Global Audit', 'mustang-wpo') . '</a>'
                );
                break;
        }
        
        echo '</div>';
    }
    
    /**
     * Get the most recent audit timestamp
     *
     * @param array $content_types Content types to query
     * @return int|null Most recent timestamp or null if no audits found
     */
    private function get_last_audit_timestamp(array $content_types): ?int {
        // Note: meta_query is necessary here to filter posts with audit data
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
        $post_ids = get_posts([
            'post_type' => $content_types,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => [
                [
                    'key' => '_mustwp_pagespeed',
                    'compare' => 'EXISTS'
                ]
            ]
        ]);
        
        if (empty($post_ids)) {
            return null;
        }
        
        $latest_timestamp = 0;
        
        foreach ($post_ids as $post_id) {
            $results = get_post_meta($post_id, '_mustwp_pagespeed', true);
            
            if (empty($results) || !is_array($results)) {
                continue;
            }
            
            // Check mobile timestamp
            if (isset($results['mobile']['timestamp'])) {
                $timestamp = (int) $results['mobile']['timestamp'];
                if ($timestamp > $latest_timestamp) {
                    $latest_timestamp = $timestamp;
                }
            }
            
            // Check desktop timestamp
            if (isset($results['desktop']['timestamp'])) {
                $timestamp = (int) $results['desktop']['timestamp'];
                if ($timestamp > $latest_timestamp) {
                    $latest_timestamp = $timestamp;
                }
            }
        }
        
        return $latest_timestamp > 0 ? $latest_timestamp : null;
    }
    
    /**
     * Get aggregated performance data for all audited content
     *
     * @param array $content_types Content types to query
     * @return array Aggregated performance data
     */
    private function get_performance_data(array $content_types): array {
        global $wpdb;
        
        $data = [
            'desktop' => [
                'poor' => 0,
                'needs_improvement' => 0,
                'good' => 0,
                'total' => 0,
                'by_type' => []
            ],
            'mobile' => [
                'poor' => 0,
                'needs_improvement' => 0,
                'good' => 0,
                'total' => 0,
                'by_type' => []
            ],
            'total_posts' => 0
        ];
        
        // Initialize by_type arrays
        foreach ($content_types as $post_type) {
            $data['desktop']['by_type'][$post_type] = [
                'poor' => 0,
                'needs_improvement' => 0,
                'good' => 0,
                'total' => 0
            ];
            $data['mobile']['by_type'][$post_type] = [
                'poor' => 0,
                'needs_improvement' => 0,
                'good' => 0,
                'total' => 0
            ];
        }
        
        // Query all published posts of selected content types
        // Note: meta_query is necessary here to filter posts with audit data
        // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
        $post_ids = get_posts([
            'post_type' => $content_types,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => [
                [
                    'key' => '_mustwp_pagespeed',
                    'compare' => 'EXISTS'
                ]
            ]
        ]);
        
        if (empty($post_ids)) {
            return $data;
        }
        
        // Count total unique posts audited
        $data['total_posts'] = count($post_ids);
        
        // Process each post's performance data
        foreach ($post_ids as $post_id) {
            $post_type = get_post_type($post_id);
            $results = get_post_meta($post_id, '_mustwp_pagespeed', true);
            
            if (empty($results) || !is_array($results)) {
                continue;
            }
            
            // Process desktop data
            if (isset($results['desktop']['score'])) {
                $score = (int) round($results['desktop']['score']);
                $category = $this->categorize_score($score);
                
                $data['desktop'][$category]++;
                $data['desktop']['total']++;
                
                if (isset($data['desktop']['by_type'][$post_type])) {
                    $data['desktop']['by_type'][$post_type][$category]++;
                    $data['desktop']['by_type'][$post_type]['total']++;
                }
            }
            
            // Process mobile data
            if (isset($results['mobile']['score'])) {
                $score = (int) round($results['mobile']['score']);
                $category = $this->categorize_score($score);
                
                $data['mobile'][$category]++;
                $data['mobile']['total']++;
                
                if (isset($data['mobile']['by_type'][$post_type])) {
                    $data['mobile']['by_type'][$post_type][$category]++;
                    $data['mobile']['by_type'][$post_type]['total']++;
                }
            }
        }
        
        return $data;
    }
    
    /**
     * Categorize score into performance category
     *
     * @param int $score PageSpeed score
     * @return string Category (poor, needs_improvement, good)
     */
    private function categorize_score(int $score): string {
        if ($score >= 90) {
            return 'good';
        } elseif ($score >= 50) {
            return 'needs_improvement';
        } else {
            return 'poor';
        }
    }
    
    /**
     * Get content type labels
     *
     * @param array $content_types Content type slugs
     * @return array Content type labels
     */
    private function get_content_type_labels(array $content_types): array {
        $labels = [];
        
        foreach ($content_types as $post_type) {
            $post_type_object = get_post_type_object($post_type);
            if ($post_type_object) {
                $labels[$post_type] = $post_type_object->label;
            } else {
                $labels[$post_type] = ucfirst($post_type);
            }
        }
        
        return $labels;
    }
    
    /**
     * Get the verdict level for device data
     *
     * Returns a verdict level key for comparison purposes.
     * Evaluated in strict hierarchical order - first matching rule wins.
     *
     * @param array $device_data Device performance data
     * @return string Verdict level key
     */
    private function get_verdict_level(array $device_data): string {
        $total = $device_data['total'];
        if ($total === 0) {
            return 'mixed'; // Default fallback for empty data
        }
        
        $good = $device_data['good'];
        $needs_improvement = $device_data['needs_improvement'];
        $poor = $device_data['poor'];
        
        // Calculate percentages as decimals (0.0 to 1.0)
        $g_pct = $good / $total;
        $n_pct = $needs_improvement / $total;
        $p_pct = $poor / $total;
        
        // Rule A: No Good Content (G == 0)
        if ($good === 0) {
            return 'no_good';
        }
        
        // Rule B: Majority Poor (p% >= 0.5)
        if ($p_pct >= 0.5) {
            return 'majority_poor';
        }
        
        // Rule C: Majority Needs Improvement (n% >= 0.5 AND G > 0)
        // Note: G > 0 is already guaranteed since we passed Rule A
        if ($n_pct >= 0.5) {
            return 'majority_needs_improvement';
        }
        
        // Rule D: Majority Good (g% >= 0.5)
        if ($g_pct >= 0.5) {
            return 'majority_good';
        }
        
        // Rule E: Mixed Performance (Fallback)
        return 'mixed';
    }
    
    /**
     * Generate performance verdict sentence
     *
     * Follows strict hierarchical rules:
     * 1. No Good Content (G == 0)
     * 2. Majority Poor (p% >= 0.5)
     * 3. Majority Needs Improvement (n% >= 0.5 AND G > 0)
     * 4. Majority Good (g% >= 0.5)
     * 5. Mixed Performance (fallback)
     *
     * Strong Majority Modifier: When any percentage >= 0.8, replace "Most" with "Almost all"
     *
     * @param array $device_data Device performance data
     * @param string $device Device type (mobile/desktop)
     * @return string Verdict sentence
     */
    private function generate_performance_verdict(array $device_data, string $device): string {
        $total = $device_data['total'];
        if ($total === 0) {
            return '';
        }
        
        $good = $device_data['good'];
        $needs_improvement = $device_data['needs_improvement'];
        $poor = $device_data['poor'];
        
        // Calculate percentages as decimals (0.0 to 1.0)
        $g_pct = $good / $total;
        $n_pct = $needs_improvement / $total;
        $p_pct = $poor / $total;
        
        $device_label = $device === 'mobile' ? esc_html__('Mobile', 'mustang-wpo') : esc_html__('Desktop', 'mustang-wpo');
        
        // Get the verdict level
        $verdict_level = $this->get_verdict_level($device_data);
        
        // Determine if we should use "Almost all" modifier (any percentage >= 0.8)
        $use_strong_modifier = ($p_pct >= 0.8 || $n_pct >= 0.8 || $g_pct >= 0.8);
        
        switch ($verdict_level) {
            case 'no_good':
                // Rule A: No content performs well
                return sprintf(
                    // translators: %s is the device type (Mobile or Desktop)
                    __('No content performs well on %s.', 'mustang-wpo'),
                    $device_label
                );
                
            case 'majority_poor':
                // Rule B: All/Almost all/Most content performs poorly
                if ($p_pct >= 1.0) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('All content performs poorly on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                if ($use_strong_modifier) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('Almost all content performs poorly on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                return sprintf(
                    // translators: %s is the device type (Mobile or Desktop)
                    __('Most content performs poorly on %s.', 'mustang-wpo'),
                    $device_label
                );
                
            case 'majority_needs_improvement':
                // Rule C: All/Almost all/Most content needs improvement
                if ($n_pct >= 1.0) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('All content needs improvement on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                if ($use_strong_modifier) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('Almost all content needs improvement on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                return sprintf(
                    // translators: %s is the device type (Mobile or Desktop)
                    __('Most content needs improvement on %s.', 'mustang-wpo'),
                    $device_label
                );
                
            case 'majority_good':
                // Rule D: All/Almost all/Most content performs well
                // Check for 100% good first
                if ($g_pct >= 1.0) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('All content performs well on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                if ($use_strong_modifier) {
                    return sprintf(
                        // translators: %s is the device type (Mobile or Desktop)
                        __('Almost all content performs well on %s.', 'mustang-wpo'),
                        $device_label
                    );
                }
                return sprintf(
                    // translators: %s is the device type (Mobile or Desktop)
                    __('Most content performs well on %s.', 'mustang-wpo'),
                    $device_label
                );
                
            case 'mixed':
            default:
                // Rule E: Mixed performance (fallback)
                return sprintf(
                    // translators: %s is the device type (Mobile or Desktop)
                    __('Content performance is mixed on %s.', 'mustang-wpo'),
                    $device_label
                );
        }
    }
    
    /**
     * Get the device status classification for overall verdict comparison
     *
     * Classification rules:
     * - Strong: >= 70% Good AND Poor <= 5%
     * - Mixed: 30-69% Good AND Poor <= 20%
     * - Weak: < 30% Good OR Poor > 20%
     *
     * @param array $device_data Device performance data
     * @return string Device status: 'strong', 'mixed', or 'weak'
     */
    private function get_device_status(array $device_data): string {
        $total = $device_data['total'];
        if ($total === 0) {
            return 'weak'; // No data = weak
        }
        
        $good = $device_data['good'];
        $poor = $device_data['poor'];
        
        // Calculate percentages as decimals (0.0 to 1.0)
        $good_pct = $good / $total;
        $poor_pct = $poor / $total;
        
        // Strong: >= 70% Good AND Poor <= 5%
        if ($good_pct >= 0.70 && $poor_pct <= 0.05) {
            return 'strong';
        }
        
        // Weak: < 30% Good OR Poor > 20%
        if ($good_pct < 0.30 || $poor_pct > 0.20) {
            return 'weak';
        }
        
        // Mixed: 30-69% Good AND Poor <= 20% (everything else)
        return 'mixed';
    }
    
    /**
     * Generate overall site verdict comparing Mobile vs Desktop
     *
     * Returns verdict and optional directional hint based on device status comparison.
     *
     * Cases:
     * 1. Mobile weaker than Desktop: "Mobile performance is holding back..."
     * 1a. Mobile Weak + Desktop Strong: "Mobile performance is significantly limiting..."
     * 2. Desktop weaker than Mobile: "Desktop performance is trailing behind..."
     * 3. Both Strong: "Your site performs well across both..."
     * 4. Both Mixed: "Performance is inconsistent across..."
     * 5. Both Weak: "Overall site performance needs improvement..."
     *
     * @param array $mobile_data Mobile performance data
     * @param array $desktop_data Desktop performance data
     * @return array{verdict: string, hint: string|null} Verdict and optional hint
     */
    private function generate_overall_verdict(array $mobile_data, array $desktop_data): array {
        // Get device status for both
        $mobile_status = $this->get_device_status($mobile_data);
        $desktop_status = $this->get_device_status($desktop_data);
        
        // Get severity values (lower = worse)
        $mobile_severity = self::DEVICE_STATUS[$mobile_status] ?? 2;
        $desktop_severity = self::DEVICE_STATUS[$desktop_status] ?? 2;
        
        $verdict = '';
        $hint = null;
        
        // Case 1: Mobile is weaker than Desktop
        if ($mobile_severity < $desktop_severity) {
            // Case 1a: Mobile Weak AND Desktop Strong - use "significantly"
            if ($mobile_status === 'weak' && $desktop_status === 'strong') {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
                $verdict = __('Mobile performance is significantly limiting overall site performance.', 'mustang-wpo');
            } else {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
                $verdict = __('Mobile performance is holding back overall site performance.', 'mustang-wpo');
            }
            // Add directional hint for mobile
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
            $hint = __('Improving mobile performance will have the biggest impact.', 'mustang-wpo');
        }
        // Case 2: Desktop is weaker than Mobile
        elseif ($desktop_severity < $mobile_severity) {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
            $verdict = __('Desktop performance is trailing behind mobile performance.', 'mustang-wpo');
            // Add directional hint for desktop
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
            $hint = __('Improving desktop performance will have the biggest impact.', 'mustang-wpo');
        }
        // Both devices have the same status
        else {
            // Case 3: Both Strong
            if ($mobile_status === 'strong') {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
                $verdict = __('Your site performs well across both mobile and desktop.', 'mustang-wpo');
                // No hint needed - positive state
            }
            // Case 4: Both Mixed
            elseif ($mobile_status === 'mixed') {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
                $verdict = __('Performance is inconsistent across your site and devices.', 'mustang-wpo');
                // No hint - no clear leverage point
            }
            // Case 5: Both Weak
            else {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Escaped at output time with esc_html()
                $verdict = __('Overall site performance needs improvement across mobile and desktop.', 'mustang-wpo');
                // No hint when both weak - would oversimplify
            }
        }
        
        return [
            'verdict' => $verdict,
            'hint' => $hint
        ];
    }
    
    /**
     * Render widget HTML
     *
     * @param array $data Performance data
     * @param array $content_type_labels Content type labels
     * @param array $content_types Content type slugs
     * @param int|null $last_audit_timestamp Last audit timestamp
     */
    private function render_widget_html(array $data, array $content_type_labels, array $content_types, ?int $last_audit_timestamp = null): void {
        // Format content types for display with bold tags
        $content_types_bold = array_map(function($type) use ($content_type_labels) {
            $label = $content_type_labels[$type] ?? ucfirst($type);
            return '<strong>' . esc_html($label) . '</strong>';
        }, $content_types);
        
        // Format content types list with commas and "and" for last item
        $content_types_formatted = '';
        $count = count($content_types_bold);
        if ($count === 1) {
            $content_types_formatted = $content_types_bold[0];
        } elseif ($count === 2) {
            $content_types_formatted = $content_types_bold[0] . ' and ' . $content_types_bold[1];
        } else {
            $last = array_pop($content_types_bold);
            $content_types_formatted = implode(', ', $content_types_bold) . ' and ' . $last;
        }
        
        $total_posts = $data['total_posts'];
        
        // Generate overall site verdict (compares Mobile vs Desktop)
        $overall_verdict_data = $this->generate_overall_verdict($data['mobile'], $data['desktop']);
        $verdict = $overall_verdict_data['verdict'];
        $hint = $overall_verdict_data['hint'];
        
        // Build the context sentence
        $context_sentence = '';
        if ($last_audit_timestamp && $total_posts > 0) {
            $timezone = wp_timezone();
            $formatted_date = wp_date(
                get_option('date_format') . ', ' . get_option('time_format'),
                $last_audit_timestamp,
                $timezone
            );
            
            $context_sentence = sprintf(
                // translators: %1$d is the number of posts, %2$s is formatted list of content types with bold tags, %3$s is the formatted date/time
                _n(
                    'Based on %1$d published %2$s, last audited %3$s.',
                    'Based on %1$d published %2$s, last audited %3$s.',
                    $total_posts,
                    'mustang-wpo'
                ),
                $total_posts,
                $content_types_formatted,
                $formatted_date
            );
        } elseif ($total_posts > 0) {
            $context_sentence = sprintf(
                // translators: %1$d is the number of posts, %2$s is formatted list of content types with bold tags
                _n(
                    'Based on %1$d published %2$s.',
                    'Based on %1$d published %2$s.',
                    $total_posts,
                    'mustang-wpo'
                ),
                $total_posts,
                $content_types_formatted
            );
        }
        ?>
        <div class="mustwp-dashboard-widget">
            <div class="mustwp-dashboard-header">
                <h3 class="mustwp-dashboard-title"><?php echo esc_html($verdict); ?></h3>
                <?php if ($hint): ?>
                    <p class="mustwp-directional-hint"><?php echo esc_html($hint); ?></p>
                <?php endif; ?>
                <?php if ($context_sentence): ?>
                    <div class="mustwp-dashboard-meta"><?php echo wp_kses_post($context_sentence); ?></div>
                <?php endif; ?>
            </div>
            
            <?php if ($data['mobile']['total'] > 0 || $data['desktop']['total'] > 0): ?>
                <!-- Mobile Section -->
                <div class="mustwp-device-section">
                    <h3 class="mustwp-device-heading"><?php esc_html_e('Mobile', 'mustang-wpo'); ?></h3>
                    <div class="mustwp-device-content">
                        <?php
                        $mobile_data = $data['mobile'];
                        $mobile_total = $mobile_data['total'];
                        $mobile_poor_pct = $mobile_total > 0 ? ($mobile_data['poor'] / $mobile_total) * 100 : 0;
                        $mobile_needs_improvement_pct = $mobile_total > 0 ? ($mobile_data['needs_improvement'] / $mobile_total) * 100 : 0;
                        $mobile_good_pct = $mobile_total > 0 ? ($mobile_data['good'] / $mobile_total) * 100 : 0;
                        ?>
                        <?php if ($mobile_total > 0): ?>
                            <?php
                            // Generate tooltips for bar segments
                            $mobile_poor_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Poor: %d piece of content scores between 0-49.', 'Poor: %d pieces of content score between 0-49.', $mobile_data['poor'], 'mustang-wpo'),
                                $mobile_data['poor']
                            );
                            $mobile_needs_improvement_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Needs improvement: %d piece of content scores between 50-89.', 'Needs improvement: %d pieces of content score between 50-89.', $mobile_data['needs_improvement'], 'mustang-wpo'),
                                $mobile_data['needs_improvement']
                            );
                            $mobile_good_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Good: %d piece of content scores between 90-100.', 'Good: %d pieces of content score between 90-100.', $mobile_data['good'], 'mustang-wpo'),
                                $mobile_data['good']
                            );
                            ?>
                            <div class="mustwp-dashboard-chart">
                                <div class="mustwp-stacked-bar">
                                    <div class="mustwp-bar-segment mustwp-bar-poor" 
                                         style="width: <?php echo esc_attr($mobile_poor_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($mobile_poor_pct); ?>"
                                         title="<?php echo esc_attr($mobile_poor_tooltip); ?>"></div>
                                    <div class="mustwp-bar-segment mustwp-bar-needs-improvement" 
                                         style="width: <?php echo esc_attr($mobile_needs_improvement_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($mobile_needs_improvement_pct); ?>"
                                         title="<?php echo esc_attr($mobile_needs_improvement_tooltip); ?>"></div>
                                    <div class="mustwp-bar-segment mustwp-bar-good" 
                                         style="width: <?php echo esc_attr($mobile_good_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($mobile_good_pct); ?>"
                                         title="<?php echo esc_attr($mobile_good_tooltip); ?>"></div>
                                </div>
                            </div>
                            
                            <div class="mustwp-dashboard-legend">
                                <div class="mustwp-legend-item" data-category="poor">
                                    <span class="mustwp-legend-dot mustwp-dot-poor"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Poor', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($mobile_data['poor']); ?></span>
                                </div>
                                <div class="mustwp-legend-item" data-category="needs_improvement">
                                    <span class="mustwp-legend-dot mustwp-dot-needs-improvement"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Needs improvement', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($mobile_data['needs_improvement']); ?></span>
                                </div>
                                <div class="mustwp-legend-item" data-category="good">
                                    <span class="mustwp-legend-dot mustwp-dot-good"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Good', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($mobile_data['good']); ?></span>
                                </div>
                            </div>
                            
                            <?php
                            // Generate summary verdict for Mobile
                            $mobile_verdict = $this->generate_performance_verdict($mobile_data, 'mobile');
                            ?>
                            <p class="mustwp-device-verdict"><?php echo esc_html($mobile_verdict); ?></p>
                        <?php else: ?>
                            <p class="mustwp-dashboard-empty">
                                <?php esc_html_e('No mobile audit data available.', 'mustang-wpo'); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <hr class="mustwp-device-separator">
                
                <!-- Desktop Section -->
                <div class="mustwp-device-section">
                    <h3 class="mustwp-device-heading"><?php esc_html_e('Desktop', 'mustang-wpo'); ?></h3>
                    <div class="mustwp-device-content">
                        <?php
                        $desktop_data = $data['desktop'];
                        $desktop_total = $desktop_data['total'];
                        $desktop_poor_pct = $desktop_total > 0 ? ($desktop_data['poor'] / $desktop_total) * 100 : 0;
                        $desktop_needs_improvement_pct = $desktop_total > 0 ? ($desktop_data['needs_improvement'] / $desktop_total) * 100 : 0;
                        $desktop_good_pct = $desktop_total > 0 ? ($desktop_data['good'] / $desktop_total) * 100 : 0;
                        ?>
                        <?php if ($desktop_total > 0): ?>
                            <?php
                            // Generate tooltips for bar segments
                            $desktop_poor_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Poor: %d piece of content scores between 0-49.', 'Poor: %d pieces of content score between 0-49.', $desktop_data['poor'], 'mustang-wpo'),
                                $desktop_data['poor']
                            );
                            $desktop_needs_improvement_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Needs improvement: %d piece of content scores between 50-89.', 'Needs improvement: %d pieces of content score between 50-89.', $desktop_data['needs_improvement'], 'mustang-wpo'),
                                $desktop_data['needs_improvement']
                            );
                            $desktop_good_tooltip = sprintf(
                                // translators: %d is the count of posts
                                _n('Good: %d piece of content scores between 90-100.', 'Good: %d pieces of content score between 90-100.', $desktop_data['good'], 'mustang-wpo'),
                                $desktop_data['good']
                            );
                            ?>
                            <div class="mustwp-dashboard-chart">
                                <div class="mustwp-stacked-bar">
                                    <div class="mustwp-bar-segment mustwp-bar-poor" 
                                         style="width: <?php echo esc_attr($desktop_poor_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($desktop_poor_pct); ?>"
                                         title="<?php echo esc_attr($desktop_poor_tooltip); ?>"></div>
                                    <div class="mustwp-bar-segment mustwp-bar-needs-improvement" 
                                         style="width: <?php echo esc_attr($desktop_needs_improvement_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($desktop_needs_improvement_pct); ?>"
                                         title="<?php echo esc_attr($desktop_needs_improvement_tooltip); ?>"></div>
                                    <div class="mustwp-bar-segment mustwp-bar-good" 
                                         style="width: <?php echo esc_attr($desktop_good_pct); ?>%" 
                                         data-pct="<?php echo esc_attr($desktop_good_pct); ?>"
                                         title="<?php echo esc_attr($desktop_good_tooltip); ?>"></div>
                                </div>
                            </div>
                            
                            <div class="mustwp-dashboard-legend">
                                <div class="mustwp-legend-item" data-category="poor">
                                    <span class="mustwp-legend-dot mustwp-dot-poor"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Poor', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($desktop_data['poor']); ?></span>
                                </div>
                                <div class="mustwp-legend-item" data-category="needs_improvement">
                                    <span class="mustwp-legend-dot mustwp-dot-needs-improvement"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Needs improvement', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($desktop_data['needs_improvement']); ?></span>
                                </div>
                                <div class="mustwp-legend-item" data-category="good">
                                    <span class="mustwp-legend-dot mustwp-dot-good"></span>
                                    <span class="mustwp-legend-label">
                                        <strong><?php esc_html_e('Good', 'mustang-wpo'); ?></strong>
                                    </span>
                                    <span class="mustwp-legend-count"><?php echo esc_html($desktop_data['good']); ?></span>
                                </div>
                            </div>
                            
                            <?php
                            // Generate summary verdict for Desktop
                            $desktop_verdict = $this->generate_performance_verdict($desktop_data, 'desktop');
                            ?>
                            <p class="mustwp-device-verdict"><?php echo esc_html($desktop_verdict); ?></p>
                        <?php else: ?>
                            <p class="mustwp-dashboard-empty">
                                <?php esc_html_e('No desktop audit data available.', 'mustang-wpo'); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Disclaimer -->
                <div class="mustwp-dashboard-disclaimer">
                    <?php
                    printf(
                        // translators: %s is a link to the performance scoring documentation
                        __('Scores are estimates based on real-world performance metrics. Learn more about them %s.', 'mustang-wpo'),
                        '<a href="https://developer.chrome.com/docs/lighthouse/performance/performance-scoring/" target="_blank" rel="noopener noreferrer">' . esc_html__('here', 'mustang-wpo') . '</a>'
                    );
                    ?>
                </div>
                
                <!-- Hidden data for JavaScript (if needed for future enhancements) -->
                <script type="application/json" id="mustwp-dashboard-data" style="display: none;">
                    <?php echo wp_json_encode($data); ?>
                </script>
            <?php else: ?>
                <p class="mustwp-dashboard-empty">
                    <?php esc_html_e('No published content has been audited yet. Run a global audit to get started.', 'mustang-wpo'); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }
}

