<?php
/**
 * Budget checking logic for WPO
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * Budgets class for performance budget management
 */
class Budgets {
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
    }
    
    /**
     * Check if audit results meet budget requirements
     *
     * @param array $audit_results Audit results for a device
     * @param array $budgets Budget configuration for post type
     * @return array Budget check results
     */
    public function check_budgets(array $audit_results, array $budgets): array {
        $results = [
            'all_pass' => true,
            'details' => []
        ];
        
        // If no budgets configured, consider it passing
        if (empty($budgets)) {
            return $results;
        }
        
        // Check each configured budget metric
        foreach ($budgets as $metric => $threshold) {
            if (!isset($audit_results[$metric])) {
                continue; // Skip if metric not available in audit results
            }
            
            $pass = $this->check_metric($metric, $audit_results[$metric], $threshold);
            $results['details'][$metric] = [
                'pass' => $pass,
                'value' => $audit_results[$metric],
                'threshold' => $threshold
            ];
            
            if (!$pass) {
                $results['all_pass'] = false;
            }
        }
        
        return $results;
    }
    
    /**
     * Check individual metric against threshold
     *
     * @param string $metric Metric name
     * @param mixed $value Actual value
     * @param mixed $threshold Threshold value
     * @return bool True if passes budget
     */
    private function check_metric(string $metric, $value, $threshold): bool {
        switch ($metric) {
            case 'lcp':
                // LCP: lower is better (seconds)
                return (float) $value <= (float) $threshold;
                
            case 'cls':
                // CLS: lower is better (unitless)
                return (float) $value <= (float) $threshold;
                
            case 'inp':
                // INP: lower is better (milliseconds)
                return (int) $value <= (int) $threshold;
                
            case 'weight':
                // Weight: lower is better (MB)
                return (float) $value <= (float) $threshold;
                
            default:
                return true; // Unknown metric, consider it passing
        }
    }
    
    /**
     * Get budget configuration for a post type
     *
     * @param string $post_type Post type
     * @return array|null Budget configuration or null if not set
     */
    public function get_budgets_for_post_type(string $post_type): ?array {
        return $this->settings['budgets'][$post_type] ?? null;
    }
    
    /**
     * Update budget configuration for a post type
     *
     * @param string $post_type Post type
     * @param array $budgets Budget configuration
     * @return bool True on success
     */
    public function update_budgets_for_post_type(string $post_type, array $budgets): bool {
        $this->settings['budgets'][$post_type] = $this->sanitize_budgets($budgets);
        return update_option('mustwp_settings', $this->settings);
    }
    
    /**
     * Sanitize budget values
     *
     * @param array $budgets Raw budget values
     * @return array Sanitized budget values
     */
    private function sanitize_budgets(array $budgets): array {
        $sanitized = [];
        
        foreach ($budgets as $metric => $value) {
            switch ($metric) {
                case 'lcp':
                case 'cls':
                case 'weight':
                    $sanitized[$metric] = (float) $value;
                    break;
                    
                case 'inp':
                    $sanitized[$metric] = (int) $value;
                    break;
                    
                default:
                    // Skip unknown metrics
                    continue 2;
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Get all configured post types with budgets
     *
     * @return array Post types with budget configurations
     */
    public function get_post_types_with_budgets(): array {
        return array_keys($this->settings['budgets'] ?? []);
    }
    
    /**
     * Get budget summary for multiple posts
     *
     * @param array $post_ids Array of post IDs
     * @return array Budget summary statistics
     */
    public function get_budget_summary(array $post_ids): array {
        $summary = [
            'total_posts' => count($post_ids),
            'audited_posts' => 0,
            'passed_budgets' => 0,
            'failed_budgets' => 0,
            'no_budgets' => 0,
            'post_type_stats' => []
        ];
        
        $api = new API();
        
        foreach ($post_ids as $post_id) {
            $post_type = get_post_type($post_id);
            $budgets = $this->get_budgets_for_post_type($post_type);
            
            // If no budgets configured for this post type, count as no budgets
            if (empty($budgets)) {
                $summary['no_budgets']++;
                
                // Track per post type
                if (!isset($summary['post_type_stats'][$post_type])) {
                    $summary['post_type_stats'][$post_type] = [
                        'total' => 0,
                        'passed' => 0,
                        'failed' => 0,
                        'no_budgets' => 0
                    ];
                }
                $summary['post_type_stats'][$post_type]['total']++;
                $summary['post_type_stats'][$post_type]['no_budgets']++;
                continue;
            }
            
            $results = $api->get_results($post_id);
            if (!$results) {
                continue; // Skip posts without audit results
            }
            
            $summary['audited_posts']++;
            
            // Check budget compliance for both devices
            $has_failure = false;
            foreach (['mobile', 'desktop'] as $device) {
                if (!isset($results[$device])) {
                    continue;
                }
                
                $budget_check = $this->check_budgets($results[$device], $budgets);
                if (!$budget_check['all_pass']) {
                    $has_failure = true;
                    break;
                }
            }
            
            if ($has_failure) {
                $summary['failed_budgets']++;
            } else {
                $summary['passed_budgets']++;
            }
            
            // Track per post type
            if (!isset($summary['post_type_stats'][$post_type])) {
                $summary['post_type_stats'][$post_type] = [
                    'total' => 0,
                    'passed' => 0,
                    'failed' => 0,
                    'no_budgets' => 0
                ];
            }
            
            $summary['post_type_stats'][$post_type]['total']++;
            if ($has_failure) {
                $summary['post_type_stats'][$post_type]['failed']++;
            } else {
                $summary['post_type_stats'][$post_type]['passed']++;
            }
        }
        
        return $summary;
    }
    
    /**
     * Get default budget values
     *
     * @return array Default budget configuration
     */
    public function get_default_budgets(): array {
        return [
            'lcp' => 3.0,
            'cls' => 0.1,
            'inp' => 200,
            'weight' => 2.0
        ];
    }
    
    /**
     * Validate budget values
     *
     * @param array $budgets Budget values to validate
     * @return array Validation results
     */
    public function validate_budgets(array $budgets): array {
        $errors = [];
        $defaults = $this->get_default_budgets();
        
        foreach ($budgets as $metric => $value) {
            if (!isset($defaults[$metric])) {
                // translators: %s is the metric name
                $errors[] = sprintf(__('Unknown budget metric: %s', 'mustang-wpo'), $metric);
                continue;
            }
            
            if (!is_numeric($value)) {
                // translators: %s is the metric name
                $errors[] = sprintf(__('Budget value for %s must be numeric', 'mustang-wpo'), $metric);
                continue;
            }
            
            $value = (float) $value;
            if ($value < 0) {
                // translators: %s is the metric name
                $errors[] = sprintf(__('Budget value for %s must be positive', 'mustang-wpo'), $metric);
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }
}

