<?php
/**
 * Global audit runner for WPO
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * Auditor class for global audit functionality
 */
class Auditor {
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes(): void {
        
        register_rest_route('mustwp/v1', '/audit', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_single_audit'],
            'permission_callback' => [$this, 'check_edit_post_permission'],
            'args' => [
                'post_id' => [
                    'required' => true,
                    'type' => 'integer',
                    'sanitize_callback' => 'absint'
                ],
                'strategy' => [
                    'required' => true,
                    'type' => 'string',
                    'enum' => ['mobile', 'desktop', 'both'],
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);
        
        register_rest_route('mustwp/v1', '/audit/batch', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_batch_audit'],
            'permission_callback' => [$this, 'check_nonce_permission'],
            'args' => [
                'post_ids' => [
                    'required' => true,
                    'type' => 'array',
                    'items' => ['type' => 'integer'],
                    'sanitize_callback' => function($post_ids) {
                        return array_map('absint', $post_ids);
                    }
                ],
                'strategy' => [
                    'required' => true,
                    'type' => 'string',
                    'enum' => ['mobile', 'desktop', 'both'],
                    'sanitize_callback' => 'sanitize_text_field'
                ]
            ]
        ]);
        
        // Add a simple test endpoint (requires authentication)
        register_rest_route('mustwp/v1', '/test', [
            'methods' => 'GET',
            'callback' => [$this, 'test_endpoint'],
            'permission_callback' => [$this, 'check_basic_permission']
        ]);
        
        // Add a completely open test endpoint
        register_rest_route('mustwp/v1', '/open-test', [
            'methods' => 'GET',
            'callback' => [$this, 'open_test_endpoint'],
            'permission_callback' => '__return_true'
        ]);
        
        // Log all registered routes for debugging
        global $wp_rest_server;
        if ($wp_rest_server) {
            $routes = $wp_rest_server->get_routes();
            $mustwp_routes = array_filter($routes, function($key) {
                return strpos($key, 'mustwp') !== false;
            }, ARRAY_FILTER_USE_KEY);
        }
        
        register_rest_route('mustwp/v1', '/audit/summary', [
            'methods' => 'GET',
            'callback' => [$this, 'get_audit_summary'],
            'permission_callback' => [$this, 'check_manage_options_permission']
        ]);
        
        register_rest_route('mustwp/v1', '/posts/titles', [
            'methods' => 'POST',
            'callback' => [$this, 'get_post_titles'],
            'permission_callback' => [$this, 'check_manage_options_permission']
        ]);
        
        register_rest_route('mustwp/v1', '/audit/progress', [
            'methods' => 'GET',
            'callback' => [$this, 'get_audit_progress'],
            'permission_callback' => [$this, 'check_manage_options_permission']
        ]);
    }
    
    /**
     * Handle single post audit
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response|WP_Error Response or error
     */
    public function handle_single_audit(\WP_REST_Request $request) {
        $post_id = $request->get_param('post_id');
        $strategy = $request->get_param('strategy');
        
        $api = new API();
        $url = $api->get_post_url($post_id);
        
        if (is_wp_error($url)) {
            return new \WP_Error('invalid_url', $url->get_error_message(), ['status' => 400]);
        }
        
        // Run audit
        if ($strategy === 'both') {
            $results = $api->audit_url_both($url);
        } else {
            $results = $api->audit_url($url, $strategy);
        }
        
        if (is_wp_error($results)) {
            return $results;
        }
        
        // Store results
        $api->store_results($post_id, $results);
        
        return new \WP_REST_Response([
            'success' => true,
            'post_id' => $post_id,
            'results' => $results
        ]);
    }
    
    /**
     * Open test endpoint (no authentication required)
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function open_test_endpoint(\WP_REST_Request $request): \WP_REST_Response {
        
        return new \WP_REST_Response([
            'success' => true,
            'message' => 'WPO open test endpoint working',
            'timestamp' => time(),
            'wp_version' => get_bloginfo('version'),
            'plugin_version' => MUSTWP_VERSION
        ]);
    }
    
    /**
     * Test endpoint
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function test_endpoint(\WP_REST_Request $request): \WP_REST_Response {
        
        $user_id = get_current_user_id();
        $is_logged_in = is_user_logged_in();
        $user_caps = [];
        
        if ($user_id) {
            $user = get_user_by('id', $user_id);
            if ($user) {
                $user_caps = $user->allcaps;
            }
        }
        
        return new \WP_REST_Response([
            'success' => true,
            'message' => 'WPO test endpoint working',
            'timestamp' => time(),
            'user_id' => $user_id,
            'is_logged_in' => $is_logged_in,
            'user_caps' => $user_caps,
            'can_edit_posts' => current_user_can('edit_posts'),
            'can_manage_options' => current_user_can('manage_options')
        ]);
    }
    
    /**
     * Handle batch audit
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response|\WP_Error Response or error
     */
    public function handle_batch_audit(\WP_REST_Request $request) {
        
        try {
            $post_ids = $request->get_param('post_ids');
            $strategy = $request->get_param('strategy');
            
            // Validate parameters
            if (empty($post_ids) || !is_array($post_ids)) {
                return new \WP_Error('invalid_params', 'Invalid post_ids parameter', ['status' => 400]);
            }
            
            if (empty($strategy)) {
                return new \WP_Error('invalid_params', 'Invalid strategy parameter', ['status' => 400]);
            }
            
            // Hardcode batch size to 1 post (2 API calls) to prevent timeout issues (especially with Cloudflare's ~100s limit)
            $batch_size = 1;
            
            // Limit to batch size to prevent oversized requests
            $post_ids = array_slice($post_ids, 0, $batch_size);
            
            $results = [];
            $errors = [];
            $processed = 0;
            
            // Track execution time to prevent timeouts (stay under Cloudflare's ~100s limit)
            $start_time = time();
            $max_execution_time = 90; // 90 seconds (leave buffer for Cloudflare's ~100s timeout limit)
            
            $api = new API();
            
            foreach ($post_ids as $index => $post_id) {
                // Check execution time - stop before timeout
                $elapsed_time = time() - $start_time;
                if ($elapsed_time >= $max_execution_time) {
                    // Return partial results to allow continuation
                    return new \WP_REST_Response([
                        'success' => true,
                        'processed' => $processed,
                        'total' => count($post_ids),
                        'results' => $results,
                        'errors' => $errors,
                        'has_more' => $processed < count($post_ids),
                        'timeout_warning' => true,
                        'message' => 'Batch processing stopped due to time limit. Partial results returned.'
                    ]);
                }
                
                $url = $api->get_post_url($post_id);
                if (is_wp_error($url)) {
                    $post_title = get_the_title($post_id);
                    $errors[] = [
                        'post_id' => $post_id,
                        'post_title' => $post_title,
                        'error' => $url->get_error_message()
                    ];
                    continue;
                }
                
                // Run audit with retry logic
                $audit_result = $this->run_audit_with_retry($api, $url, $strategy);
                
                if (is_wp_error($audit_result)) {
                    $post_title = get_the_title($post_id);
                    $errors[] = [
                        'post_id' => $post_id,
                        'post_title' => $post_title,
                        'error' => $audit_result->get_error_message()
                    ];
                    
                    // If it's a quota/403 error, stop processing
                    if (in_array($audit_result->get_error_code(), ['quota_exceeded', 'forbidden'], true)) {
                        break;
                    }
                    continue;
                }
                
                // Store results
                $api->store_results($post_id, $audit_result);
                
                $post_title = get_the_title($post_id);
                $results[] = [
                    'post_id' => $post_id,
                    'post_title' => $post_title,
                    'results' => $audit_result
                ];
                
                $processed++;
            }
            
            return new \WP_REST_Response([
                'success' => true,
                'processed' => $processed,
                'total' => count($post_ids),
                'results' => $results,
                'errors' => $errors,
                'has_more' => $processed < count($post_ids)
            ]);
            
        } catch (\Exception $e) {
            return new \WP_Error('batch_error', 'Batch processing failed: ' . $e->getMessage(), ['status' => 500]);
        } catch (\Error $e) {
            return new \WP_Error('batch_fatal_error', 'Batch processing fatal error: ' . $e->getMessage(), ['status' => 500]);
        }
    }
    
    /**
     * Run audit with retry logic
     *
     * @param API $api API instance
     * @param string $url URL to audit
     * @param string $strategy Strategy
     * @return array|WP_Error Audit results or error
     */
    private function run_audit_with_retry(API $api, string $url, string $strategy) {
        $max_retries = 3;
        $retry_delay = 2; // Start with 2 seconds
        
        for ($attempt = 0; $attempt <= $max_retries; $attempt++) {
            
            if ($strategy === 'both') {
                $result = $api->audit_url_both($url);
            } else {
                $result = $api->audit_url($url, $strategy);
            }
            
            if (!is_wp_error($result)) {
                return $result;
            }
            
            $error_code = $result->get_error_code();
            
            // Don't retry certain errors
            if (in_array($error_code, ['quota_exceeded', 'forbidden', 'bad_request', 'lighthouse_error'], true)) {
                return $result;
            }
            
            // Don't retry on last attempt
            if ($attempt === $max_retries) {
                return $result;
            }
            
            // Exponential backoff with jitter
            $jitter = wp_rand(1, 1000) / 1000; // 0-1 second jitter
            $sleep_time = $retry_delay + $jitter;
            sleep((int) $sleep_time); // Cast to int to fix fatal error
            $retry_delay *= 2; // Exponential backoff
        }
        
        return $result;
    }
    
    /**
     * Get audit summary
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function get_audit_summary(\WP_REST_Request $request) {
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        $post_ids = $this->get_post_ids_for_content_types($content_types);
        
        $budgets = new Budgets();
        $budget_summary = $budgets->get_budget_summary($post_ids);
        
        $api = new API();
        $score_summary = $this->get_score_summary($post_ids, $api);
        
        return new \WP_REST_Response([
            'budget_summary' => $budget_summary,
            'score_summary' => $score_summary,
            'total_posts' => count($post_ids),
            'post_ids' => $post_ids
        ]);
    }
    
    /**
     * Get post IDs for selected content types
     *
     * @param array $content_types Content types
     * @return array Post IDs
     */
    public function get_post_ids_for_content_types(array $content_types): array {
        $post_ids = [];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        foreach ($content_types as $post_type) {
            $posts = get_posts([
                'post_type' => $post_type,
                'post_status' => 'publish',
                'numberposts' => -1,
                'fields' => 'ids'
            ]);
            
            $post_ids = array_merge($post_ids, $posts);
        }
        
        return $post_ids;
    }
    
    /**
     * Get score summary statistics
     *
     * @param array $post_ids Post IDs
     * @param API $api API instance
     * @return array Score summary
     */
    private function get_score_summary(array $post_ids, API $api): array {
        $mobile_scores = [];
        $desktop_scores = [];
        
        foreach ($post_ids as $post_id) {
            $results = $api->get_results($post_id);
            if (!$results) {
                continue;
            }
            
            if (isset($results['mobile']['score'])) {
                $mobile_scores[] = $results['mobile']['score'];
            }
            
            if (isset($results['desktop']['score'])) {
                $desktop_scores[] = $results['desktop']['score'];
            }
        }
        
        return [
            'mobile' => [
                'count' => count($mobile_scores),
                'average' => count($mobile_scores) > 0 ? round(array_sum($mobile_scores) / count($mobile_scores)) : 0,
                'min' => count($mobile_scores) > 0 ? min($mobile_scores) : 0,
                'max' => count($mobile_scores) > 0 ? max($mobile_scores) : 0
            ],
            'desktop' => [
                'count' => count($desktop_scores),
                'average' => count($desktop_scores) > 0 ? round(array_sum($desktop_scores) / count($desktop_scores)) : 0,
                'min' => count($desktop_scores) > 0 ? min($desktop_scores) : 0,
                'max' => count($desktop_scores) > 0 ? max($desktop_scores) : 0
            ]
        ];
    }
    
    /**
     * Check nonce permission (for AJAX requests)
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if nonce is valid
     */
    public function check_nonce_permission(\WP_REST_Request $request): bool {
        $nonce = $request->get_header('X-WP-Nonce');
        
        if (!$nonce) {
            return false;
        }
        
        $is_valid = wp_verify_nonce($nonce, 'wp_rest');
        
        if ($is_valid) {
            // Also check if user can edit posts
            $can_edit = current_user_can('edit_posts');
            return $can_edit;
        }
        
        return false;
    }
    
    /**
     * Check basic permission (any logged-in user)
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if user is logged in
     */
    public function check_basic_permission(\WP_REST_Request $request): bool {
        $is_logged_in = is_user_logged_in();
        return $is_logged_in;
    }
    
    /**
     * Check edit posts permission (for batch operations)
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if user can edit posts
     */
    public function check_edit_posts_permission(\WP_REST_Request $request): bool {
        $can_edit = current_user_can('edit_posts');
        return $can_edit;
    }
    
    /**
     * Check edit post permission
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if user can edit post
     */
    public function check_edit_post_permission(\WP_REST_Request $request): bool {
        $post_id = $request->get_param('post_id');
        $can_edit = current_user_can('edit_post', $post_id);
        return $can_edit;
    }
    
    /**
     * Get post titles endpoint
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function get_post_titles(\WP_REST_Request $request) {
        $post_ids = $request->get_param('post_ids');
        $titles = [];
        
        if (is_array($post_ids)) {
            foreach ($post_ids as $post_id) {
                $titles[$post_id] = get_the_title($post_id);
            }
        }
        
        return new \WP_REST_Response([
            'titles' => $titles
        ]);
    }
    
    /**
     * Debug post IDs endpoint
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function debug_post_ids(\WP_REST_Request $request) {
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        $post_ids = $this->get_post_ids_for_content_types($content_types);
        
        return new \WP_REST_Response([
            'content_types' => $content_types,
            'post_ids' => $post_ids,
            'total_count' => count($post_ids),
            'settings' => $this->settings
        ]);
    }
    
    /**
     * Debug settings endpoint
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function debug_settings(\WP_REST_Request $request) {
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        $post_ids = $this->get_post_ids_for_content_types($content_types);
        
        return new \WP_REST_Response([
            'settings' => $this->settings,
            'content_types' => $content_types,
            'post_ids' => $post_ids,
            'total_posts' => count($post_ids),
            'available_post_types' => get_post_types(['public' => true], 'names')
        ]);
    }
    
    /**
     * Update audit progress
     *
     * @param string $audit_id Audit ID
     * @param array $progress Progress data
     * @return void
     */
    private function update_audit_progress(string $audit_id, array $progress): void {
        $existing = get_transient('mustwp_audit_progress_' . $audit_id) ?: [];
        $updated = array_merge($existing, $progress);
        set_transient('mustwp_audit_progress_' . $audit_id, $updated, HOUR_IN_SECONDS);
    }
    
    /**
     * Get audit progress
     *
     * @param \WP_REST_Request $request REST request
     * @return \WP_REST_Response Response
     */
    public function get_audit_progress(\WP_REST_Request $request) {
        $audit_id = $request->get_param('audit_id');
        
        if (!$audit_id) {
            return new \WP_Error('missing_audit_id', 'Audit ID is required', ['status' => 400]);
        }
        
        $progress = get_transient('mustwp_audit_progress_' . $audit_id);
        
        if (!$progress) {
            return new \WP_REST_Response([
                'status' => 'not_found',
                'message' => 'No progress data found for this audit'
            ]);
        }
        
        return new \WP_REST_Response($progress);
    }
    
    /**
     * Check manage options permission
     *
     * @param \WP_REST_Request $request REST request
     * @return bool True if user can manage options
     */
    public function check_manage_options_permission(\WP_REST_Request $request): bool {
        $can_manage = current_user_can('manage_options');
        return $can_manage;
    }
}

