<?php
/**
 * Google PageSpeed Insights API wrapper
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * PageSpeed API wrapper class
 */
class API {
    
    /**
     * PSI API base URL
     */
    private const PSI_BASE_URL = 'https://www.googleapis.com/pagespeedonline/v5/runPagespeed';
    
    /**
     * Rate limit transient key
     */
    private const RATE_LIMIT_KEY = 'mustwp_rate_limit';
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
    }
    
    /**
     * Check if a valid API key is available
     *
     * @return bool True if valid API key is available
     */
    public function has_valid_api_key(): bool {
        if (empty($this->settings['api_key'])) {
            return false;
        }
        
        $settings = new Settings();
        $decrypted_key = $settings->decrypt_api_key($this->settings['api_key']);
        
        return !empty($decrypted_key) && strpos($decrypted_key, 'AIza') === 0;
    }
    
    /**
     * Get API key requirement error
     *
     * @return WP_Error Error message for missing API key
     */
    private function get_api_key_required_error(): \WP_Error {
        return new \WP_Error(
            'api_key_required',
            sprintf(
                // translators: %s is the URL to the Mustang WPO Integrations page
                __('Google PageSpeed Insights API key is required. Please add a valid API key in <a href="%s">Mustang WPO Integrations</a>.', 'mustang-wpo'),
                admin_url('admin.php?page=mustwp-integrations')
            )
        );
    }
    
    /**
     * Run PageSpeed audit for a URL
     *
     * @param string $url The URL to audit
     * @param string $strategy Strategy: 'mobile' or 'desktop'
     * @return array|WP_Error Audit results or error
     */
    public function audit_url(string $url, string $strategy = 'mobile') {
        // Check if valid API key is available
        if (!$this->has_valid_api_key()) {
            return $this->get_api_key_required_error();
        }
        
        // Validate strategy
        if (!in_array($strategy, ['mobile', 'desktop'], true)) {
            return new \WP_Error('invalid_strategy', __('Invalid strategy. Must be mobile or desktop.', 'mustang-wpo'));
        }
        
        // Check rate limit
        if (!$this->check_rate_limit()) {
            return new \WP_Error('rate_limit', __('Rate limit exceeded. Please wait before making another request.', 'mustang-wpo'));
        }
        
        // Build API URL
        $api_url = $this->build_api_url($url, $strategy);
        
        // Make request
        $response = wp_remote_get($api_url, [
            'timeout' => 60, // Increased from 30 to 60 seconds
            'headers' => [
                'User-Agent' => 'WP-WPO/' . MUSTWP_VERSION . ' (WordPress/' . get_bloginfo('version') . ')',
                'Referer' => home_url()
            ]
        ]);
        
        // Handle request errors
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Handle HTTP errors
        if ($status_code !== 200) {
            return $this->handle_api_error($status_code, $body);
        }
        
        // Parse response
        $data = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new \WP_Error('json_parse', __('Failed to parse API response.', 'mustang-wpo'));
        }
        
        // Extract metrics
        return $this->extract_metrics($data, $strategy);
    }
    
    /**
     * Run audit for both mobile and desktop
     *
     * @param string $url The URL to audit
     * @return array|WP_Error Audit results or error
     */
    public function audit_url_both(string $url) {
        $mobile_result = $this->audit_url($url, 'mobile');
        if (is_wp_error($mobile_result)) {
            return $mobile_result;
        }
        
        $desktop_result = $this->audit_url($url, 'desktop');
        if (is_wp_error($desktop_result)) {
            return $desktop_result;
        }
        
        return [
            'mobile' => $mobile_result,
            'desktop' => $desktop_result
        ];
    }
    
    /**
     * Build API URL
     *
     * @param string $url Target URL
     * @param string $strategy Strategy
     * @return string API URL
     */
    private function build_api_url(string $url, string $strategy): string {
        // Do NOT pre-encode the URL; http_build_query will encode parameters appropriately
        $params = [
            'url' => $url,
            'strategy' => $strategy
        ];
        
        // Add API key (required for plugin to function)
        $settings = new Settings();
        $decrypted_key = $settings->decrypt_api_key($this->settings['api_key']);
        
        // API key is required and validated in audit_url method
        $params['key'] = sanitize_text_field($decrypted_key);
        
        return self::PSI_BASE_URL . '?' . http_build_query($params);
    }
    
    /**
     * Check rate limit: enforce <= 240 requests per minute across all strategies
     */
    private function check_rate_limit(): bool {
        $windowSeconds = 60;
        $maxRequests = 240;
        $key = self::RATE_LIMIT_KEY;
        $bucket = get_transient($key);
        $now = time();
        if (!is_array($bucket)) {
            $bucket = ['count' => 0, 'reset' => $now + $windowSeconds];
        }
        if ($now > ($bucket['reset'] ?? 0)) {
            $bucket = ['count' => 0, 'reset' => $now + $windowSeconds];
        }
        if (($bucket['count'] ?? 0) >= $maxRequests) {
            return false;
        }
        $bucket['count'] = ($bucket['count'] ?? 0) + 1;
        set_transient($key, $bucket, $windowSeconds);
        return true;
    }
    
    /**
     * Handle API errors
     *
     * @param int $status_code HTTP status code
     * @param string $body Response body
     * @return WP_Error Error object
     */
    private function handle_api_error(int $status_code, string $body): \WP_Error {
        $error_data = json_decode($body, true);
        $error_message = $error_data['error']['message'] ?? __('Unknown API error', 'mustang-wpo');
        
        // Log the full error for debugging
        // Log error without sensitive data
        $safe_body = $this->sanitize_log_data($body);
        
        switch ($status_code) {
            case 400:
                // Provide more specific error message for 400 errors
                if (strpos($error_message, 'invalid argument') !== false) {
                    return new \WP_Error('bad_request', __('Invalid URL or API request. Please check the URL format and try again.', 'mustang-wpo'));
                }
                // translators: %s is the error message
                return new \WP_Error('bad_request', sprintf(__('Bad request: %s', 'mustang-wpo'), $error_message));
            case 403:
                if (strpos($error_message, 'quota') !== false) {
                    return new \WP_Error('quota_exceeded', __('API quota exceeded. Please add an API key or reduce audit frequency.', 'mustang-wpo'));
                }
                // translators: %s is the error message
                return new \WP_Error('forbidden', sprintf(__('API access forbidden: %s', 'mustang-wpo'), $error_message));
            case 429:
                return new \WP_Error('rate_limit', __('API rate limit exceeded. Please wait before making another request.', 'mustang-wpo'));
            case 500:
                // Handle Lighthouse server errors gracefully
                if (strpos($error_message, 'Lighthouse returned error') !== false) {
                    return new \WP_Error('lighthouse_error', __('PageSpeed API temporarily unavailable. Please try again in a few minutes.', 'mustang-wpo'));
                }
                return new \WP_Error('server_error', sprintf(__('PageSpeed API server error. Please try again later.', 'mustang-wpo')));
            case 502:
            case 503:
            case 504:
                return new \WP_Error('gateway_error', __('PageSpeed API temporarily unavailable. Please try again in a few minutes.', 'mustang-wpo'));
            default:
                // translators: %1$d is the HTTP status code, %2$s is the error message
                return new \WP_Error('api_error', sprintf(__('API error (%1$d): %2$s', 'mustang-wpo'), $status_code, $error_message));
        }
    }
    
    /**
     * Extract metrics from PSI response
     *
     * @param array $data PSI response data
     * @param string $strategy Strategy used
     * @return array Extracted metrics
     */
    private function extract_metrics(array $data, string $strategy): array {
        $lighthouse = $data['lighthouseResult'] ?? [];
        $audits = $lighthouse['audits'] ?? [];
        
        // Extract performance score
        $categories = $lighthouse['categories'] ?? [];
        $performance = $categories['performance'] ?? [];
        $score = round(($performance['score'] ?? 0) * 100);
        
        // Extract Core Web Vitals
        $lcp_audit = $audits['largest-contentful-paint'] ?? [];
        $lcp_value = $lcp_audit['numericValue'] ?? 0;
        $lcp = round($lcp_value / 1000, 2); // Convert ms to seconds
        
        $cls_audit = $audits['cumulative-layout-shift'] ?? [];
        $cls = round($cls_audit['numericValue'] ?? 0, 3);
        
        // INP (Interaction to Next Paint) is reported under 'interaction-to-next-paint'
        $inp_audit = $audits['interaction-to-next-paint'] ?? [];
        $inp = round($inp_audit['numericValue'] ?? 0);
        
        // Extract total byte weight
        $weight_audit = $audits['total-byte-weight'] ?? [];
        $weight_bytes = $weight_audit['numericValue'] ?? 0;
        $weight = round($weight_bytes / (1024 * 1024), 2); // Convert bytes to MB
        
        return [
            'score' => $score,
            'lcp' => $lcp,
            'cls' => $cls,
            'inp' => $inp,
            'weight' => $weight,
            'timestamp' => time()
        ];
    }
    
    /**
     * Store audit results in post meta
     *
     * @param int $post_id Post ID
     * @param array $results Audit results
     * @return bool True on success, false on failure
     */
    public function store_results(int $post_id, array $results): bool {
        // update_post_meta returns meta_id (int) on add/update, or bool false on failure.
        // Normalize to strict bool to satisfy return type.
        return update_post_meta($post_id, '_mustwp_pagespeed', $results) !== false;
    }
    
    /**
     * Get stored audit results
     *
     * @param int $post_id Post ID
     * @return array|null Audit results or null if not found
     */
    public function get_results(int $post_id): ?array {
        $results = get_post_meta($post_id, '_mustwp_pagespeed', true);
        return empty($results) ? null : $results;
    }
    
    /**
     * Get post URL for audit
     *
     * @param int $post_id Post ID
     * @return string|WP_Error Post URL or error
     */
    public function get_post_url(int $post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return new \WP_Error('post_not_found', __('Post not found.', 'mustang-wpo'));
        }
        
        $url = get_permalink($post_id);
        if (!$url) {
            return new \WP_Error('invalid_permalink', __('Could not generate permalink.', 'mustang-wpo'));
        }
        
        // Debug: Log the URL being audited
        
        // Validate URL format
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            return new \WP_Error('invalid_url_format', __('Generated URL has invalid format.', 'mustang-wpo'));
        }
        
        return $url;
    }
    
    /**
     * Sanitize log data to remove sensitive information
     *
     * @param string $data Raw log data
     * @return string Sanitized log data
     */
    private function sanitize_log_data(string $data): string {
        // Remove API keys and other sensitive data from logs
        $sanitized = $data;
        
        // Remove potential API key patterns
        $sanitized = preg_replace('/["\']?key["\']?\s*:\s*["\'][^"\']+["\']/', '"key":"[REDACTED]"', $sanitized);
        $sanitized = preg_replace('/["\']?api_key["\']?\s*:\s*["\'][^"\']+["\']/', '"api_key":"[REDACTED]"', $sanitized);
        
        // Remove URLs that might contain sensitive data
        $sanitized = preg_replace('/https?:\/\/[^\s"\']+/', '[URL_REDACTED]', $sanitized);
        
        return $sanitized;
    }
}

