<?php
/**
 * Admin functionality for WPO
 *
 * @package MUSTWP
 */
 
declare(strict_types=1);
 
namespace MUSTWP;

/**
 * Admin class for list table integration
 */
class Admin {
    
    /**
     * Plugin settings
     */
    private array $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->settings = get_option('mustwp_settings', []);
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks(): void {
        add_action('admin_init', [$this, 'add_pagespeed_columns']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }
    
    /**
     * Add PageSpeed columns to post list tables
     */
    public function add_pagespeed_columns(): void {
        $content_types = $this->settings['content_types'] ?? ['post', 'page'];
        $allowed_types = ['post', 'page']; // Free version limitation
        
        // Filter to only allow 'post' and 'page' in free version
        $content_types = array_intersect($content_types, $allowed_types);
        
        foreach ($content_types as $post_type) {
            add_filter("manage_{$post_type}_posts_columns", [$this, 'add_pagespeed_column']);
            add_action("manage_{$post_type}_posts_custom_column", [$this, 'display_pagespeed_column'], 10, 2);
        }
    }
    
    /**
     * Add PageSpeed column to post list table
     *
     * @param array $columns Existing columns
     * @return array Modified columns
     */
    public function add_pagespeed_column(array $columns): array {
        // Add PageSpeed column to the far right
        $columns['mustwp_pagespeed'] = __('Performance', 'mustang-wpo');
        
        return $columns;
    }
    
    /**
     * Display PageSpeed column content
     *
     * @param string $column_name Column name
     * @param int $post_id Post ID
     */
    public function display_pagespeed_column(string $column_name, int $post_id): void {
        if ($column_name !== 'mustwp_pagespeed') {
            return;
        }
        
        // Check if post is published
        $post_status = get_post_status($post_id);
        if ($post_status !== 'publish') {
            // Get status label for tooltip
            $status_object = get_post_status_object($post_status);
            $status_label = $status_object ? $status_object->label : ucfirst($post_status);
            echo wp_kses_post($this->render_na_badge_for_status($status_label));
            return;
        }
        
        $api = new API();
        $results = $api->get_results($post_id);
        
        if (!$results) {
            echo wp_kses_post($this->render_not_audited_badge());
            return;
        }
        
        echo wp_kses_post($this->render_pagespeed_badges($results));
    }
    
    /**
     * Render NA badge for posts that are not published
     *
     * @param string $status_label Post status label
     * @return string HTML for NA badge
     */
    private function render_na_badge_for_status(string $status_label): string {
        return sprintf(
            '<span class="mustwp-badge mustwp-badge-na" title="%s">NA</span>',
            esc_attr(sprintf(
                // translators: %s is the post status label (e.g., "Draft", "Pending Review")
                __('Only published content can be audited. Current status: %s', 'mustang-wpo'),
                $status_label
            ))
        );
    }
    
    /**
     * Render badge for published posts that haven't been audited yet
     *
     * @return string HTML for not audited badge
     */
    private function render_not_audited_badge(): string {
        return sprintf(
            '<span class="mustwp-badge mustwp-badge-na" title="%s">–</span>',
            esc_attr__('This post hasn\'t been audited yet.', 'mustang-wpo')
        );
    }
    
    /**
     * Render PageSpeed badges for mobile and desktop
     *
     * @param array $results Audit results
     * @return string HTML for badges
     */
    private function render_pagespeed_badges(array $results): string {
        $output = '';
        
        // Mobile badge
        if (isset($results['mobile'])) {
            $mobile = $results['mobile'];
            $output .= $this->render_score_badge($mobile['score'], 'mobile', $mobile['timestamp']);
        }
        
        // Desktop badge
        if (isset($results['desktop'])) {
            $desktop = $results['desktop'];
            $output .= $this->render_score_badge($desktop['score'], 'desktop', $desktop['timestamp']);
        }
        
        return $output;
    }
    
    /**
     * Render individual score badge
     *
     * @param int $score PageSpeed score
     * @param string $device Device type (mobile/desktop)
     * @param int $timestamp Audit timestamp
     * @return string HTML for badge
     */
    private function render_score_badge($score, string $device, $timestamp): string {
        $score = (int) round($score);
        $timestamp = (int) $timestamp;
        $class = $this->get_badge_class($score);
        $device_label = $device === 'mobile' ? __('Mobile', 'mustang-wpo') : __('Desktop', 'mustang-wpo');
        $performance_label = $this->get_performance_label($score);
        
        // Use WordPress timezone handling
        $timezone = wp_timezone();
        $formatted_time = wp_date(
            get_option('date_format') . ', ' . get_option('time_format'),
            $timestamp,
            $timezone
        );
        
        $tooltip = sprintf(
            '%s: %s | Last audited: %s',
            $device_label,
            $performance_label,
            $formatted_time
        );
        
        return sprintf(
            '<span class="mustwp-badge mustwp-badge-%s" title="%s">%d</span> ',
            esc_attr($class),
            esc_attr($tooltip),
            $score
        );
    }
    
    /**
     * Get performance label based on score
     *
     * @param int $score PageSpeed score
     * @return string Performance label
     */
    private function get_performance_label(int $score): string {
        if ($score >= 90) {
            return __('Good', 'mustang-wpo');
        } elseif ($score >= 50) {
            return __('Needs improvement', 'mustang-wpo');
        } else {
            return __('Poor', 'mustang-wpo');
        }
    }
    
    /**
     * Get badge CSS class based on score
     *
     * @param int $score PageSpeed score
     * @return string CSS class
     */
    private function get_badge_class(int $score): string {
        if ($score >= 90) {
            return 'good';
        } elseif ($score >= 50) {
            return 'warning';
        } else {
            return 'poor';
        }
    }
    
    /**
     * Get budget warning indicator
     *
     * @param array $results Audit results
     * @return string|null HTML for warning or null
     */
    private function get_budget_warning(array $results): ?string {
        $budgets = new Budgets();
        $post_type = get_post_type();
        
        // Check if budgets are configured for this post type
        $post_type_budgets = $this->settings['budgets'][$post_type] ?? null;
        if (empty($post_type_budgets)) {
            return null;
        }
        
        // Check budget compliance
        $has_failure = false;
        
        foreach (['mobile', 'desktop'] as $device) {
            if (!isset($results[$device])) {
                continue;
            }
            
            $device_results = $results[$device];
            $budget_check = $budgets->check_budgets($device_results, $post_type_budgets);
            
            if (!$budget_check['all_pass']) {
                $has_failure = true;
                break;
            }
        }
        
        if ($has_failure) {
            return sprintf(
                '<span class="mustwp-budget-warning" title="%s">⚠</span> ',
                esc_attr__('Budget fail', 'mustang-wpo')
            );
        }
        
        return null;
    }
    
    /**
     * Enqueue admin assets
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_assets(string $hook): void {
        // Only enqueue on post list pages
        if (!str_contains($hook, 'edit.php')) {
            return;
        }
        
        wp_enqueue_style(
            'mustwp-admin',
            MUSTWP_PLUGIN_URL . 'build/css/admin.css',
            [],
            MUSTWP_VERSION
        );
        
        // Add inline CSS for PageSpeed column width and cursor
        wp_add_inline_style('mustwp-admin', '
            .wp-list-table .column-mustwp_pagespeed {
                width: 6em;
            }
            .mustwp-badge {
                cursor: default;
            }
        ');
    }
}

