/**
 * Gutenberg editor integration for MUSTWP - Vanilla JavaScript Approach
 */

(function() {
    'use strict';

    // Wait for WordPress to be ready
    wp.domReady(function() {
        console.log('MUSTWP: Starting editor integration');
        
        // Check for required APIs
        if (typeof wp.plugins === 'undefined') {
            console.error('MUSTWP: wp.plugins not available');
            return;
        }
        
        // Check for new API first (WordPress 6.6+), fallback to old API for backward compatibility
        if (typeof wp.editor === 'undefined' && typeof wp.editPost === 'undefined') {
            console.error('MUSTWP: wp.editor or wp.editPost not available');
            return;
        }

        const { registerPlugin } = wp.plugins;
        // Use new API if available (WordPress 6.6+), fallback to deprecated API for older WordPress versions
        const { PluginDocumentSettingPanel } = wp.editor || wp.editPost;
        const { useState, useEffect, createElement } = wp.element;
        const { Button, Spinner, Notice } = wp.components;
        const { apiFetch } = wp;
        const { __ } = wp.i18n;

        console.log('MUSTWP: All APIs available, creating panel');

        /**
         * MUSTWP Panel Component
         */
        function MUSTWPEditorPanel() {
            const [results, setResults] = useState(null);
            const [loading, setLoading] = useState(false);
            const [error, setError] = useState(null);
            const [postId, setPostId] = useState(null);

            // Get current post ID
            useEffect(function() {
                if (typeof MUSTWP !== 'undefined' && MUSTWP.postId) {
                    setPostId(parseInt(MUSTWP.postId));
                    console.log('MUSTWP: Post ID from localized data:', MUSTWP.postId);
                } else {
                    // Fallback to DOM element
                    const postIdElement = document.querySelector('#post_ID');
                    if (postIdElement) {
                        setPostId(parseInt(postIdElement.value));
                        console.log('MUSTWP: Post ID from DOM:', postIdElement.value);
                    }
                }
            }, []);

            // Load existing results
            useEffect(function() {
                if (postId) {
                    loadResults();
                }
            }, [postId]);

            /**
             * Load existing audit results
             */
            const loadResults = async function() {
                try {
                    console.log('MUSTWP: Loading results for post:', postId);
                    const response = await apiFetch({
                        path: '/mustwp/v1/editor/results?post_id=' + postId,
                        method: 'GET'
                    });

                    if (response.success && response.has_results) {
                        setResults(response.results);
                        console.log('MUSTWP: Results loaded:', response.results);
                    }
                } catch (err) {
                    console.error('MUSTWP: Failed to load results:', err);
                }
            };

            /**
             * Run PageSpeed audit
             */
            const runAudit = async function() {
                if (!postId) return;

                console.log('MUSTWP: Starting audit for post:', postId);
                setLoading(true);
                setError(null);

                try {
                    const response = await apiFetch({
                        path: '/mustwp/v1/editor/audit',
                        method: 'POST',
                        data: {
                            post_id: postId,
                            strategy: 'both'
                        }
                    });

                    if (response.success) {
                        setResults(response.results);
                        console.log('MUSTWP: Audit completed:', response.results);
                    } else {
                        setError(response.message || 'Audit failed');
                        console.error('MUSTWP: Audit failed:', response.message);
                    }
                } catch (err) {
                    setError(err.message || 'Network error occurred');
                    console.error('MUSTWP: Audit error:', err);
                } finally {
                    setLoading(false);
                }
            };

            /**
             * Get performance label based on score
             */
            const getPerformanceLabel = function(score) {
                if (score >= 90) return 'Good';
                if (score >= 50) return 'Needs improvement';
                return 'Poor';
            };

            /**
             * Render score badge using createElement
             */
            const renderScoreBadge = function(score, device) {
                if (score === null || score === undefined) {
                    return createElement('div', { className: 'mustwp-editor-score-row' },
                        createElement('span', { className: 'mustwp-badge mustwp-badge-na' }, 'NA'),
                        createElement('span', null,
                            device === 'mobile' ? 'Mobile: ' : 'Desktop: ',
                            createElement('strong', null, 'Not audited')
                        )
                    );
                }

                const badgeClass = score.score >= 90 ? 'good' : score.score >= 50 ? 'warning' : 'poor';
                const performanceLabel = getPerformanceLabel(score.score);

                return createElement('div', { className: 'mustwp-editor-score-row' },
                    createElement('span', { className: 'mustwp-badge mustwp-badge-' + badgeClass }, score.score),
                    createElement('span', null,
                        device === 'mobile' ? 'Mobile: ' : 'Desktop: ',
                        createElement('strong', null, performanceLabel)
                    )
                );
            };

            /**
             * Render mobile and desktop scores
             */
            const renderScores = function() {
                // Check post status
                const isPublished = typeof MUSTWP !== 'undefined' && MUSTWP.postStatus === 'publish';
                
                // If not published, show NA badges (message will be shown below button)
                if (!isPublished) {
                    return createElement('div', null,
                        createElement('div', { className: 'mustwp-editor-score-row' },
                            createElement('span', { className: 'mustwp-badge mustwp-badge-na' }, 'NA'),
                            createElement('span', null, 'Mobile: ', createElement('strong', null, 'Not audited'))
                        ),
                        createElement('div', { className: 'mustwp-editor-score-row' },
                            createElement('span', { className: 'mustwp-badge mustwp-badge-na' }, 'NA'),
                            createElement('span', null, 'Desktop: ', createElement('strong', null, 'Not audited'))
                        )
                    );
                }
                
                // If published but no results, show – badges (message will be shown below button)
                if (!results) {
                    return createElement('div', null,
                        createElement('div', { className: 'mustwp-editor-score-row' },
                            createElement('span', { className: 'mustwp-badge mustwp-badge-na' }, '–'),
                            createElement('span', null, 'Mobile: ', createElement('strong', null, 'Not audited'))
                        ),
                        createElement('div', { className: 'mustwp-editor-score-row' },
                            createElement('span', { className: 'mustwp-badge mustwp-badge-na' }, '–'),
                            createElement('span', null, 'Desktop: ', createElement('strong', null, 'Not audited'))
                        )
                    );
                }

                // Has results, show scores
                return createElement('div', null,
                    renderScoreBadge(results.mobile, 'mobile'),
                    renderScoreBadge(results.desktop, 'desktop')
                );
            };

            /**
             * Render timestamp
             */
            const renderTimestamp = function() {
                // Don't show timestamp if post is not published
                const isPublished = typeof MUSTWP !== 'undefined' && MUSTWP.postStatus === 'publish';
                if (!isPublished) {
                    return null;
                }
                
                if (!results || !results.mobile || !results.desktop) {
                    return null;
                }

                // Use the pre-formatted timestamp from PHP
                const formattedTime = results.mobile.formatted_timestamp || results.desktop.formatted_timestamp;
                if (!formattedTime) return null;

                return createElement('div', { className: 'mustwp-editor-timestamp' },
                    'Last audited: ' + formattedTime
                );
            };

            const renderTooltip = function() {
                // If results exist, API key must be valid (audits can't run without it)
                // So skip API key check if results exist
                const hasResults = results && (results.mobile || results.desktop);
                
                // Check if API key is valid - show message for all post statuses if no API key
                // Only check API key if there are no results (results prove API key exists)
                if (!hasResults) {
                    const hasApiKey = typeof MUSTWP !== 'undefined' && MUSTWP.hasApiKey;
                    if (!hasApiKey) {
                        const apiKeyMessage = typeof MUSTWP !== 'undefined' && MUSTWP.strings && MUSTWP.strings.apiKeyRequired 
                            ? MUSTWP.strings.apiKeyRequired 
                            : 'A valid Google PageSpeed Insights API key is required for the plugin to function, please add one on the Integrations page.';
                        return createElement('div', { className: 'mustwp-editor-timestamp' },
                            apiKeyMessage
                        );
                    }
                }
                
                // Check post status
                const isPublished = typeof MUSTWP !== 'undefined' && MUSTWP.postStatus === 'publish';
                
                // For non-published posts, show message below button
                if (!isPublished) {
                    return createElement('div', { className: 'mustwp-editor-timestamp' },
                        'Only published content can be audited, Google\'s PageSpeed Insights requires a public URL.'
                    );
                }
                
                // For published but not audited, show message below button
                if (!results) {
                    return createElement('div', { className: 'mustwp-editor-timestamp' },
                        'This post hasn\'t been audited yet.'
                    );
                }
                
                // If button is disabled for other reasons, show tooltip
                if (buttonState.disabled && buttonState.tooltip) {
                    return createElement('div', { className: 'mustwp-editor-timestamp' },
                        buttonState.tooltip
                    );
                }
                
                return null;
            };

            /**
             * Get button text and state based on post status and audit results
             */
            const getButtonState = function() {
                // If results exist, API key must be valid (audits can't run without it)
                const hasResults = results && (results.mobile || results.desktop);
                
                // Check if API key is valid - disable button if no API key
                // Only check API key if there are no results (results prove API key exists)
                if (!hasResults) {
                    const hasApiKey = typeof MUSTWP !== 'undefined' && MUSTWP.hasApiKey;
                    if (!hasApiKey) {
                        return {
                            text: 'Run Audit',
                            disabled: true,
                            tooltip: null // Message is shown via renderTooltip()
                        };
                    }
                }
                
                // Use localized data from PHP
                const isPublished = typeof MUSTWP !== 'undefined' && MUSTWP.postStatus === 'publish';
                
                if (!isPublished) {
                    return {
                        text: 'Publish to Audit',
                        disabled: true,
                        tooltip: null // Message is shown via renderTooltip()
                    };
                }
                
                if (hasResults) {
                    return {
                        text: 'Re-run Audit',
                        disabled: false,
                        tooltip: null
                    };
                }
                
                return {
                    text: 'Run Audit',
                    disabled: false,
                    tooltip: null
                };
            };

            const buttonState = getButtonState();

            return createElement(PluginDocumentSettingPanel, {
                name: 'mustwp-panel',
                title: (typeof MUSTWP !== 'undefined' && MUSTWP.strings && MUSTWP.strings.panelTitle) ? MUSTWP.strings.panelTitle : 'Mustang WPO',
                className: 'mustwp-editor-panel'
            },
                createElement('span', null,
                    createElement('strong', null, 'Performance Scores')
                ),
                renderScores(),
                error && createElement('div', { className: 'mustwp-editor-error' }, 'Error: ' + error),
                createElement('button', {
                    onClick: runAudit,
                    disabled: loading || !postId || buttonState.disabled,
                    className: 'button is-primary'
                }, loading ? [
                    createElement(Spinner, { key: 'spinner' }),
                    'Auditing...'
                ] : buttonState.text),
                renderTooltip(),
                renderTimestamp()
            );
        }

        try {
            registerPlugin('mustwp-editor', {
                render: MUSTWPEditorPanel,
                icon: 'performance'
            });
            console.log('MUSTWP: Plugin registered successfully');
        } catch (err) {
            console.error('MUSTWP: Failed to register plugin:', err);
        }
    });

})();