<?php
/**
 * Plugin Name: MTM AI Share Button and Link Generator
 * Plugin URI:  https://wordpress.org/plugins/mtm-ai-link-generator/
 * Description: Display AI search links with a global prompt and per-AI overrides (ChatGPT, Claude, Perplexity, Google/Gemini, Grok). Settings page and a shortcode [mtm_ai_links].
 * Version:     1.0.1
 * Author:      MTM, YSC
 * Author URI:  https://www.movingtrafficmedia.com/
 * Text Domain: mtm-ai-link-generator
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;

/**
 * Main plugin class.
 *
 * - Uses Settings API (single option array).
 * - Provides shortcode [mtm_ai_links].
 * - Enqueues a small frontend stylesheet.
 */
final class MTM_AI_Links_Plugin {

	/** Option name where plugin settings are stored (single option array). */
	const OPTION_NAME = 'mtm_ai_links_options_v1';

	/** Text domain for translations. */
	const TEXT_DOMAIN = 'mtm-ai-link-generator';

	/** Plugin version. */
	const VERSION = '1.1.0';

	/** URL to plugin directory. */
	private $plugin_url;

	/** Filesystem path to plugin directory. */
	private $plugin_path;

	/** Default settings. */
	private $defaults = [
		'section_title' => '',   // Section H2 title
		'title_color' 	=> '#000000',
		'global_text'   => '',   // Prompt used for all AIs unless overridden
		// For each AI: enable flag and optional override text
		'chatgpt_enable'   => 0,
		'chatgpt_text'     => '',
		'claude_enable'    => 0,
		'claude_text'      => '',
		'perplexity_enable'=> 0,
		'perplexity_text'  => '',
		'google_enable'    => 0,
		'google_text'      => '',
		'grok_enable'      => 0,
		'grok_text'        => '',
	];

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->plugin_url  = plugin_dir_url( __FILE__ );
		$this->plugin_path = plugin_dir_path( __FILE__ );

		// Load translations
		// add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );

		// Admin: settings page and registration
		add_action( 'admin_menu', [ $this, 'register_settings_page' ] );
		add_action( 'admin_init', [ $this, 'register_settings' ] );

		// Frontend: enqueue styles
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );

		// Shortcode
		add_shortcode( 'mtm_ai_links', [ $this, 'render_shortcode' ] );

		// Add "Settings" link to plugin row
    	add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), [ $this, 'add_settings_link' ] );
	}

	/**
	 * Load plugin textdomain for translations.
	 */
	// public function load_textdomain() {
	// 	load_plugin_textdomain( 'mtm-ai-link-generator', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	// }

	/**
	 * Adds a "Settings" link to the plugin on the Plugins page
	 */
	public function add_settings_link( $links ) {
		$settings_link = '<a href="' . admin_url( 'options-general.php?page=mtm-ai-links' ) . '">' . esc_html__( 'Settings', 'mtm-ai-link-generator' ) . '</a>';
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Register the settings page under "Settings".
	 */
	public function register_settings_page() {
		add_options_page(
			__( 'MTM AI Links', 'mtm-ai-link-generator' ),
			__( 'MTM AI Links', 'mtm-ai-link-generator' ),
			'manage_options',
			'mtm-ai-links',
			[ $this, 'render_settings_page' ]
		);
	}

	/**
	 * Register plugin settings using the Settings API.
	 */
	public function register_settings() {
		// Register a single option that will hold all plugin settings
		register_setting(
			'mtm_ai_links_group',      // option group
			self::OPTION_NAME,        // option name (array)
			[ $this, 'sanitize_options' ] // sanitize callback
		);

		// Add a section
		add_settings_section(
			'mtm_ai_links_section',
			__( 'AI Search Links Settings', 'mtm-ai-link-generator' ),
			[ $this, 'render_section_description' ],
			'mtm-ai-links'
		);

		// Section title
		add_settings_field(
			'section_title',
			__( 'Section title', 'mtm-ai-link-generator' ),
			[ $this, 'render_text_field' ],
			'mtm-ai-links',
			'mtm_ai_links_section',
			[
				'id'          => 'section_title',
				'placeholder' => __( 'e.g. Ask AI for a summary', 'mtm-ai-link-generator' ),
			]
		);

		// Title color
		add_settings_field(
			'title_color',
			__( 'Title Color', 'mtm-ai-link-generator' ),
			[ $this, 'render_color_field' ],
			'mtm-ai-links',
			'mtm_ai_links_section',
			[
				'id' => 'title_color'
			]
		);

		// Global prompt
		add_settings_field(
			'global_text',
			__( 'Global search text (prompt)', 'mtm-ai-link-generator' ),
			[ $this, 'render_text_field' ],
			'mtm-ai-links',
			'mtm_ai_links_section',
			[
				'id'          => 'global_text',
				'placeholder' => __( 'Enter the prompt used by AIs', 'mtm-ai-link-generator' ),
			]
		);

		// AI platforms fields: enable checkbox + optional override text
		$ais = [
			'chatgpt'    => __( 'ChatGPT', 'mtm-ai-link-generator' ),
			'claude'     => __( 'Claude', 'mtm-ai-link-generator' ),
			'perplexity' => __( 'Perplexity', 'mtm-ai-link-generator' ),
			'google'     => __( 'Google/Gemini', 'mtm-ai-link-generator' ),
			'grok'       => __( 'Grok (X)', 'mtm-ai-link-generator' ),
		];

		foreach ( $ais as $key => $label ) {
			// enable checkbox
			add_settings_field(
				$key . '_enable',
				sprintf( /* translators: %s: AI label */ __( '%s - Enable', 'mtm-ai-link-generator' ), $label ),
				[ $this, 'render_checkbox_field' ],
				'mtm-ai-links',
				'mtm_ai_links_section',
				[ 'id' => $key . '_enable' ]
			);

			// optional custom text
			add_settings_field(
				$key . '_text',
				sprintf( /* translators: %s: AI label */ __( '%s - Optional prompt', 'mtm-ai-link-generator' ), $label ),
				[ $this, 'render_text_field' ],
				'mtm-ai-links',
				'mtm_ai_links_section',
				[
					'id'          => $key . '_text',
					'placeholder' => sprintf( /* translators: %s: AI label */ __( 'Optional custom prompt for %s', 'mtm-ai-link-generator' ), $label ),
				]
			);
		}
	}

	/**
	 * Sanitize all settings in a single array.
	 *
	 * @param array $input Raw input from options form.
	 * @return array Sanitized options.
	 */
	public function sanitize_options( $input ) {
		$clean = $this->defaults;

		if ( ! is_array( $input ) ) {
			// Unexpected input — return defaults
			return $clean;
		}

		// Text fields: sanitize_text_field
		$clean['section_title'] = isset( $input['section_title'] ) ? sanitize_text_field( $input['section_title'] ) : '';
		$clean['global_text']   = isset( $input['global_text'] ) ? sanitize_textarea_field( $input['global_text'] ) : '';
		$clean['title_color'] = isset( $input['title_color'] ) ? sanitize_hex_color( $input['title_color'] ) : '#000000';

		// Manage AI toggles and optional text
		$ai_keys = [ 'chatgpt', 'claude', 'perplexity', 'google', 'grok' ];
		foreach ( $ai_keys as $k ) {
			$enable_key = $k . '_enable';
			$text_key   = $k . '_text';

			$clean[ $enable_key ] = ! empty( $input[ $enable_key ] ) ? 1 : 0;
			$clean[ $text_key ]   = isset( $input[ $text_key ] ) ? sanitize_text_field( $input[ $text_key ] ) : '';
		}

		return $clean;
	}

	/**
	 * Render description for settings section.
	 */
	public function render_section_description() {
		/* translators: description for settings page */
		echo '<p>' . esc_html__( 'Configure which AI search links to show and the default prompt used when a specific AI override is not provided. Use the shortcode [mtm_ai_links] to render the links on any page or post.', 'mtm-ai-link-generator' ) . '</p>';
	}

	/**
	 * Render a text input field (used for text and small textarea).
	 *
	 * $args must include 'id' and may include 'placeholder'.
	 *
	 * @param array $args Field arguments passed by add_settings_field.
	 */
	public function render_text_field( $args ) {
		$options     = get_option( self::OPTION_NAME, $this->defaults );
		$id          = isset( $args['id'] ) ? $args['id'] : '';
		$value       = isset( $options[ $id ] ) ? $options[ $id ] : '';
		$placeholder = isset( $args['placeholder'] ) ? $args['placeholder'] : '';

		// If this is the global_text we prefer a textarea for longer prompts
		if ( 'global_text' === $id ) {
			printf(
				'<textarea name="%1$s[%2$s]" rows="4" class="large-text" placeholder="%3$s">%4$s</textarea>',
				esc_attr( self::OPTION_NAME ),
				esc_attr( $id ),
				esc_attr( $placeholder ),
				esc_textarea( $value )
			);
		} else {
			printf(
				'<input type="text" name="%1$s[%2$s]" value="%3$s" class="regular-text" placeholder="%4$s" />',
				esc_attr( self::OPTION_NAME ),
				esc_attr( $id ),
				esc_attr( $value ),
				esc_attr( $placeholder )
			);
		}

		// Add a small separator to make the form cleaner
		echo '<p class="description" style="margin-top:6px;"></p>';
	}

	/**
	 * Render a checkbox field.
	 *
	 * @param array $args Field arguments.
	 */
	public function render_checkbox_field( $args ) {
		$options = get_option( self::OPTION_NAME, $this->defaults );
		$id      = isset( $args['id'] ) ? $args['id'] : '';
		$value   = ! empty( $options[ $id ] ) ? 1 : 0;

		printf(
			'<label><input type="checkbox" name="%1$s[%2$s]" value="1" %3$s /> %4$s</label>',
			esc_attr( self::OPTION_NAME ),
			esc_attr( $id ),
			checked( 1, $value, false ),
			esc_html__( 'Enable', 'mtm-ai-link-generator' )
		);
	}

	/**
	 * Render a color picker field.
	 */
	public function render_color_field( $args ) {
		$options = get_option( self::OPTION_NAME, $this->defaults );
		$id      = isset( $args['id'] ) ? $args['id'] : '';
		$value   = isset( $options[ $id ] ) ? $options[ $id ] : '#000000';

		printf(
			'<input type="color" name="%1$s[%2$s]" value="%3$s" />',
			esc_attr( self::OPTION_NAME ),
			esc_attr( $id ),
			esc_attr( $value )
		);
	}

	/**
	 * Render the settings page HTML.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'mtm-ai-link-generator' ) );
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'MTM AI Links Settings', 'mtm-ai-link-generator' ); ?></h1>
			<form method="post" action="options.php">
				<?php
				settings_fields( 'mtm_ai_links_group' );
				do_settings_sections( 'mtm-ai-links' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Enqueue frontend assets (CSS). Only enqueue when shortcode is present in the post content or on archive / single pages.
	 */
	public function enqueue_frontend_assets() {
		// We keep this light — only a small CSS file
		$handle = 'mtm-ai-links-style';
		$src    = $this->plugin_url . 'assets/mtm-ai-link.css';

		// If stylesheet file does not exist, skip
		if ( ! file_exists( $this->plugin_path . 'assets/mtm-ai-link.css' ) ) {
			return;
		}

		wp_register_style( $handle, $src, [], self::VERSION );
		// Enqueue on all pages — you could optimize by detecting shortcode in post content.
		wp_enqueue_style( $handle );
	}

	/**
	 * Render the shortcode output. Returns HTML string.
	 *
	 * Shortcode: [mtm_ai_links]
	 *
	 * Uses sanitized options and escapes all output.
	 *
	 * @return string HTML string (escaped).
	 */
	public function render_shortcode() {
		$options = get_option( self::OPTION_NAME, $this->defaults );

		// If no prompt defined and no overrides, return empty string
		if ( empty( $options['global_text'] ) ) {
			// Also allow building from individual AI texts (optional) – if none, return empty
			$has_any = false;
			foreach ( [ 'chatgpt', 'claude', 'perplexity', 'google', 'grok' ] as $k ) {
				if ( ! empty( $options[ $k . '_enable' ] ) ) {
					$has_any = true;
					break;
				}
			}
			if ( ! $has_any ) {
				return '';
			}
		}

		// Map AI definitions: url template (append url-encoded prompt) and image path
		$ais = [
			'chatgpt' => [
				'label' => __( 'ChatGPT', 'mtm-ai-link-generator' ),
				'url'   => 'https://chat.openai.com/?q=',
				'img'   => $this->get_asset_url( 'assets/images/chatgpt.svg' ),
				'alt'   => __( 'ChatGPT', 'mtm-ai-link-generator' ),
			],
			'claude' => [
				'label' => __( 'Claude', 'mtm-ai-link-generator' ),
				'url'   => 'https://claude.ai/new?q=',
				'img'   => $this->get_asset_url( 'assets/images/claude.svg' ),
				'alt'   => __( 'Claude', 'mtm-ai-link-generator' ),
			],
			'perplexity' => [
				'label' => __( 'Perplexity', 'mtm-ai-link-generator' ),
				'url'   => 'https://www.perplexity.ai/search/new?q=',
				'img'   => $this->get_asset_url( 'assets/images/perplexity.svg' ),
				'alt'   => __( 'Perplexity', 'mtm-ai-link-generator' ),
			],
			'google' => [
				'label' => __( 'Google/Gemini', 'mtm-ai-link-generator' ),
				'url'   => 'https://www.google.com/search?q=',
				'img'   => $this->get_asset_url( 'assets/images/gemini.svg' ),
				'alt'   => __( 'Google / Gemini', 'mtm-ai-link-generator' ),
			],
			'grok' => [
				'label' => __( 'Grok (X)', 'mtm-ai-link-generator' ),
				'url'   => 'https://x.com/i/grok?text=',
				'img'   => $this->get_asset_url( 'assets/images/grok.svg' ),
				'alt'   => __( 'Grok (X)', 'mtm-ai-link-generator' ),
			],
		];

		// Build output
		$container_class = 'mtm-ai-links-container';
		$items           = [];

		foreach ( $ais as $key => $ai ) {
			$enabled = ! empty( $options[ $key . '_enable' ] );
			if ( ! $enabled ) {
				continue;
			}

			$custom_text = ! empty( $options[ $key . '_text' ] ) ? $options[ $key . '_text' ] : $options['global_text'];
			if ( '' === trim( $custom_text ) ) {
				// if no text at all, skip
				continue;
			}

			$href = $ai['url'] . rawurlencode( $custom_text );

			// Safe: ensure we have an image URL (fallback to empty)
			$img = '';
			if ( ! empty( $ai['img'] ) ) {
				$img = esc_url( $ai['img'] );
			}

			$aria_label = sprintf(
				/* translators: %s = AI tool name (ChatGPT, Claude, Gemini, etc.) */
				__('Open this resource in %s (opens in a new tab)', 'mtm-ai-link-generator'),
				$ai['label']
			);

			$items[] = [
				'href' 			=> $href,
				'img'  			=> $img,
				'alt'  			=> $ai['alt'],
				'label'			=> $ai['label'],
				'aria_label' 	=> $aria_label
			];
		}

		// If no items, return empty
		if ( empty( $items ) ) {
			return '';
		}

		// Start output buffer
		ob_start();
		?>
		<div class="<?php echo esc_attr( $container_class ); ?>">
			<?php if ( ! empty( $options['section_title'] ) ) : ?>
				<h2 class="mtm-ai-links-title" style="color:<?php echo esc_attr( $options['title_color'] ); ?>">
					<?php echo esc_html( $options['section_title'] ); ?>
				</h2>
			<?php endif; ?>

			<ul class="mtm-ai-links-list">
				<?php foreach ( $items as $it ) : ?>
					<li class="mtm-ai-links-item">
						<a href="<?php echo esc_url( $it['href'] ); ?>" target="_blank" rel="noopener noreferrer" aria-label="<?php echo esc_attr( $it['aria_label'] ); ?>">
							<?php if ( ! empty( $it['img'] ) ) : ?>
								<img class="mtm-ai-links-img" src="<?php echo esc_url($it['img']); ?>" alt="<?php echo esc_attr( $it['alt'] ?? '' ); ?>" width="48" height="48" />
							<?php else : ?>
								<span class="mtm-ai-links-text"><?php echo esc_html( $it['label'] ); ?></span>
							<?php endif; ?>
						</a>
					</li>
				<?php endforeach; ?>
			</ul>
		</div>
		<?php
		$html = ob_get_clean();

		// Return sanitized HTML (allow only the markup we generated)
		$allowed = [
			'div'   => [ 'class' => true ],
			'h2'    => [ 'class' => true, 'style' => true ],
			'ul'    => [ 'class' => true ],
			'li'    => [ 'class' => true ],
			'a'     => [ 'href' => true, 'target' => true, 'rel' => true, 'aria-label' => true ],
			'img'   => [ 'src' => true, 'alt' => true, 'width' => true, 'height' => true, 'class' => true ],
			'span'  => [ 'class' => true ],
		];

		return wp_kses( $html, $allowed );
	}

	/**
	 * Helper: return plugin asset URL if file exists, otherwise empty string.
	 *
	 * @param string $relative Relative path inside plugin folder.
	 * @return string URL or empty string.
	 */
	private function get_asset_url( $relative ) {
		$path = $this->plugin_path . ltrim( $relative, '/\\' );
		if ( file_exists( $path ) ) {
			return $this->plugin_url . ltrim( $relative, '/\\' );
		}
		return '';
	}
}

/**
 * Initialize the plugin.
 */
function mtm_ai_links_init() {
	// Only initialize once
	static $instance = null;
	if ( null === $instance ) {
		$instance = new MTM_AI_Links_Plugin();
	}
}
add_action( 'plugins_loaded', 'mtm_ai_links_init' );
