<?php
/*
Plugin Name: Motendo Widget
Description: Wyświetla ostatnie ogłoszenia lub promowane elementy z Motendo.
Version: 1.2
Author: D@ni3l Ajeto
License: GPLv2 or later
*/

namespace MotendoWidget;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Zabezpieczenie przed bezpośrednim dostępem
}

/**
 * Globalna funkcja sanitizująca wartość checkboxa.
 *
 * @param string $input
 * @return string
 */
if ( ! function_exists( 'MotendoWidget\motendo_widget_sanitize_checkbox' ) ) {
	function motendo_widget_sanitize_checkbox( $input ) {
		return ( 'yes' === $input ) ? 'yes' : 'no';
	}
}

/**
 * Główna klasa wtyczki.
 */
class Plugin {
	/** Czas ważności cache’u (w sekundach). */
	const TRANSIENT_EXPIRATION = HOUR_IN_SECONDS;

	public function __construct() {
		// Ładowanie tłumaczeń
		add_action( 'plugins_loaded', [ $this, 'load_textdomain' ] );
		// Rejestracja shortcode
		add_shortcode( 'motendo_widget', [ $this, 'shortcode_handler' ] );
		// Ładowanie stylów
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
		// Automatyczne wstawianie widgetu pod treścią wpisu
		add_filter( 'the_content', [ $this, 'auto_insert_widget' ] );
		// Rejestracja ustawień w panelu administratora
		add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
		add_action( 'admin_init', [ $this, 'register_settings' ] );
		// Rejestracja widgetu
		add_action( 'widgets_init', [ $this, 'register_widget' ] );
	}

	/**
	 * Ładuje pliki tłumaczeń.
	 */
	public function load_textdomain() {
		load_plugin_textdomain( 'motendo-widget', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Callback shortcode.
	 *
	 * @param array $atts
	 * @return string
	 */
	public function shortcode_handler( $atts ) {
		return $this->render_widget( $atts, false );
	}

	/**
	 * Automatyczne wstawianie widgetu do treści posta.
	 *
	 * @param string $content
	 * @return string
	 */
	public function auto_insert_widget( $content ) {
		if ( is_single() && in_the_loop() && is_main_query() && get_option( 'motendo_widget_auto_insert', 'yes' ) === 'yes' ) {
			$atts        = [
				'source'  => get_option( 'motendo_widget_source', 'latesearch' ),
				'limit'   => get_option( 'motendo_widget_limit', 3 ),
				'lang'    => get_option( 'motendo_widget_lang', 'pl' ),
				'format'  => get_option( 'motendo_widget_format', 'json' ),
				'phrase'  => get_option( 'motendo_widget_phrase', '' ),
				'user_id' => get_option( 'motendo_widget_user_id', '' ),
			];
			$widget_html = $this->render_widget( $atts, false );
			$content    .= '<div class="motendo-widget-container">' . $widget_html . '</div>';
		}
		return $content;
	}

	/**
	 * Buduje URL do API przy użyciu http_build_query.
	 *
	 * @param array $atts
	 * @return string
	 */
	protected function build_api_url( $atts ) {
		$source  = sanitize_text_field( $atts['source'] ?? 'latesearch' );
		$limit   = intval( $atts['limit'] ?? 3 );
		$lang    = sanitize_text_field( $atts['lang'] ?? 'pl' );
		$format  = sanitize_text_field( $atts['format'] ?? 'json' );
		$phrase  = sanitize_text_field( $atts['phrase'] ?? '' );
		$user_id = intval( $atts['user_id'] ?? 0 );

		// Jeśli nie podano frazy – próba pobrania tagów wpisu
		if ( empty( $phrase ) ) {
			global $post;
			$tags = get_the_tags( $post->ID );
			if ( ! empty( $tags ) && is_array( $tags ) ) {
				$tag_names = array_map( function( $tag ) {
					return $tag->name;
				}, $tags );
				$phrase = sanitize_text_field( $tag_names[0] );
			}
		}

		$query_args = [
			'source'        => $source,
			'desktop_limit' => $limit,
			'mobile_limit'  => $limit,
			'format'        => $format,
			'lang'          => $lang,
			'show_city'     => 'FALSE',
			'show_image'    => 'TRUE',
			'show_cost'     => 'FALSE',
			'phrase'        => $phrase,
		];
		if ( $user_id ) {
			$query_args['user_id'] = $user_id;
		}
		$api_url = 'https://www.motendo.com/widget/?' . http_build_query( $query_args );
		return $api_url;
	}

	/**
	 * Pobiera dane z API z mechanizmem cache’owania.
	 *
	 * @param string $api_url
	 * @return array|\WP_Error
	 */
	protected function get_api_data( $api_url ) {
		$transient_key = 'motendo_widget_' . md5( $api_url );
		$data          = get_transient( $transient_key );
		if ( false === $data ) {
			$response = wp_remote_get( $api_url, [ 'timeout' => 10 ] );
			if ( is_wp_error( $response ) ) {
				return new \WP_Error( 'motendo_widget_error', __( 'Nie udało się pobrać danych.', 'motendo-widget' ) );
			}
			$body = wp_remote_retrieve_body( $response );
			$data = json_decode( $body, true );
			set_transient( $transient_key, $data, apply_filters( 'motendo_widget_transient_expiration', self::TRANSIENT_EXPIRATION ) );
		}
		return $data;
	}

	/**
	 * Renderuje zawartość widgetu.
	 *
	 * @param array $atts
	 * @param bool  $is_sidebar Czy wywoływane z widgetu
	 * @return string
	 */
	public function render_widget( $atts, $is_sidebar = false ) {
		$atts = shortcode_atts(
			[
				'source'  => 'latesearch',
				'limit'   => 3,
				'lang'    => 'pl',
				'format'  => 'json',
				'phrase'  => '',
				'user_id' => '',
			],
			$atts,
			'motendo_widget'
		);

		$api_url = $this->build_api_url( $atts );

		if ( sanitize_text_field( $atts['format'] ) === 'html' ) {
			$iframe = sprintf(
				'<iframe src="%s" style="display:block;margin:0 auto;border:none;height:400px;width:100%%;max-width:700px;" id="iframe_motendo_widget"></iframe>',
				esc_url( $api_url )
			);
			return apply_filters( 'motendo_widget_output', $iframe, $atts );
		}

		$data = $this->get_api_data( $api_url );
		if ( is_wp_error( $data ) ) {
			return esc_html( $data->get_error_message() );
		}
		if ( empty( $data['desktop_items'] ) ) {
			return esc_html__( 'Brak wyników do wyświetlenia.', 'motendo-widget' );
		}
		$extra_class = $is_sidebar ? 'motendo-widget-sidebar' : 'motendo-widget-post';

		ob_start();
		?>
		<div class="<?php echo esc_attr( 'motendo-widget ' . $extra_class ); ?>">
			<?php foreach ( $data['desktop_items'] as $item ) : ?>
				<div class="motendo-item">
					<?php if ( ! empty( $item['image'] ) ) : ?>
						<a href="<?php echo esc_url( $item['url'] ); ?>">
							<?php
							$item_title = isset( $item['title'] ) ? esc_attr( $item['title'] ) : '';
							// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
							echo $this->render_image( $item['image'], $item_title );
							?>
						</a>
					<?php endif; ?>
					<h4>
						<a href="<?php echo esc_url( $item['url'] ); ?>">
							<?php echo isset( $item['title'] ) ? esc_html( $item['title'] ) : ''; ?>
						</a>
					</h4>
				</div>
			<?php endforeach; ?>
		</div>
		<?php
		$output = ob_get_clean();
		return apply_filters( 'motendo_widget_output', $output, $data, $atts );
	}

	/**
	 * Renderuje znacznik <img> – zapewnia domyślny obraz, jeśli URL jest pusty.
	 *
	 * @param string $url
	 * @param string $alt
	 * @return string
	 */
	protected function render_image( $url, $alt = '' ) {
            // Pobranie lokalnego obrazu blank.jpg z katalogu assets/img/
            $default_image = plugins_url( 'assets/img/blank.jpg', __FILE__ );
            $url           = ! empty( $url ) ? esc_url( $url ) : esc_url( $default_image );
            $img_html      = sprintf(
                '<img src="%s" alt="%s" class="motendo-widget-image" />',
                $url,
                esc_attr( $alt )
            );
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            return wp_filter_content_tags( $img_html );
        }

	/**
	 * Ładuje pliki CSS.
	 */
	public function enqueue_assets() {
		wp_enqueue_style( 'motendo-widget-style', plugins_url( 'style.css', __FILE__ ), [], '1.2' );
	}

	/**
	 * Dodaje stronę ustawień do panelu administratora.
	 */
	public function add_settings_page() {
		add_options_page(
			esc_html__( 'Motendo Widget Settings', 'motendo-widget' ),
			esc_html__( 'Motendo Widget', 'motendo-widget' ),
			'manage_options',
			'motendo_widget',
			[ $this, 'render_settings_page' ]
		);
	}

	/**
	 * Wyświetla stronę ustawień.
	 */
	public function render_settings_page() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Ustawienia Motendo Widget', 'motendo-widget' ); ?></h1>
			<form method="post" action="options.php">
				<?php
				settings_fields( 'motendo_widget_settings' );
				do_settings_sections( 'motendo_widget' );
				submit_button();
				?>
			</form>
			<h2><?php esc_html_e( 'Opis funkcjonalności', 'motendo-widget' ); ?></h2>
			<p><?php esc_html_e( 'Plugin Motendo Widget umożliwia wyświetlanie ogłoszeń z serwisu Motendo pod każdym postem lub w dowolnym miejscu za pomocą shortcode.', 'motendo-widget' ); ?></p>
			<h3><?php esc_html_e( 'Jak używać shortcode?', 'motendo-widget' ); ?></h3>
			<p><?php esc_html_e( 'Aby wstawić widget w treści strony lub wpisu, użyj następującego shortcode:', 'motendo-widget' ); ?></p>
			<pre>[motendo_widget source="latesearch" limit="3" phrase="rolnicze" lang="pl" format="json"]</pre>
		</div>
		<?php
	}

	/**
	 * Rejestruje ustawienia w panelu administratora.
	 */
	public function register_settings() {
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_auto_insert', [ 'sanitize_callback' => 'MotendoWidget\motendo_widget_sanitize_checkbox' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_limit', [ 'sanitize_callback' => 'absint' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_source', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_lang', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_phrase', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_format', [ 'sanitize_callback' => 'sanitize_text_field' ] );
		/* phpcs:ignore PluginCheck.CodeAnalysis.SettingSanitization.register_settingDynamic */
		register_setting( 'motendo_widget_settings', 'motendo_widget_user_id', [ 'sanitize_callback' => 'absint' ] );

		add_settings_section( 'motendo_widget_main', esc_html__( 'Główne ustawienia', 'motendo-widget' ), null, 'motendo_widget' );

		add_settings_field( 'motendo_widget_auto_insert', esc_html__( 'Wyświetl pod każdym postem', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_auto_insert', 'yes' );
			echo '<input type="checkbox" name="motendo_widget_auto_insert" value="yes" ' . checked( $value, 'yes', false ) . '> ' . esc_html__( 'Tak', 'motendo-widget' );
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_limit', esc_html__( 'Liczba ogłoszeń', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_limit', 3 );
			echo '<input type="number" name="motendo_widget_limit" value="' . esc_attr( $value ) . '" min="1">';
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_source', esc_html__( 'Źródło danych', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_source', 'latesearch' );
			echo '<select name="motendo_widget_source">
					<option value="latesearch" ' . selected( $value, 'latesearch', false ) . '>' . esc_html__( 'Fraza z tagów', 'motendo-widget' ) . '</option>
					<option value="latest" ' . selected( $value, 'latest', false ) . '>' . esc_html__( 'Ostatnie ogłoszenia', 'motendo-widget' ) . '</option>
				  </select>';
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_user_id', esc_html__( 'ID użytkownika', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_user_id', '' );
			echo '<input type="text" name="motendo_widget_user_id" value="' . esc_attr( $value ) . '">';
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_lang', esc_html__( 'Język', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_lang', 'pl' );
			echo '<select name="motendo_widget_lang">
					<option value="pl" ' . selected( $value, 'pl', false ) . '>' . esc_html__( 'Polski', 'motendo-widget' ) . '</option>
					<option value="en" ' . selected( $value, 'en', false ) . '>' . esc_html__( 'Angielski', 'motendo-widget' ) . '</option>
					<option value="de" ' . selected( $value, 'de', false ) . '>' . esc_html__( 'Niemiecki', 'motendo-widget' ) . '</option>
					<option value="ua" ' . selected( $value, 'ua', false ) . '>' . esc_html__( 'Ukraiński', 'motendo-widget' ) . '</option>
				  </select>';
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_phrase', esc_html__( 'Szukana fraza (jeśli brak tagów)', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_phrase', '' );
			echo '<input type="text" name="motendo_widget_phrase" value="' . esc_attr( $value ) . '">';
		}, 'motendo_widget', 'motendo_widget_main' );

		add_settings_field( 'motendo_widget_format', esc_html__( 'Format danych', 'motendo-widget' ), function () {
			$value = get_option( 'motendo_widget_format', 'json' );
			echo '<select name="motendo_widget_format">
					<option value="json" ' . selected( $value, 'json', false ) . '>' . esc_html__( 'JSON', 'motendo-widget' ) . '</option>
					<option value="html" ' . selected( $value, 'html', false ) . '>' . esc_html__( 'HTML', 'motendo-widget' ) . '</option>
				  </select>';
		}, 'motendo_widget', 'motendo_widget_main' );
	}

	/**
	 * Rejestracja widgetu.
	 */
	public function register_widget() {
		register_widget( 'MotendoWidget\Widget' );
	}
}

// Tworzymy globalną instancję wtyczki, aby była używana przez widget.
$GLOBALS['motendo_widget_instance'] = new Plugin();

/**
 * Klasa widgetu – umożliwia dodanie wtyczki do obszaru widgetów.
 */
class Widget extends \WP_Widget {
	public function __construct() {
		parent::__construct(
			'motendo_widget',
			esc_html__( 'Motendo Widget', 'motendo-widget' ),
			[ 'description' => esc_html__( 'Wyświetla ogłoszenia z Motendo w sidebarze lub innych miejscach z widżetami.', 'motendo-widget' ) ]
		);
	}

	/**
	 * Wyświetlanie widgetu.
	 *
	 * @param array $args
	 * @param array $instance
	 */
	public function widget( $args, $instance ) {
		echo wp_kses_post( $args['before_widget'] );
		if ( ! empty( $instance['title'] ) ) {
			echo wp_kses_post( $args['before_title'] ) . esc_html( $instance['title'] ) . wp_kses_post( $args['after_title'] );
		}
		$atts = [
			'source'  => ! empty( $instance['source'] ) ? $instance['source'] : 'latesearch',
			'limit'   => ! empty( $instance['limit'] ) ? intval( $instance['limit'] ) : 3,
			'lang'    => ! empty( $instance['lang'] ) ? $instance['lang'] : 'pl',
			'format'  => ! empty( $instance['format'] ) ? $instance['format'] : 'json',
			'phrase'  => ! empty( $instance['phrase'] ) ? $instance['phrase'] : '',
			'user_id' => ! empty( $instance['user_id'] ) ? intval( $instance['user_id'] ) : '',
		];
		global $motendo_widget_instance;
		echo wp_kses_post( $motendo_widget_instance->render_widget( $atts, true ) );
		echo wp_kses_post( $args['after_widget'] );
	}

	/**
	 * Formularz w panelu widgetu.
	 *
	 * @param array $instance
	 */
	public function form( $instance ) {
		$defaults = [
			'title'  => esc_html__( 'Ogłoszenia Motendo', 'motendo-widget' ),
			'source' => 'latesearch',
			'limit'  => 1,
			'lang'   => 'pl',
			'phrase' => '',
			'format' => 'json',
			'user_id'=> '',
		];
		$instance = wp_parse_args( (array) $instance, $defaults );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Tytuł:', 'motendo-widget' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $instance['title'] ); ?>">
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'source' ) ); ?>"><?php esc_html_e( 'Źródło danych:', 'motendo-widget' ); ?></label>
			<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'source' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'source' ) ); ?>">
				<option value="latesearch" <?php selected( $instance['source'], 'latesearch' ); ?>><?php esc_html_e( 'Fraza z tagów', 'motendo-widget' ); ?></option>
				<option value="latest" <?php selected( $instance['source'], 'latest' ); ?>><?php esc_html_e( 'Ostatnie ogłoszenia', 'motendo-widget' ); ?></option>
			</select>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>"><?php esc_html_e( 'Liczba ogłoszeń:', 'motendo-widget' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'limit' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'limit' ) ); ?>" type="number" value="<?php echo esc_attr( $instance['limit'] ); ?>" min="1">
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'lang' ) ); ?>"><?php esc_html_e( 'Język:', 'motendo-widget' ); ?></label>
			<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'lang' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'lang' ) ); ?>">
				<option value="pl" <?php selected( $instance['lang'], 'pl' ); ?>><?php esc_html_e( 'Polski', 'motendo-widget' ); ?></option>
				<option value="en" <?php selected( $instance['lang'], 'en' ); ?>><?php esc_html_e( 'Angielski', 'motendo-widget' ); ?></option>
				<option value="de" <?php selected( $instance['lang'], 'de' ); ?>><?php esc_html_e( 'Niemiecki', 'motendo-widget' ); ?></option>
				<option value="ua" <?php selected( $instance['lang'], 'ua' ); ?>><?php esc_html_e( 'Ukraiński', 'motendo-widget' ); ?></option>
			</select>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'phrase' ) ); ?>"><?php esc_html_e( 'Szukana fraza:', 'motendo-widget' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'phrase' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'phrase' ) ); ?>" type="text" value="<?php echo esc_attr( $instance['phrase'] ); ?>">
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'format' ) ); ?>"><?php esc_html_e( 'Format:', 'motendo-widget' ); ?></label>
			<select class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'format' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'format' ) ); ?>">
				<option value="json" <?php selected( $instance['format'], 'json' ); ?>><?php esc_html_e( 'JSON', 'motendo-widget' ); ?></option>
				<option value="html" <?php selected( $instance['format'], 'html' ); ?>><?php esc_html_e( 'HTML', 'motendo-widget' ); ?></option>
			</select>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'user_id' ) ); ?>"><?php esc_html_e( 'ID użytkownika:', 'motendo-widget' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'user_id' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'user_id' ) ); ?>" type="text" value="<?php echo esc_attr( $instance['user_id'] ); ?>">
		</p>
		<?php
	}

	/**
	 * Aktualizuje ustawienia widgetu.
	 *
	 * @param array $new_instance
	 * @param array $old_instance
	 * @return array
	 */
	public function update( $new_instance, $old_instance ) {
		$instance                = $old_instance;
		$instance['title']       = sanitize_text_field( $new_instance['title'] );
		$instance['source']      = sanitize_text_field( $new_instance['source'] );
		$instance['limit']       = absint( $new_instance['limit'] );
		$instance['lang']        = sanitize_text_field( $new_instance['lang'] );
		$instance['phrase']      = sanitize_text_field( $new_instance['phrase'] );
		$instance['format']      = sanitize_text_field( $new_instance['format'] );
		$instance['user_id']     = absint( $new_instance['user_id'] );
		return $instance;
	}
}
