<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer;

/**
 * Fires during plugin uninstallation
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * If uninstall not called from WordPress, then die.
 */
if (! defined('WP_UNINSTALL_PLUGIN')) wp_die('Something is not right.');

/**
 * If the current user is not allowed to activate plugins, then die.
 */
if (! current_user_can('activate_plugins')) wp_die('Something is not right.');

/**
 * All function calls to be fired on uninstallation.
 */
Mop_Ai_Indexer_Uninstaller::delete_options();
Mop_Ai_Indexer_Uninstaller::delete_generated_files();

/**
 * Contains methods to be fired during plugin uninstallation.
 *
 * This class is to contain methods to delete plugin specific options,
 * transients, database tables, meta data, posts of custom post types,
 * posts and pages, uploads directory with uploaded files, user roles
 * and user capabilities.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Uninstaller {

	/**
	 * Delete plugin specific options.
	 *
	 * @since    1.0.0
	 */
	public static function delete_options(): void {

		$options = array(
			'mop_ai_indexer_iset',
			'mop_ai_indexer_config',
			'mop_ai_indexer_last_generated',
			'mop_ai_indexer_last_generated_file',
			'mop_ai_indexer_scheduled_job_state',
			'mop_ai_indexer_automation_secret',
			'mop-ai-indexer-index-generation-logs',
			'mop-ai-indexer-index-file-meta'
		);

		foreach ($options as $option) {
			delete_option($option);
		}
	}

	/**
	 * Delete generated index files from the plugin uploads directory.
	 *
	 * The generator writes a temp file first and then renames it to one of the supported
	 * filenames (llms.txt, llms-full.txt). We delete all supported variants to be safe.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public static function delete_generated_files(): void {

		$upload_dir = wp_upload_dir();
		$basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$basedir = wp_normalize_path($basedir);

		if ($basedir === '') return;

		$plugin_uploads_path = wp_normalize_path(trailingslashit($basedir) . 'mop-ai-indexer/');
		if (! is_dir($plugin_uploads_path)) return;

		self::delete_dir_recursive($plugin_uploads_path);
	}

	/**
	 * Recursively delete a directory.
	 *
	 * @param  string $dir Directory path.
	 * @return void
	 */
	private static function delete_dir_recursive(string $dir): void {

		$dir = wp_normalize_path($dir);
		if ($dir === '' || ! is_dir($dir)) return;

		$items = @scandir($dir); // phpcs:ignore WordPress.WP.AlternativeFunctions.scandir_scandir -- Cleanup of plugin-owned generated files in allowlisted uploads directory.
		if (! is_array($items)) return;

		foreach ($items as $item) {

			if ($item === '.' || $item === '..') continue;

			$path = wp_normalize_path(trailingslashit($dir) . (string)$item);

			if (is_dir($path)) {
				self::delete_dir_recursive($path);
				continue;
			}

			if (is_file($path)) {
				@unlink($path); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleanup of plugin-owned generated files in allowlisted uploads directory.
			}
		}

		@rmdir($dir); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir -- Cleanup of plugin-owned generated files in allowlisted uploads directory.
	}
}
