<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer;

/**
 * Plugin Name: MOP AI Indexer
 * Plugin URI: https://ministryofplugins.com/mop-ai-indexer-docs/
 * Description: Automatically generate llms.txt index files from website content for AI/LLM crawlers, with post-type-level and per-post configurations.
 * Version: 1.0.6
 * Requires at least: 5.6
 * Requires PHP: 7.4
 * Author: Anjana Hemachandra
 * Author URI: https://ministryofplugins.com/anjana-hemachandra
 * License: GPL-2.0+
 * License URI: https://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * Text Domain: mop-ai-indexer
 * Domain Path: /languages
 * 
 * This program is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License version 2, as published by the Free Software Foundation. You may NOT assume
 * that you can use any other version of the GPL.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Autoloader to load class files.
 *
 * @since     1.0.0
 * @param     string    $class_name    The class name.
 */
function mop_ai_indexer_autoload(string $class_name): void {

	if (strncmp('Mop_Ai_Indexer', $class_name, strlen('Mop_Ai_Indexer')) !== 0) return;

	$class_map = array();
	$dir_depth = count(explode("\\", $class_name)) - 1;

	if ($dir_depth === 2) {
		$class_map = array('admin/' => 'Mop_Ai_Indexer\\Admin\\',
			'includes/' => 'Mop_Ai_Indexer\\Includes\\');

	} elseif ($dir_depth === 3) {
		$class_map = array('admin/logic/' => 'Mop_Ai_Indexer\\Admin\\Logic\\',
			'includes/logic/' => 'Mop_Ai_Indexer\\Includes\\Logic\\');
	}

	if (empty($class_map)) return;

	foreach ($class_map as $dir_path => $namespace) {

		$namespace_len = strlen($namespace);

		if (strncmp($namespace, $class_name, $namespace_len) === 0) {
	        
			$base_dir  = plugin_dir_path(__FILE__) . $dir_path;
			$file_name = substr($class_name, $namespace_len);
			$file_name = str_replace('_', '-', $file_name);
			$file_name = strtolower($file_name);
			$file_name = 'class-' . $file_name . '.php';
			$file = $base_dir . $file_name;
			$file = wp_normalize_path($file);
			if (file_exists($file)) {
				require $file;
			}
			break;
		}
	}
}
spl_autoload_register(__NAMESPACE__ . '\\mop_ai_indexer_autoload');

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer, Mop_Ai_Indexer_Requirement_Checker};

/**
 * Name of the plugin.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_NAME
 */
define('MOP_AI_INDEXER_NAME', 'MOP AI Indexer');

/**
 * Current version of the plugin.
 * Use SemVer - https://semver.org
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_VERSION
 */
define('MOP_AI_INDEXER_VERSION', '1.0.6');

/**
 * Author of the plugin.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_AUTHOR
 */
define('MOP_AI_INDEXER_AUTHOR', 'Anjana Hemachandra');

/**
 * Plugin URI.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_URI
 */
define('MOP_AI_INDEXER_URI', 'https://ministryofplugins.com/mop-ai-indexer-docs/');

/**
 * Plugin slug.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_SLUG
 */
define('MOP_AI_INDEXER_SLUG', 'mop-ai-indexer');

/**
 * URL directory path (with trailing slash) for the plugin __FILE__ passed in.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_DIR_URL    Example Value: http://localhost/yoursite/wp-content/plugins/mop-ai-indexer/
 */
define('MOP_AI_INDEXER_DIR_URL', plugin_dir_url(__FILE__));

/**
 * Filesystem directory path (with trailing slash) for the plugin __FILE__ passed in.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_DIR_PATH    Example Value: C:\wamp\www\yoursite\wp-content\plugins\mop-ai-indexer/
 */
define('MOP_AI_INDEXER_DIR_PATH', plugin_dir_path(__FILE__));

/**
 * Basename of the plugin.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_BASENAME    Example Value: mop-ai-indexer/mop-ai-indexer.php
 */
define('MOP_AI_INDEXER_BASENAME', plugin_basename(__FILE__));

/**
 * Below constants are plugin parameters.
 * ***These can be re-defined in wp-config.php file, if the website developer needed.***
 */

/**
 * Required minimum PHP version.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_MIN_PHP_VERSION
 */
if (! defined('MOP_AI_INDEXER_MIN_PHP_VERSION')) {
	define('MOP_AI_INDEXER_MIN_PHP_VERSION', '7.4.0');
}

/**
 * Required minimum WordPress version.
 * 
 * @since 1.0.0
 * @var   string   MOP_AI_INDEXER_MIN_WP_VERSION
 */
if (! defined('MOP_AI_INDEXER_MIN_WP_VERSION')) {
	define('MOP_AI_INDEXER_MIN_WP_VERSION', '5.6.0');
}

/**
 * Endpoint behaviour parameters (redefinable constants).
 *
 * These are intentionally implemented as constants (instead of admin UI toggles)
 * so site developers can control behaviour in wp-config.php.
 */

/**
 * Send X-Robots-Tag: noindex, nofollow for endpoint responses.
 *
 * Default: true (recommended).
 *
 * @since 1.0.0
 * @var   bool MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER
 */
if (! defined('MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER')) {
	define('MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER', true);
}

/**
 * Exclude the endpoint from WordPress core XML sitemaps (wp-sitemap.xml).
 *
 * Default: true.
 *
 * Set MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP to false to include the endpoint.
 *
 * Note: MOP AI Indexer will not register a sitemap entry when the noindex header is enabled.
 *
 * @since 1.0.0
 * @var   bool MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP
 */
if (! defined('MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP')) {
	define('MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP', true);
}

/**
 * Output a discovery <link ...> hint in the HTML <head>.
 *
 * Default: true.
 *
 * @since 1.0.0
 * @var   bool MOP_AI_INDEXER_ENDPOINT_HEAD_HINT_ENABLED
 */
if (! defined('MOP_AI_INDEXER_ENDPOINT_HEAD_HINT_ENABLED')) {
	define('MOP_AI_INDEXER_ENDPOINT_HEAD_HINT_ENABLED', true);
}

/**
 * Output a discovery comment in WordPress virtual robots.txt output.
 *
 * Default: true.
 *
 * @since 1.0.0
 * @var   bool MOP_AI_INDEXER_ENDPOINT_ROBOTS_TXT_HINT_ENABLED
 */
if (! defined('MOP_AI_INDEXER_ENDPOINT_ROBOTS_TXT_HINT_ENABLED')) {
	define('MOP_AI_INDEXER_ENDPOINT_ROBOTS_TXT_HINT_ENABLED', true);
}

/**
 * Log endpoint hits to a file in uploads directory.
 *
 * Default: false.
 *
 * @since 1.0.0
 * @var   bool MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS
 */
if (! defined('MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS')) {
	define('MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS', false);
}

/**
 * The code that runs during plugin activation.
 * This action is documented in includes/class-mop-ai-indexer-activator.php
 */
register_activation_hook(__FILE__, array(__NAMESPACE__ . '\\Includes\\Mop_Ai_Indexer_Activator', 'activate'));

/**
 * The code that runs during plugin deactivation.
 * This action is documented in includes/class-mop-ai-indexer-deactivator.php
 */
register_deactivation_hook(__FILE__, array(__NAMESPACE__ . '\\Includes\\Mop_Ai_Indexer_Deactivator', 'deactivate'));

/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then kicking off the plugin from this point in the file does
 * not affect the page life cycle.
 *
 * @since    1.0.0
 * @see      Mop_Ai_Indexer::run()
 * @see      Mop_Ai_Indexer_Requirement_Checker::get_requirement_errors()
 */
function run_mop_ai_indexer(): void {

	/**
	 * Check PHP Version and WordPress Version, then prevent the plugin execution
	 * if it doesn't meet minimum requirements.
	 */
	$requirement_errors = Mop_Ai_Indexer_Requirement_Checker::get_requirement_errors();
	if (empty($requirement_errors)) {
		$plugin = new Mop_Ai_Indexer();
		$plugin->run();
	}
}
run_mop_ai_indexer();