<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Optional XML sitemap integration for MOP AI Indexer endpoints.
 *
 * Default behaviour: exclude the endpoint from XML sitemaps.
 *
 * Control via constants:
 * - MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP (default true). Set to false to include the endpoint
 *
 * Safety rule:
 * - If MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER is enabled (default true), we do NOT register
 *   a sitemap entry even if MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP is false. Including a URL
 *   that is explicitly noindex is confusing and can create false signals.
 *
 * Important implementation note:
 * - WordPress core sitemap filenames use hyphens as delimiters.
 * - Therefore the provider key MUST NOT contain hyphens, otherwise routing breaks and
 *   the request may fall through to the normal frontend (blog page).
 *
 * This file registers the provider key as "mopaiindexer" (no hyphen), which will produce:
 * - /wp-sitemap-mopaiindexer-1.xml
 *
 * Dash fix (and ONLY this fix):
 * - WordPress routing requires the provider key to remain "mopaiindexer".
 * - To expose the dashed URL you want, we provide an alias endpoint:
 *   /wp-sitemap-mop-ai-indexer-1.xml
 *   that outputs the same XML as /wp-sitemap-mopaiindexer-1.xml.
 * - We do NOT change anything else about your logic.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer_Endpoint_Router};

/**
 * Integrates MOP AI Indexer endpoint URLs into WordPress core sitemaps.
 *
 * This class is to integrate MOP AI Indexer endpoint URLs into WordPress core sitemaps.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Sitemap_Integration {

	/**
	 * Register a core sitemap provider when enabled.
	 *
	 * This uses WordPress core sitemaps (wp-sitemap.xml). SEO plugins often provide their own
	 * sitemap stacks; those can be added later as separate integrations.
	 *
	 * @since  1.0.0
	 * @see      Mop_Ai_Indexer_Sitemap_Provider
	 * @return void
	 */
	public function maybe_register_sitemap_provider(\WP_Sitemaps $sitemaps): void {

		$exclude = true;
		if (defined('MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP')) {
			$exclude = (bool) MOP_AI_INDEXER_ENDPOINT_EXCLUDE_IN_SITEMAP;
		}

		if ($exclude) return;

		/**
		 * Do not include if noindex header is enabled.
		 */
		$noindex_enabled = true;
		if (defined('MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER')) {
			$noindex_enabled = (bool) MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER;
		}
		if ($noindex_enabled) return;

		if (! function_exists('wp_sitemaps_get_server')) return;

		$server = wp_sitemaps_get_server();
		if (! $server || ! is_object($server)) return;

		/**
		 * Provider key MUST NOT contain hyphens, otherwise WordPress core may mis-parse
		 * /wp-sitemap-{provider}-{subtype}-{page}.xml style filenames.
		 */
		$provider_key = 'mopaiindexer';
		$provider     = new Mop_Ai_Indexer_Sitemap_Provider();

		/**
		 * Some WP builds expose $server->register_provider(), others do not.
		 */
		if (method_exists($server, 'register_provider')) {
			$server->register_provider($provider_key, $provider);
		} elseif (
			isset($server->registry)
			&& is_object($server->registry)
			&& method_exists($server->registry, 'add_provider')
		) {
			$server->registry->add_provider($provider_key, $provider);
		} else {
			// No compatible registration API found; fail silently.
			return;
		}

		/**
		 * Dash fix: serve /wp-sitemap-mop-ai-indexer-1.xml as an alias that outputs the same XML.
		 * This prevents WordPress from mis-parsing the dashed name and falling back to the blog page.
		 */
		add_action('template_redirect', array($this, 'maybe_render_dashed_alias_sitemap'), 0);
	}

	/**
	 * Dash fix: render the dashed alias sitemap URL as XML.
	 *
	 * URL served:
	 * - /wp-sitemap-mop-ai-indexer-1.xml
	 *
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @return void
	 */
	public function maybe_render_dashed_alias_sitemap(): void {

		$request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash((string)$_SERVER['REQUEST_URI'])) : '';

		$path = $request_uri ? (string) wp_parse_url($request_uri, PHP_URL_PATH) : '';
		$path = $path ? ltrim($path, '/') : '';

		// Only intercept the single alias file we want.
		if ($path !== 'wp-sitemap-mop-ai-indexer-1.xml') {
			return;
		}

		$file_name = Mop_Ai_Indexer_Endpoint_Router::get_canonical_file_name();
		$loc       = Mop_Ai_Indexer_Endpoint_Router::get_endpoint_url($file_name);

		$last_ts = get_option('mop_ai_indexer_last_generated', '');
		$last_ts = is_string($last_ts) ? sanitize_text_field($last_ts) : '';
		$last_ts = absint($last_ts);

		$lastmod = '';
		if ($last_ts > 0) {
			$lastmod = gmdate('c', $last_ts);
		}

		status_header(200);
		header('Content-Type: application/xml; charset=UTF-8');

		echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
		echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
		echo "  <url>\n";
		echo '    <loc>' . esc_html(esc_url($loc)) . "</loc>\n";
		if ($lastmod !== '') {
			echo '    <lastmod>' . esc_html($lastmod) . "</lastmod>\n";
		}
		echo "  </url>\n";
		echo "</urlset>\n";
		exit;
	}
}

/**
 * Core sitemap provider for the MOP AI Indexer endpoint.
 *
 * @since 1.0.0
 */
class Mop_Ai_Indexer_Sitemap_Provider extends \WP_Sitemaps_Provider {

	/**
	 * Provider name used in sitemap route.
	 *
	 * Must match the provider key used during registration.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	public $name = 'mopaiindexer';

	/**
	 * Object type label.
	 *
	 * Must match the provider key used during registration.
	 *
	 * @since 1.0.0
	 * @var   string
	 */
	public $object_type = 'mopaiindexer';

	/**
	 * Return object subtypes for this provider.
	 *
	 * Note: Return types intentionally omitted for maximum compatibility across WP builds.
	 *
	 * @since  1.0.0
	 * @return array
	 */
	public function get_object_subtypes() {
		return array();
	}

	/**
	 * Get a list of URLs for a sitemap page.
	 *
	 * Note: Return types intentionally omitted for maximum compatibility across WP builds.
	 *
	 * @since  1.0.0
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @param  int    $page_num       Page number.
	 * @param  string $object_subtype Optional subtype.
	 * @return array
	 */
	public function get_url_list($page_num, $object_subtype = '') {

		$page_num = absint($page_num);
		if ($page_num > 1) return array();

		$file_name = Mop_Ai_Indexer_Endpoint_Router::get_canonical_file_name();
		$loc       = Mop_Ai_Indexer_Endpoint_Router::get_endpoint_url($file_name);

		$last_ts = get_option('mop_ai_indexer_last_generated', '');
		$last_ts = is_string($last_ts) ? sanitize_text_field($last_ts) : '';
		$last_ts = absint($last_ts);

		$item = array(
			'loc' => $loc,
		);

		if ($last_ts > 0) {
			$item['lastmod'] = gmdate('c', $last_ts);
		}

		return array($item);
	}

	/**
	 * Return the max number of pages for this provider.
	 *
	 * Note: Return types intentionally omitted for maximum compatibility across WP builds.
	 *
	 * @since  1.0.0
	 * @param  string $object_subtype Optional subtype.
	 * @return int
	 */
	public function get_max_num_pages($object_subtype = '') {

		return 1;
	}
}