<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Scheduled generation runner using self-chaining loopback HTTP requests.
 *
 * This runner is designed to mimic the existing step-by-step AJAX generation pipeline,
 * but without a browser. A single WP-Cron trigger starts a job, and each step request
 * dispatches the next step via a signed loopback request.
 *
 * Security model:
 * - A per-site secret is stored in the options table.
 * - Each loopback step request must include a short-lived HMAC signature.
 * - The request must also match the currently active job_id and expected_step.
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_File_Manager};

/**
 * Runs scheduled index generation via WP-Cron using signed step requests.
 *
 * This class is to run scheduled index generation via WP-Cron using signed step requests.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Scheduled_Runner {

	/**
	 * Option key used to store the scheduled job state.
	 */
	private const JOB_STATE_OPTION = 'mop_ai_indexer_scheduled_job_state';

	/**
	 * Option key used to store the HMAC secret for scheduled loopback requests.
	 */
	private const AUTOMATION_SECRET_OPTION = 'mop_ai_indexer_automation_secret';

	/**
	 * AJAX action for the loopback runner endpoint.
	 */
	private const RUNNER_ACTION = 'mop_ai_indexer_scheduled_step_runner';

	/**
	 * Maximum number of invalid signature attempts before temporary rate limiting.
	 */
	private const MAX_INVALID_ATTEMPTS = 25;

	/**
	 * Transient key for invalid attempt counter (short-lived).
	 */
	private const INVALID_ATTEMPTS_TRANSIENT = 'mop_ai_indexer_runner_invalid_attempts';

	/**
	 * Starts a scheduled generation job.
	 *
	 * Called by the WP-Cron hook. This method schedules the first loopback step.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 */
	public function cron_start_scheduled_generation(): void {

		$state = $this->get_job_state();

		if (is_array($state) && isset($state['status']) && $state['status'] === 'running') {
			return;
		}

		$job_id = bin2hex(random_bytes(16));

		$state = array(
			'job_id' => $job_id,
			'status' => 'running',
			'expected_step' => 0,
			'started_at' => current_time('timestamp'),
			'updated_at' => current_time('timestamp'),
		);

		update_option(self::JOB_STATE_OPTION, $state, false);

		$this->dispatch_step($job_id, 0);
	}

	/**
	 * AJAX endpoint: runs one scheduled step then dispatches the next loopback request if required.
	 *
	 * Note: This endpoint is registered for both logged-in and nopriv contexts.
	 *
	 * @since 1.0.0
	 * @see      Mop_Ai_Indexer_File_Manager
	 * @access public
	 * @return void
	 */
	public function ajax_step_runner(): void {

		$request_method = isset($_SERVER['REQUEST_METHOD']) ? sanitize_text_field(wp_unslash((string)$_SERVER['REQUEST_METHOD'])) : '';
		if ($request_method !== 'POST') {
			wp_die('Invalid request method.');
		}

		// phpcs:disable WordPress.Security.NonceVerification.Missing -- Signed loopback endpoint; request integrity enforced via HMAC signature and job state; supports nopriv.
		$job_id = isset($_POST['job_id']) ? sanitize_text_field(wp_unslash((string)$_POST['job_id'])) : '';
		$step = isset($_POST['step']) ? absint(wp_unslash((string)$_POST['step'])) : 0;
		$ts = isset($_POST['ts']) ? absint(wp_unslash((string)$_POST['ts'])) : 0;
		$sig = isset($_POST['sig']) ? sanitize_text_field(wp_unslash((string)$_POST['sig'])) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Missing -- See note above.

		$this->enforce_invalid_attempt_limit();

		if (! $this->verify_signed_request($job_id, $step, $ts, $sig)) {
			$this->count_invalid_attempt();
			wp_die('Invalid signature.');
		}

		$state = $this->get_job_state();

		if (! is_array($state) || empty($state['job_id']) || empty($state['status'])) {
			wp_die('No active job.');
		}

		if ($state['status'] !== 'running') {
			wp_die('Job not running.');
		}

		if (! hash_equals((string)$state['job_id'], (string)$job_id)) {
			wp_die('Job mismatch.');
		}

		$expected_step = isset($state['expected_step']) ? absint($state['expected_step']) : 0;
		if ($step !== $expected_step) {
			wp_die('Unexpected step.');
		}

		$state['updated_at'] = current_time('timestamp');
		update_option(self::JOB_STATE_OPTION, $state, false);

		$file_manager = Mop_Ai_Indexer_File_Manager::get_instance();

		if ($step === 0) {

			$result = $file_manager->scheduled_start_generation();

			if (! $result['success']) {
				$this->fail_job($job_id);
				wp_die('Step failed.');
			}

			$state['expected_step'] = 1;
			$state['updated_at'] = current_time('timestamp');
			update_option(self::JOB_STATE_OPTION, $state, false);

			$this->dispatch_step($job_id, 1);
			exit;
		}

		$result = $file_manager->scheduled_process_generation();

		if (! $result['success']) {
			$this->fail_job($job_id);
			wp_die('Step failed.');
		}

		$status = (isset($result['data']['status'])) ? (string)$result['data']['status'] : '';

		if ($status === 'running') {

			$state['expected_step'] = $step + 1;
			$state['updated_at'] = current_time('timestamp');
			update_option(self::JOB_STATE_OPTION, $state, false);

			$this->dispatch_step($job_id, $step + 1);
			exit;
		}

		$this->complete_job($job_id);
		exit;
	}

	/**
	 * Dispatches the next loopback step request.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $job_id Job identifier.
	 * @param int $step Step number.
	 * @return void
	 */
	private function dispatch_step(string $job_id, int $step): void {

		$url = admin_url('admin-ajax.php');

		$ts = time();

		$payload = array(
			'action' => self::RUNNER_ACTION,
			'job_id' => $job_id,
			'step' => $step,
			'ts' => $ts,
		);

		$payload['sig'] = $this->sign_payload($job_id, $step, $ts);

		wp_remote_post($url, array(
			'timeout' => 0.01,
			'blocking' => false,
			'sslverify' => apply_filters('https_local_ssl_verify', false), // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core hook name.
			'body' => $payload,
		));
	}

	/**
	 * Returns the currently stored job state.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return array|null
	 */
	private function get_job_state(): ?array {

		$state = get_option(self::JOB_STATE_OPTION);

		return (is_array($state)) ? $state : null;
	}

	/**
	 * Marks job as failed.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $job_id Job id.
	 * @return void
	 */
	private function fail_job(string $job_id): void {

		$state = $this->get_job_state();

		if (! is_array($state) || empty($state['job_id']) || ! hash_equals((string)$state['job_id'], (string)$job_id)) {
			return;
		}

		$state['status'] = 'failed';
		$state['updated_at'] = current_time('timestamp');

		update_option(self::JOB_STATE_OPTION, $state, false);
	}

	/**
	 * Marks job as complete and clears state.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $job_id Job id.
	 * @return void
	 */
	private function complete_job(string $job_id): void {

		$state = $this->get_job_state();

		if (! is_array($state) || empty($state['job_id']) || ! hash_equals((string)$state['job_id'], (string)$job_id)) {
			return;
		}

		$state['status'] = 'complete';
		$state['updated_at'] = current_time('timestamp');

		update_option(self::JOB_STATE_OPTION, $state, false);
	}

	/**
	 * Generates (or loads) the stored HMAC secret.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return string
	 */
	private function get_secret(): string {

		$secret = get_option(self::AUTOMATION_SECRET_OPTION);

		if (is_string($secret) && $secret !== '') {
			return $secret;
		}

		$secret = bin2hex(random_bytes(32));
		update_option(self::AUTOMATION_SECRET_OPTION, $secret, false);

		return $secret;
	}

	/**
	 * Signs a payload for a scheduled loopback call.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $job_id Job id.
	 * @param int $step Step.
	 * @param int $ts Timestamp.
	 * @return string
	 */
	private function sign_payload(string $job_id, int $step, int $ts): string {

		$secret = $this->get_secret();

		$data = $job_id . '|' . (string)$step . '|' . (string)$ts;

		return hash_hmac('sha256', $data, $secret);
	}

	/**
	 * Verifies a signed loopback request.
	 *
	 * @since 1.0.0
	 * @access private
	 * @param string $job_id Job id.
	 * @param int $step Step.
	 * @param int $ts Timestamp.
	 * @param string $sig Signature.
	 * @return bool
	 */
	private function verify_signed_request(string $job_id, int $step, int $ts, string $sig): bool {

		if ($job_id === '' || $sig === '' || $ts <= 0) {
			return false;
		}

		if (abs(time() - $ts) > 120) {
			return false;
		}

		$expected = $this->sign_payload($job_id, $step, $ts);

		return hash_equals($expected, $sig);
	}

	/**
	 * Counts invalid attempts and stores in a short-lived transient.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private function count_invalid_attempt(): void {

		$attempts = get_transient(self::INVALID_ATTEMPTS_TRANSIENT);
		$attempts = is_int($attempts) ? $attempts : 0;

		$attempts++;

		set_transient(self::INVALID_ATTEMPTS_TRANSIENT, $attempts, 10 * MINUTE_IN_SECONDS);
	}

	/**
	 * Blocks requests if invalid attempts exceed the threshold.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 */
	private function enforce_invalid_attempt_limit(): void {

		$attempts = get_transient(self::INVALID_ATTEMPTS_TRANSIENT);
		$attempts = is_int($attempts) ? $attempts : 0;

		if ($attempts >= self::MAX_INVALID_ATTEMPTS) {
			wp_die('Temporarily blocked.');
		}
	}
}
