<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Controls X-Robots-Tag headers for MOP AI Indexer endpoints.
 *
 * By default, these endpoint files are meant for AI agents and should not clutter
 * classic search engine indexes. The preferred mechanism is an HTTP response header.
 *
 * Important: There is no admin UI toggle for this behaviour. It is controlled via
 * a redefinable constant in the main plugin file:
 *
 * - MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER (default true)
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Applies X-Robots-Tag headers to discourage indexing of MOP AI Indexer endpoints.
 *
 * This class is to apply X-Robots-Tag headers to prevent indexing of MOP AI Indexer endpoints.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Robots_Indexing_Guard {

	/**
	 * Conditionally send an X-Robots-Tag header for endpoint responses.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function maybe_send_noindex_header(): void {

		$enabled = true;
		if (defined('MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER')) {
			$enabled = (bool)MOP_AI_INDEXER_ENDPOINT_SEND_NOINDEX_HEADER;
		}

		if (! $enabled) return;

		/**
		 * "noindex, nofollow" is intentionally strict:
		 * - prevents indexing of the endpoint URL
		 * - discourages link-following from the endpoint
		 *
		 * Even if some crawlers interpret "nofollow" differently, the goal here is
		 * to make the intent explicit and avoid accidental SEO side effects.
		 */
		header('X-Robots-Tag: noindex, nofollow', true);
	}
}
