<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Resolves indexability for a given post, based on configured rules.
 *
 * The index generator can optionally respect SEO and robots directives.
 * When enabled, items are excluded if they are:
 * - Explicitly excluded via MOP AI Indexer per-post toggle (mop_ai_indexer_exclude).
 * - Marked as noindex/nofollow by common SEO plugins.
 * - Blocked by robots.txt rules.
 * - Marked noindex site-wide via WordPress "Discourage search engines" setting.
 * - (Optional) Detected as noindex/nofollow via HTTP headers / HTML meta output.
 *
 * This class is designed to be defensive:
 * - It does NOT require SEO plugins to be active; it inspects stored data when present.
 * - It caches expensive lookups such as robots.txt parsing and AIOSEO table discovery.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Resolves per-post indexability directives from robots rules, SEO plugins, and headers.
 *
 * This class is to resolve per-post indexability directives from robots, SEO plugins, and headers.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Indexability_Resolver {

	/**
	 * Cached robots rules.
	 *
	 * @var array|null
	 */
	private static $robots_rules = null;

	/**
	 * Cached AIOSEO table availability and column support.
	 *
	 * @var array|null
	 */
	private static $aioseo_table_info = null;

	/**
	 * Returns an indexability verdict for a given post.
	 *
	 * @since   1.0.0
	 * @param   int   $post_id Post ID.
	 * @param   array $iset    Index settings array (mop_ai_indexer_iset).
	 * @return  array{excluded:bool,reasons:array,sources:array}
	 */
	public static function get_verdict(int $post_id, array $iset): array {

		$post_id = absint($post_id);
		if ($post_id <= 0) {
			return array(
				'excluded' => true,
				'reasons'  => array('invalid_post_id'),
				'sources'  => array('mop-ai-indexer'),
			);
		}

		/**
		 * Always respect the MOP AI Indexer per-post exclusion toggle (manual override),
		 * regardless of whether "Respect SEO Configurations" is enabled.
		 */
		$manual_exclude = get_post_meta($post_id, 'mop_ai_indexer_exclude', true);
		if ((string)$manual_exclude === '1') {
			return array(
				'excluded' => true,
				'reasons'  => array('manual_exclude'),
				'sources'  => array('mop-ai-indexer'),
			);
		}

		/**
		 * Respect SEO configurations only when explicitly enabled via settings.
		 */
		$respect_seo = isset($iset['iset_respect_seo_config']) ? (string)$iset['iset_respect_seo_config'] : '';
		if ($respect_seo !== 'respect-seo') {
			return array(
				'excluded' => false,
				'reasons'  => array(),
				'sources'  => array(),
			);
		}

		$reasons = array();
		$sources = array();

		/**
		 * WordPress global "Discourage search engines from indexing this site" setting.
		 *
		 * When blog_public is disabled, WordPress signals "noindex" via its robots API.
		 * In this mode, it is safer to exclude everything from the generated index file.
		 */
		$blog_public = get_option('blog_public', '1');
		if ((string)$blog_public === '0') {
			return array(
				'excluded' => true,
				'reasons'  => array('wp_blog_public_disabled'),
				'sources'  => array('wordpress-core'),
			);
		}

		$url = get_permalink($post_id);
		$url = is_string($url) ? $url : '';
		if ($url === '') {
			return array(
				'excluded' => true,
				'reasons'  => array('missing_permalink'),
				'sources'  => array('wordpress-core'),
			);
		}

		/**
		 * robots.txt blocking check.
		 *
		 * If a URL path is disallowed for User-agent "*", exclude it.
		 */
		$path = wp_parse_url($url, PHP_URL_PATH);
		$path = is_string($path) ? $path : '/';
		$path = $path !== '' ? $path : '/';

		if (self::is_blocked_by_robots_txt($path)) {
			$reasons[] = 'robots_txt_blocked';
			$sources[] = 'robots.txt';
			return array(
				'excluded' => true,
				'reasons'  => $reasons,
				'sources'  => $sources,
			);
		}

		/**
		 * Collect directives from different sources. In MOP AI Indexer, nofollow is treated
		 * as equivalent to noindex for exclusion purposes, to avoid leaking pages that
		 * were intentionally marked as non-crawlable by the site owner.
		 */
		$has_noindex  = false;
		$has_nofollow = false;

		// WordPress core per-post directive flags (defensive).
		$w_noindex = get_post_meta($post_id, '_wp_robots_noindex', true);
		if (self::is_truthy($w_noindex)) {
			$has_noindex = true;
			$reasons[] = 'wp_robots_noindex';
			$sources[] = 'wordpress-core';
		}

		$w_nofollow = get_post_meta($post_id, '_wp_robots_nofollow', true);
		if (self::is_truthy($w_nofollow)) {
			$has_nofollow = true;
			$reasons[] = 'wp_robots_nofollow';
			$sources[] = 'wordpress-core';
		}

		$w_robots = get_post_meta($post_id, '_wp_robots', true);
		if (is_array($w_robots)) {
			if (self::array_has_token($w_robots, 'noindex')) {
				$has_noindex = true;
				$reasons[] = 'wp_robots_array_noindex';
				$sources[] = 'wordpress-core';
			}
			if (self::array_has_token($w_robots, 'nofollow')) {
				$has_nofollow = true;
				$reasons[] = 'wp_robots_array_nofollow';
				$sources[] = 'wordpress-core';
			}
		}

		// Yoast SEO.
		$yoast_noindex = get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
		if ((string)$yoast_noindex === '1') {
			$has_noindex = true;
			$reasons[] = 'yoast_noindex';
			$sources[] = 'yoast';
		}

		$yoast_nofollow = get_post_meta($post_id, '_yoast_wpseo_meta-robots-nofollow', true);
		if ((string)$yoast_nofollow === '1') {
			$has_nofollow = true;
			$reasons[] = 'yoast_nofollow';
			$sources[] = 'yoast';
		}

		$yoast_adv = get_post_meta($post_id, '_yoast_wpseo_meta-robots-adv', true);
		if (is_string($yoast_adv) && $yoast_adv !== '') {

			$adv_tokens = self::csv_tokens($yoast_adv);
			if (in_array('noindex', $adv_tokens, true)) {
				$has_noindex = true;
				$reasons[] = 'yoast_adv_noindex';
				$sources[] = 'yoast';
			}
			if (in_array('nofollow', $adv_tokens, true)) {
				$has_nofollow = true;
				$reasons[] = 'yoast_adv_nofollow';
				$sources[] = 'yoast';
			}
		}

		// Rank Math.
		$rm_robots = get_post_meta($post_id, 'rank_math_robots', true);
		if (is_array($rm_robots)) {
			if (in_array('noindex', $rm_robots, true)) {
				$has_noindex = true;
				$reasons[] = 'rank_math_noindex';
				$sources[] = 'rank-math';
			}
			if (in_array('nofollow', $rm_robots, true)) {
				$has_nofollow = true;
				$reasons[] = 'rank_math_nofollow';
				$sources[] = 'rank-math';
			}
		} elseif (is_string($rm_robots) && $rm_robots !== '') {
			$rm_tokens = self::csv_tokens($rm_robots);
			if (in_array('noindex', $rm_tokens, true)) {
				$has_noindex = true;
				$reasons[] = 'rank_math_noindex';
				$sources[] = 'rank-math';
			}
			if (in_array('nofollow', $rm_tokens, true)) {
				$has_nofollow = true;
				$reasons[] = 'rank_math_nofollow';
				$sources[] = 'rank-math';
			}
		}

		// SEOPress.
		$sp_index = get_post_meta($post_id, '_seopress_robots_index', true);
		if ((string)$sp_index === 'yes') {
			$has_noindex = true;
			$reasons[] = 'seopress_noindex';
			$sources[] = 'seopress';
		}

		$sp_follow = get_post_meta($post_id, '_seopress_robots_follow', true);
		if ((string)$sp_follow === 'yes') {
			$has_nofollow = true;
			$reasons[] = 'seopress_nofollow';
			$sources[] = 'seopress';
		}

		// AIOSEO.
		$aioseo = self::get_aioseo_directives($post_id);
		if (! empty($aioseo['noindex'])) {
			$has_noindex = true;
			$reasons[] = 'aioseo_noindex';
			$sources[] = 'aioseo';
		}
		if (! empty($aioseo['nofollow'])) {
			$has_nofollow = true;
			$reasons[] = 'aioseo_nofollow';
			$sources[] = 'aioseo';
		}

		/**
		 * WooCommerce visibility exclusions for products.
		 *
		 * Products can be intentionally hidden from search/catalog via the product_visibility taxonomy.
		 * If so, exclude them from the index to avoid surfacing hidden inventory to crawlers.
		 */
		$post_type = get_post_type($post_id);
		if ($post_type === 'product') {

			if (taxonomy_exists('product_visibility')) {

				if (has_term('exclude-from-search', 'product_visibility', $post_id) || has_term('exclude-from-catalog', 'product_visibility', $post_id)) {
					$reasons[] = 'woocommerce_visibility_excluded';
					$sources[] = 'woocommerce';
					return array(
						'excluded' => true,
						'reasons'  => $reasons,
						'sources'  => $sources,
					);
				}
			}
		}

		/**
		 * Optional strict HTTP-level detection.
		 *
		 * Some sites signal noindex/nofollow only via headers or runtime-generated meta tags.
		 * This mode fetches the permalink and inspects:
		 * - X-Robots-Tag response header
		 * - <meta name="robots"> directive in HTML <head>
		 *
		 * This is intentionally OFF by default due to cost, and results are cached.
		 */
		$strict = isset($iset['iset_strict_indexability_check']) ? (string)$iset['iset_strict_indexability_check'] : '';
		if ($strict === 'strict-check') {

			$http = self::get_http_directives($post_id, $url);
			if (! empty($http['noindex'])) {
				$has_noindex = true;
				$reasons[] = 'http_noindex';
				$sources[] = 'http';
			}
			if (! empty($http['nofollow'])) {
				$has_nofollow = true;
				$reasons[] = 'http_nofollow';
				$sources[] = 'http';
			}
		}

		if ($has_noindex || $has_nofollow) {
			return array(
				'excluded' => true,
				'reasons'  => array_values(array_unique($reasons)),
				'sources'  => array_values(array_unique($sources)),
			);
		}

		return array(
			'excluded' => false,
			'reasons'  => array_values(array_unique($reasons)),
			'sources'  => array_values(array_unique($sources)),
		);
	}

	/**
	 * Returns whether a robots.txt rule blocks a given URL path for User-agent "*".
	 *
	 * This is a best-effort robots.txt evaluation meant for content exclusion only.
	 * It supports:
	 * - Disallow
	 * - Allow (overrides matching Disallow when more specific)
	 * - Wildcards (*) and end-of-line anchors ($) in patterns
	 *
	 * @since   1.0.0
	 * @param   string $path URL path (e.g., "/my-page/").
	 * @return  bool
	 */
	private static function is_blocked_by_robots_txt(string $path): bool {

		$rules = self::get_robots_rules();

		$allow = isset($rules['allow']) && is_array($rules['allow']) ? $rules['allow'] : array();
		$disallow = isset($rules['disallow']) && is_array($rules['disallow']) ? $rules['disallow'] : array();

		$best_len = -1;
		$best_allow = false;

		foreach ($allow as $pattern) {

			if (! self::robots_pattern_matches($pattern, $path)) continue;

			$len = self::robots_pattern_len($pattern);

			// Prefer the most specific rule; if tied, Allow wins over Disallow.
			if ($len > $best_len || ($len === $best_len && $best_allow === false)) {
				$best_len = $len;
				$best_allow = true;
			}
		}

		foreach ($disallow as $pattern) {

			if (! self::robots_pattern_matches($pattern, $path)) continue;

			$len = self::robots_pattern_len($pattern);

			// Disallow only wins when it is strictly more specific than the best rule so far.
			if ($len > $best_len) {
				$best_len = $len;
				$best_allow = false;
			}
		}

		// If the most specific match is a Disallow rule, the path is blocked.
		return ($best_len >= 0 && $best_allow === false);
	}

	/**
	 * Loads and caches robots.txt rules for User-agent "*".
	 *
	 * @since   1.0.0
	 * @return  array{allow:array,disallow:array}
	 */
	private static function get_robots_rules(): array {

		if (is_array(self::$robots_rules)) return self::$robots_rules;

		$transient_key = 'mop_ai_indexer_robots_rules_cache_v1';
		$cached = get_transient($transient_key);
		if (is_array($cached) && isset($cached['allow']) && isset($cached['disallow'])) {
			self::$robots_rules = $cached;
			return self::$robots_rules;
		}

		$robots_body = '';

		// 1) Try physical robots.txt first.
		$response = wp_remote_get(home_url('/robots.txt'), array(
			'timeout'     => 10,
			'redirection' => 3,
		));

		if (! is_wp_error($response)) {
			$code = wp_remote_retrieve_response_code($response);
			$body = wp_remote_retrieve_body($response);
			if ($code >= 200 && $code < 300 && is_string($body) && trim($body) !== '') {
				$robots_body = $body;
			}
		}

		// 2) Fallback to WordPress virtual robots endpoint.
		if ($robots_body === '') {
			$response = wp_remote_get(home_url('/?robots=1'), array(
				'timeout'     => 10,
				'redirection' => 3,
			));

			if (! is_wp_error($response)) {
				$code = wp_remote_retrieve_response_code($response);
				$body = wp_remote_retrieve_body($response);
				if ($code >= 200 && $code < 300 && is_string($body) && trim($body) !== '') {
					$robots_body = $body;
				}
			}
		}

		$parsed = self::parse_robots_txt($robots_body);

		// Cache for 12 hours to minimize repeated HTTP fetches during generation.
		set_transient($transient_key, $parsed, 12 * HOUR_IN_SECONDS);

		self::$robots_rules = $parsed;
		return self::$robots_rules;
	}

	/**
	 * Parses robots.txt text and extracts Allow/Disallow rules for User-agent "*".
	 *
	 * @since   1.0.0
	 * @param   string $robots_body Raw robots.txt content.
	 * @return  array{allow:array,disallow:array}
	 */
	private static function parse_robots_txt(string $robots_body): array {

		$rules = array(
			'allow'   => array(),
			'disallow' => array(),
		);

		if ($robots_body === '') return $rules;

		$lines = preg_split('/\r\n|\r|\n/', $robots_body);
		if (! is_array($lines)) return $rules;

		$current_agents = array();
		$seen_directive = false;

		foreach ($lines as $line) {

			$line = is_string($line) ? $line : '';
			$line = preg_replace('/\s*#.*$/', '', $line);
			$line = trim((string)$line);

			if ($line === '') continue;

			$lower = strtolower($line);

			if (strpos($lower, 'user-agent:') === 0) {

				$ua = trim(substr($line, strlen('user-agent:')));
				$ua = strtolower($ua);

				// When a new user-agent line appears after directives, treat it as a new group.
				if ($seen_directive) {
					$current_agents = array();
					$seen_directive = false;
				}

				if ($ua !== '') {
					$current_agents[] = $ua;
				}

				continue;
			}

			if (empty($current_agents)) continue;

			$is_star_group = in_array('*', $current_agents, true);
			if (! $is_star_group) continue;

			if (strpos($lower, 'disallow:') === 0) {

				$path = trim(substr($line, strlen('disallow:')));
				if ($path !== '') $rules['disallow'][] = $path;
				$seen_directive = true;
				continue;
			}

			if (strpos($lower, 'allow:') === 0) {

				$path = trim(substr($line, strlen('allow:')));
				if ($path !== '') $rules['allow'][] = $path;
				$seen_directive = true;
				continue;
			}
		}

		return $rules;
	}

	/**
	 * Returns AIOSEO directives for a post.
	 *
	 * @since   1.0.0
	 * @param   int $post_id Post ID.
	 * @return  array{noindex:bool,nofollow:bool}
	 */
	private static function get_aioseo_directives(int $post_id): array {

		$noindex = false;
		$nofollow = false;

		global $wpdb;

		$info = self::get_aioseo_table_info();
		if (! empty($info['exists'])) {

			$table = isset($info['table']) ? (string)$info['table'] : '';
			$has_noindex = ! empty($info['has_robots_noindex']);
			$has_nofollow = ! empty($info['has_robots_nofollow']);

			if ($table !== '' && ($has_noindex || $has_nofollow)) {

				$cols = array();
				if ($has_noindex) $cols[] = 'robots_noindex';
				if ($has_nofollow) $cols[] = 'robots_nofollow';

				$select_cols = implode(', ', $cols);
				if ($select_cols !== '') {

					// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- AIOSEO integration: read-only lookup of robots directives. The column list is built from an allowlist of known column names, the table name is prefix-derived, and the post_id is prepared; caching is handled at higher level during generation.
					$row = $wpdb->get_row(
						$wpdb->prepare(
							"SELECT {$select_cols} FROM {$table} WHERE post_id = %d",
							$post_id
						)
					);
					// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- End AIOSEO directives lookup.

					if ($row) {
						if ($has_noindex && isset($row->robots_noindex) && absint($row->robots_noindex) === 1) {
							$noindex = true;
						}
						if ($has_nofollow && isset($row->robots_nofollow) && absint($row->robots_nofollow) === 1) {
							$nofollow = true;
						}
					}
				}
			}
		}

		/**
		 * Safe fallback for older AIOSEO installs (post meta).
		 *
		 * AIOSEO storage formats changed over time. We only treat a value as enabled when
		 * it is clearly truthy and not an explicit "0".
		 */
		$a_noindex = get_post_meta($post_id, '_aioseo_noindex', true);
		if (self::is_truthy($a_noindex)) $noindex = true;

		$a_nofollow = get_post_meta($post_id, '_aioseo_nofollow', true);
		if (self::is_truthy($a_nofollow)) $nofollow = true;

		return array(
			'noindex'  => $noindex,
			'nofollow' => $nofollow,
		);
	}

	/**
	 * Discovers the AIOSEO posts table and whether relevant columns exist.
	 *
	 * @since   1.0.0
	 * @return  array
	 */
	private static function get_aioseo_table_info(): array {

		if (is_array(self::$aioseo_table_info)) return self::$aioseo_table_info;

		global $wpdb;

		$table = $wpdb->prefix . 'aioseo_posts';

		$exists = ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) === $table); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter -- AIOSEO table existence check is read-only and cached via static property for the duration of the request.

		$info = array(
			'exists' => $exists,
			'table'  => $table,
			'has_robots_noindex'  => false,
			'has_robots_nofollow' => false,
		);

		if (! $exists) {
			self::$aioseo_table_info = $info;
			return self::$aioseo_table_info;
		}

		/**
		 * Check whether columns exist before attempting to read them.
		 *
		 * This keeps the resolver defensive across different AIOSEO versions and migrations.
		 */
		// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- AIOSEO schema detection is read-only, uses a prefix-derived table name, and is cached in a static property for the duration of the request.
		$columns = $wpdb->get_results("SHOW COLUMNS FROM {$table}");
		// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, PluginCheck.Security.DirectDB.UnescapedDBParameter, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- End AIOSEO schema detection.
		$colnames = array();

		if (is_array($columns)) {
			foreach ($columns as $col) {
				if (isset($col->Field)) {
					$colnames[] = (string)$col->Field;
				}
			}
		}

		$info['has_robots_noindex']  = in_array('robots_noindex', $colnames, true);
		$info['has_robots_nofollow'] = in_array('robots_nofollow', $colnames, true);

		self::$aioseo_table_info = $info;
		return self::$aioseo_table_info;
	}

	/**
	 * Performs an HTTP-level directive check and caches the result.
	 *
	 * @since   1.0.0
	 * @param   int    $post_id Post ID.
	 * @param   string $url     Permalink URL.
	 * @return  array{noindex:bool,nofollow:bool}
	 */
	private static function get_http_directives(int $post_id, string $url): array {

		$noindex = false;
		$nofollow = false;

		$post = get_post($post_id);
		$modified = '';
		if ($post && isset($post->post_modified_gmt)) {
			$modified = (string)$post->post_modified_gmt;
		}

		$key = 'mop_ai_indexer_http_robots_' . md5($url);
		$cached = get_transient($key);

		if (is_array($cached) && isset($cached['modified']) && (string)$cached['modified'] === $modified) {
			return array(
				'noindex'  => ! empty($cached['noindex']),
				'nofollow' => ! empty($cached['nofollow']),
			);
		}

		/**
		 * Step 1: HEAD request for X-Robots-Tag.
		 */
		$response = wp_remote_head($url, array(
			'timeout'     => 10,
			'redirection' => 5,
		));

		$xrobots = '';
		if (! is_wp_error($response)) {
			$xrobots = wp_remote_retrieve_header($response, 'x-robots-tag');
			if (is_array($xrobots)) $xrobots = implode(',', $xrobots);
			$xrobots = is_string($xrobots) ? $xrobots : '';
		}

		$tokens = self::csv_tokens($xrobots);
		if (in_array('noindex', $tokens, true)) $noindex = true;
		if (in_array('nofollow', $tokens, true)) $nofollow = true;

		/**
		 * Step 2: If header is not present (or empty), fall back to a GET request
		 * to detect <meta name="robots"> directives in the rendered HTML head.
		 */
		if ($xrobots === '') {

			$response = wp_remote_get($url, array(
				'timeout'     => 10,
				'redirection' => 5,
			));

			if (! is_wp_error($response)) {

				$code = wp_remote_retrieve_response_code($response);
				$body = wp_remote_retrieve_body($response);

				if ($code >= 200 && $code < 300 && is_string($body) && $body !== '') {

					$meta = self::extract_robots_meta_content($body);
					$meta_tokens = self::csv_tokens($meta);

					if (in_array('noindex', $meta_tokens, true)) $noindex = true;
					if (in_array('nofollow', $meta_tokens, true)) $nofollow = true;
				}
			}
		}

		set_transient($key, array(
			'modified' => $modified,
			'noindex'  => $noindex ? 1 : 0,
			'nofollow' => $nofollow ? 1 : 0,
		), 12 * HOUR_IN_SECONDS);

		return array(
			'noindex'  => $noindex,
			'nofollow' => $nofollow,
		);
	}

	/**
	 * Extracts a robots meta tag content value from HTML.
	 *
	 * This is a minimal parser intended for detection, not for full HTML parsing.
	 *
	 * @since   1.0.0
	 * @param   string $html HTML response body.
	 * @return  string Robots content string, or empty.
	 */
	private static function extract_robots_meta_content(string $html): string {

		$head = $html;

		// Work on the head only when possible.
		if (preg_match('/<head\b[^>]*>(.*?)<\/head>/is', $html, $m)) {
			$head = isset($m[1]) ? (string)$m[1] : $html;
		}

		// Match: <meta name="robots" content="...">
		if (preg_match('/<meta\s+[^>]*name=[\'"]robots[\'"][^>]*>/i', $head, $tag)) {

			$meta_tag = isset($tag[0]) ? (string)$tag[0] : '';
			if ($meta_tag !== '' && preg_match('/content=[\'"]([^\'"]+)[\'"]/i', $meta_tag, $cm)) {
				return isset($cm[1]) ? (string)$cm[1] : '';
			}
		}

		return '';
	}

	/**
	 * Converts a robots.txt pattern into a match test.
	 *
	 * @since   1.0.0
	 * @param   string $pattern Raw Allow/Disallow pattern.
	 * @param   string $path    URL path.
	 * @return  bool
	 */
	private static function robots_pattern_matches(string $pattern, string $path): bool {

		$pattern = trim($pattern);
		if ($pattern === '') return false;

		$anchored = (substr($pattern, -1) === '$');
		if ($anchored) {
			$pattern = substr($pattern, 0, -1);
		}

		$regex = preg_quote($pattern, '#');
		$regex = str_replace('\*', '.*', $regex);

		$regex = '#^' . $regex;
		if ($anchored) {
			$regex .= '$';
		}

		$regex .= '#';

		return (bool)preg_match($regex, $path);
	}

	/**
	 * Returns an approximate pattern length used for specificity comparison.
	 *
	 * @since   1.0.0
	 * @param   string $pattern Pattern.
	 * @return  int
	 */
	private static function robots_pattern_len(string $pattern): int {

		$pattern = trim($pattern);
		$pattern = rtrim($pattern, '$');
		$pattern = str_replace('*', '', $pattern);

		return strlen($pattern);
	}

	/**
	 * Returns true when a value can be treated as boolean true.
	 *
	 * @since   1.0.0
	 * @param   mixed $value Value.
	 * @return  bool
	 */
	private static function is_truthy($value): bool {

		if (is_bool($value)) return $value;
		if (is_int($value)) return ($value === 1);
		if (is_float($value)) return ((int)$value === 1);

		$val = is_string($value) ? strtolower(trim($value)) : '';

		if ($val === '1' || $val === 'yes' || $val === 'true' || $val === 'on') return true;
		return false;
	}

	/**
	 * Splits comma-separated robots directives into normalized tokens.
	 *
	 * @since   1.0.0
	 * @param   string $csv CSV string.
	 * @return  array
	 */
	private static function csv_tokens(string $csv): array {

		$csv = strtolower(trim($csv));
		if ($csv === '') return array();

		$parts = preg_split('/\s*,\s*/', $csv);
		$out = array();

		if (! is_array($parts)) return $out;

		foreach ($parts as $p) {
			$p = trim((string)$p);
			if ($p === '') continue;

			// Some headers include directives separated by semicolons as well.
			$sub = preg_split('/\s*;\s*/', $p);
			if (is_array($sub)) {
				foreach ($sub as $s) {
					$s = trim((string)$s);
					if ($s !== '') {

						// X-Robots-Tag can include user-agent prefixes, e.g. "googlebot: noindex".
						if (strpos($s, ':') !== false) {
							$pair = preg_split('/\s*:\s*/', $s, 2);
							if (is_array($pair) && isset($pair[1]) && trim((string)$pair[1]) !== '') {
								$s = trim((string)$pair[1]);
							}
						}

						$out[] = $s;
					}
				}
			} else {
				$out[] = $p;
			}
		}

		$out = array_map('strval', $out);
		$out = array_values(array_unique($out));

		return $out;
	}

	/**
	 * Returns true if an array contains a token, either as a key or a value.
	 *
	 * @since   1.0.0
	 * @param   array  $arr   Array.
	 * @param   string $token Token.
	 * @return  bool
	 */
	private static function array_has_token(array $arr, string $token): bool {

		$token = strtolower(trim($token));
		if ($token === '') return false;

		foreach ($arr as $k => $v) {

			if (is_string($k) && strtolower($k) === $token) return true;
			if (is_string($v) && strtolower($v) === $token) return true;
			if (is_bool($v) && $v === true && is_string($k) && strtolower($k) === $token) return true;
		}

		return false;
	}
}
