<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Provides endpoint discovery hints for AI agents.
 *
 * Notes:
 * - <link> hints in HTML <head> are optional and not part of a formal spec.
 * - robots.txt has no standardized directive for llms.txt / mop-ai-indexer; we only add a comment line.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer_Endpoint_Router};

/**
 * Outputs optional discovery hints for MOP AI Indexer endpoints.
 *
 * This class is to output optional discovery hints for MOP AI Indexer endpoints.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Index_Discovery {

	/**
	 * Output a discovery hint in the HTML <head>.
	 *
	 * This can help user-agents that crawl HTML and look for known link relations.
	 * The hint is intentionally conservative and optional.
	 *
	 * @since  1.0.0
	 * @see    Mop_Ai_Indexer_Endpoint_Router
	 * @return void
	 */
	public function output_head_hint(): void {

		if (defined('MOP_AI_INDEXER_ENDPOINT_HEAD_HINT_ENABLED') && ! MOP_AI_INDEXER_ENDPOINT_HEAD_HINT_ENABLED) return;

		$file_name = Mop_Ai_Indexer_Endpoint_Router::get_canonical_file_name();
		$url = Mop_Ai_Indexer_Endpoint_Router::get_endpoint_url($file_name);

		/**
		 * Use rel="llms" only when the canonical filename is llms.txt, otherwise use rel="mop-ai-index".
		 * This keeps semantics simple while still providing a breadcrumb.
		 */
		$rel = ($file_name === 'llms.txt') ? 'llms' : 'mop-ai-index';

		echo "\n" . '<link rel="' . esc_attr($rel) . '" href="' . esc_url($url) . '" />' . "\n";
	}

	/**
	 * Add a conservative discovery hint in robots.txt as a comment line.
	 *
	 * This runs only for WordPress' virtual robots.txt output (/?robots=1).
	 * If the site has a physical robots.txt file, WordPress will not filter it.
	 *
	 * @since  1.0.0
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @param  string $output Existing robots.txt output.
	 * @param  bool   $public Whether the site is considered public.
	 * @return string
	 */
	public function add_robots_txt_hint(string $output, bool $public): string {

		if (defined('MOP_AI_INDEXER_ENDPOINT_ROBOTS_TXT_HINT_ENABLED') && ! MOP_AI_INDEXER_ENDPOINT_ROBOTS_TXT_HINT_ENABLED) {
			return (string)$output;
		}

		$file_name = Mop_Ai_Indexer_Endpoint_Router::get_canonical_file_name();
		$url = Mop_Ai_Indexer_Endpoint_Router::get_endpoint_url($file_name);

		$output = (string)$output;
		if ($output !== '' && substr($output, -1) !== "\n") {
			$output .= "\n";
		}

		$output .= '# MOP AI Index: ' . $url . "\n";

		return $output;
	}
}
