<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Manages the index file (generation/deletion/status)
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Manages the index file (generation/deletion/status).
 *
 * This class is to manage the index file (generation/deletion/status).
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_File_Manager {

	/**
	 * Contains the class instance.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @var        Mop_Ai_Indexer_File_Manager $instance Contains the class instance for singleton purposes.
	 */
	private static ?self $instance = null;

	/**
	 * Getter method to get the class instance.
	 *
	 * Creates an instance of the class and returns it respecting the singleton pattern.
	 *
	 * @since      1.0.0
	 * @access     public
	 * @see        Mop_Ai_Indexer_File_Manager
	 * @return     Mop_Ai_Indexer_File_Manager Mop_Ai_Indexer_File_Manager object.
	 */
	public static function get_instance(): self {

		if (self::$instance === null) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 *
	 * Declared as private to prevent direct instantiation (singleton).
	 *
	 * @since      1.0.0
	 * @access     private
	 * @return     void
	 */
	private function __construct() {}

	/**
	 * Clone.
	 *
	 * Declared as private to prevent cloning of the singleton instance.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @return     void
	 */
	private function __clone() {}


	/**
	 * Option slug for storing the latest generation/deletion log set.
	 *
	 * NOTE: Hyphens are intentionally used as per plugin specification.
	 *
	 * @since  1.0.0
	 */
	private const GENERATION_LOGS_OPTION = 'mop-ai-indexer-index-generation-logs';

	/**
	 * Option slug for storing latest index file generation metadata (size/duration).
	 *
	 * This option stores only the latest completed generation meta.
	 *
	 * @since  1.0.0
	 */
	private const INDEX_FILE_META_OPTION = 'mop-ai-indexer-index-file-meta';

	/**
	 * Public getter for generation/deletion logs display data.
	 *
	 * Returns a payload suitable for the admin UI:
	 * - logs_title: H3 title string
	 * - process: current/last process key
	 * - logs: array of formatted log lines
	 *
	 * @since  1.0.0
	 * @return array
	 */
	public function get_generation_logs_display_data(): array {

		return $this->get_generation_logs_payload();
	}

	/**
	 * Reset the stored generation/deletion logs for a new process run.
	 *
	 * The option only stores the latest set of logs across:
	 * - Manual generation (AJAX)
	 * - Manual deletion (AJAX)
	 * - Scheduled generation (CRON) [future]
	 *
	 * @since  1.0.0
	 * @param  string $process Process key.
	 * @return void
	 */
	private function reset_generation_logs(string $process): void {

		$process = $this->sanitize_log_process($process);

		$data = array(
			'process' => $process,
			'updated_at' => current_time('timestamp'),
			'logs' => array(),
		);

		update_option(self::GENERATION_LOGS_OPTION, $data, false);
	}

	/**
	 * Append a single log line to the stored latest log set.
	 *
	 * Duplicate consecutive (type+message) entries are ignored to prevent noisy UI output.
	 *
	 * @since  1.0.0
	 * @param  string $type Log type (STATUS/ERROR/etc).
	 * @param  string $message Log message.
	 * @return void
	 */
	private function append_generation_log(string $type, string $message): void {

		$type = strtoupper(trim((string)$type));
		$message = trim((string)$message);

		if ($type === '') $type = 'STATUS';
		if ($message === '') return;

		$data = get_option(self::GENERATION_LOGS_OPTION, array());
		$data = is_array($data) ? $data : array();

		$process = isset($data['process']) ? (string)$data['process'] : 'manual_generation';
		$process = $this->sanitize_log_process($process);

		$logs = (isset($data['logs']) && is_array($data['logs'])) ? $data['logs'] : array();

		$last = (! empty($logs)) ? $logs[count($logs) - 1] : null;
		if (is_array($last) && isset($last['type']) && isset($last['message'])) {
			if ((string)$last['type'] === $type && (string)$last['message'] === $message) {
				return;
			}
		}

		$logs[] = array(
			'timestamp' => current_time('timestamp'),
			'type' => $type,
			'message' => $message,
		);

		/**
		 * Hard limit to prevent unbounded option growth.
		 */
		if (count($logs) > 250) {
			$logs = array_slice($logs, -250);
		}

		$data = array(
			'process' => $process,
			'updated_at' => current_time('timestamp'),
			'logs' => $logs,
		);

		update_option(self::GENERATION_LOGS_OPTION, $data, false);
	}

	/**
	 * Build a UI payload from the stored logs option.
	 *
	 * @since  1.0.0
	 * @return array
	 */
	private function get_generation_logs_payload(): array {

		$data = get_option(self::GENERATION_LOGS_OPTION, array());
		$data = is_array($data) ? $data : array();

		$process = isset($data['process']) ? (string)$data['process'] : 'manual_generation';
		$process = $this->sanitize_log_process($process);

		$logs = (isset($data['logs']) && is_array($data['logs'])) ? $data['logs'] : array();

		$lines = array();
		foreach ($logs as $log) {
			if (! is_array($log)) continue;
			$lines[] = $this->format_generation_log_line($log);
		}

		return array(
			'logs_title' => $this->get_logs_title_for_process($process),
			'process' => $process,
			'logs' => $lines,
		);
	}

	/**
	 * Sanitizes process keys for logs storage.
	 *
	 * @since  1.0.0
	 * @param  string $process Process key.
	 * @return string
	 */
	private function sanitize_log_process(string $process): string {

		$process = trim((string)$process);

		$allowed = array(
			'manual_generation',
			'manual_deletion',
			'scheduled_generation',
		);

		if (! in_array($process, $allowed, true)) {
			$process = 'manual_generation';
		}

		return $process;
	}

	/**
	 * Maps the process key to the H3 title required by the UI.
	 *
	 * @since  1.0.0
	 * @param  string $process Process key.
	 * @return string
	 */
	private function get_logs_title_for_process(string $process): string {

		$process = $this->sanitize_log_process($process);

		if ($process === 'manual_deletion') {
			return esc_html__('Index File Generation: Manual Deletion (AJAX)', 'mop-ai-indexer');
		}

		if ($process === 'scheduled_generation') {
			return esc_html__('Index File Generation: Scheduled Generation (WP-CRON)', 'mop-ai-indexer');
		}

		return esc_html__('Index File Generation: Manual Generation (AJAX)', 'mop-ai-indexer');
	}

	/**
	 * Formats a single stored log entry into a display line:
	 * "January 12, 2026 10:34 PM - [STATUS] Message"
	 *
	 * @since  1.0.0
	 * @param  array $log Log entry array.
	 * @return string
	 */
	private function format_generation_log_line(array $log): string {

		$ts = isset($log['timestamp']) ? absint($log['timestamp']) : current_time('timestamp');
		$type = isset($log['type']) ? strtoupper((string)$log['type']) : 'STATUS';
		$message = isset($log['message']) ? (string)$log['message'] : '';

		$date_str = date_i18n('F j, Y g:i A', $ts);

		return $date_str . ' - [' . $type . '] ' . $message;
	}

	/**
	 * Start an index file generation job via AJAX.
	 *
	 * Initializes and stores a transient job structure and writes the file header to a temp file.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     public
	 * @return     void
	 */
	private function start_generation_core(string $process): array {

		$process = $this->sanitize_log_process($process);

		$job_key = self::get_job_key();

		/**
		 * Prevent starting a new job while one is still running.
		 */
		$existing_job = get_transient($job_key);
		if (is_array($existing_job) && isset($existing_job['status']) && $existing_job['status'] === 'running') {

			$logs_payload = $this->get_generation_logs_payload();

			return array('success' => true, 'data' => array_merge(array(
				'status' => 'running',
				'process' => $process,
				'percent' => $this->calc_percent($existing_job),
				'message' => esc_html__('File generation is already running.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $logs_payload));
		}

		$started_log_message = ($process === 'scheduled_generation')
		? esc_html__('Scheduled index file generation started.', 'mop-ai-indexer')
		: esc_html__('Manual index file generation started.', 'mop-ai-indexer');

		/**
		 * Start a fresh manual generation (AJAX) log set.
		 */
		$this->reset_generation_logs($process);
		$this->append_generation_log('STATUS', $started_log_message);

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = (is_array($mop_ai_indexer_iset)) ? $mop_ai_indexer_iset : array();

		$file_name = isset($mop_ai_indexer_iset['iset_file_name_format']) ? sanitize_text_field((string)$mop_ai_indexer_iset['iset_file_name_format']) : 'llms.txt';
		$file_name = $this->sanitize_file_name($file_name);

		/**
		 * Retrieve index configuration.
		 *
		 * On a fresh install, the mop_ai_indexer_config option may not exist yet. In that case,
		 * we must use the same defaults displayed in the Index Manager page so the user
		 * can generate the index file without saving configuration first.
		 */
		$defaults = Mop_Ai_Indexer_Defaults::get_default_index_config(Mop_Ai_Indexer_Defaults::get_public_post_types_in_order());

		$mop_ai_indexer_config = get_option('mop_ai_indexer_config', $defaults);
		$mop_ai_indexer_config = (is_array($mop_ai_indexer_config)) ? $mop_ai_indexer_config : array();

		/**
		 * Ensure missing post type configs are populated from defaults.
		 */
		$mop_ai_indexer_config = Mop_Ai_Indexer_Defaults::merge_config_with_defaults($mop_ai_indexer_config, $defaults);

		$post_type_configs = (isset($mop_ai_indexer_config['post_type_config']) && is_array($mop_ai_indexer_config['post_type_config'])) ? $mop_ai_indexer_config['post_type_config'] : array();

		/**
		 * Prepare ordered post type list.
		 */
		$post_types_to_index = $this->get_post_types_in_order($post_type_configs);

		if (empty($post_types_to_index)) {
			$this->append_generation_log('ERROR', esc_html__('No post types are selected to include in the index file.', 'mop-ai-indexer'));
			return array('success' => false, 'data' => array_merge(array(
				'message' => esc_html__('No post types are selected to include in the index file.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		$upload_dir = wp_upload_dir();
		$upload_basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$upload_basedir = wp_normalize_path($upload_basedir);

		if ($upload_basedir === '' || ! is_dir($upload_basedir)) {
			$this->append_generation_log('ERROR', esc_html__('Unable to access the uploads directory.', 'mop-ai-indexer'));
			return array('success' => false, 'data' => array_merge(array(
				'message' => esc_html__('Unable to access the uploads directory.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		$plugin_uploads_path = wp_normalize_path(trailingslashit($upload_basedir) . 'mop-ai-indexer/');
		if (! is_dir($plugin_uploads_path)) {
			wp_mkdir_p($plugin_uploads_path);
		}
		if (! is_dir($plugin_uploads_path)) {
			$this->append_generation_log('ERROR', esc_html__('Unable to access the plugin uploads directory.', 'mop-ai-indexer'));
			return array('success' => false, 'data' => array_merge(array(
				'message' => esc_html__('Unable to access the plugin uploads directory.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		$base_name = preg_replace('/\.txt$/', '', $file_name);
		$base_name = $base_name ? $base_name : 'mop-ai-indexer';

		/**
		 * Temp file is written first, then renamed into the final file.
		 *
		 * Example:
		 * - llms-temp.txt -> llms.txt (or llms-full-temp.txt -> llms-full.txt)
		 */
		$temp_file = trailingslashit($plugin_uploads_path) . $base_name . '-temp.txt';
		$temp_file = wp_normalize_path($temp_file);

		$final_file = trailingslashit($plugin_uploads_path) . $file_name;
		$final_file = wp_normalize_path($final_file);

		/**
		 * Initialize job data.
		 */
		$started_at = current_time('timestamp');
		$started_micro = microtime(true);
		$job = array(
			'status' => 'running',
			'process' => $process,
			'phase' => 'overview',
			'started_at' => $started_at,
			'started_micro' => $started_micro,
			'file_name' => $file_name,
			'temp_file' => $temp_file,
			'final_file' => $final_file,
			'post_types' => $post_types_to_index,
			'pt_cursor' => 0,
			'offset' => 0,
			'processed' => 0,
			'total' => $this->calc_total_units($post_types_to_index),
			'chunk_size' => 25,
			'overview_headers_written' => array(),
			'detailed_headers_written' => array(),
		);

		/**
		 * Create/overwrite temp file and write header.
		 */
		$header_written = $this->write_initial_header($job);
		if (! $header_written) {
			$this->append_generation_log('ERROR', esc_html__('Unable to write the temporary file in uploads directory.', 'mop-ai-indexer'));
			return array('success' => false, 'data' => array_merge(array(
				'message' => esc_html__('Unable to write the temporary file in uploads directory.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		$this->reset_index_file_meta($process, $file_name, $started_at, $started_micro);

		set_transient($job_key, $job, HOUR_IN_SECONDS);

		return array('success' => true, 'data' => array_merge(array(
			'status' => 'running',
			'percent' => $this->calc_percent($job),
			'message' => $started_log_message,
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Process generation core.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @see      Mop_Ai_Indexer_Cache_Manager
	 * @return   array
	 */
	private function process_generation_core(): array {

		$job_key = self::get_job_key();
		$job = get_transient($job_key);

		if (! is_array($job) || empty($job)) {
			return array('success' => false, 'data' => array_merge(array(
				'message' => esc_html__('No active generation job found.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		if (! isset($job['status']) || $job['status'] !== 'running') {
			return array('success' => true, 'data' => array_merge(array(
				'status' => 'completed',
				'percent' => 100,
				'message' => esc_html__('No active generation job found.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}

		$phase = isset($job['phase']) ? (string)$job['phase'] : 'overview';

		if ($phase === 'overview') {

			$job = $this->process_overview_phase($job);

		} elseif ($phase === 'detailed') {

			$job = $this->process_detailed_phase($job);

		}else{

			$job['status'] = 'error';
			$job['error_message'] = esc_html__('Unknown generation phase.', 'mop-ai-indexer');
		}

		/**
		 * Handle completion / errors.
		 */
		if ($job['status'] === 'error') {
			set_transient($job_key, $job, HOUR_IN_SECONDS);
			$err_msg = isset($job['error_message']) ? (string)$job['error_message'] : esc_html__('File generation failed.', 'mop-ai-indexer');
			$this->append_generation_log('ERROR', $err_msg);
			return array('success' => false, 'data' => array_merge(array(
				'message' => $err_msg,
			), $this->get_generation_logs_payload()));
		}

		if ($job['status'] === 'completed') {

			$is_finalized = $this->finalize_file($job);

			if (! $is_finalized) {
				$job['status'] = 'error';
				$job['error_message'] = $this->get_finalize_error_message($job);
				set_transient($job_key, $job, HOUR_IN_SECONDS);
				$this->append_generation_log('ERROR', (string)$job['error_message']);
				return array('success' => false, 'data' => array_merge(array(
					'message' => (string)$job['error_message'],
				), $this->get_generation_logs_payload()));
			}

			$this->store_last_generated($job['file_name'], current_time('timestamp'));

			$this->finalize_index_file_meta($job);

			delete_transient($job_key);

			$complete_log_message = (isset($job['process']) && $job['process'] === 'scheduled_generation')
			? esc_html__('Scheduled index file generation is complete.', 'mop-ai-indexer')
			: esc_html__('Manual index file generation is complete.', 'mop-ai-indexer');
			$this->append_generation_log('STATUS', $complete_log_message);
			$cache_context = (isset($job['process']) && $job['process'] === 'scheduled_generation') ? 'scheduled_generation_cron' : 'manual_generation_ajax';
			$purge_result = Mop_Ai_Indexer_Cache_Manager::maybe_purge_after_success($cache_context, true);
			if (is_array($purge_result)) {
				$purged_items = (isset($purge_result['purged']) && is_array($purge_result['purged'])) ? $purge_result['purged'] : array();
				if (! empty($purged_items)) {
					/* translators: %s: comma-separated list of cache layers purged. */
					$this->append_generation_log('STATUS', sprintf(esc_html__('Caches purged: %s.', 'mop-ai-indexer'), implode(', ', $purged_items)));
				} else {
					$this->append_generation_log('STATUS', esc_html__('Cache purge attempted, but no supported caches were detected.', 'mop-ai-indexer'));
				}
				$purge_errors = (isset($purge_result['errors']) && is_array($purge_result['errors'])) ? $purge_result['errors'] : array();
				foreach ($purge_errors as $purge_error) {
					$this->append_generation_log('ERROR', (string)$purge_error);
				}
			}
			return array('success' => true, 'data' => array_merge(array(
				'status' => 'completed',
				'percent' => 100,
				'message' => $complete_log_message,
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}
		set_transient($job_key, $job, HOUR_IN_SECONDS);

		$progress_message = $this->get_progress_message($job);
		$this->append_generation_log('STATUS', $progress_message);

		return array('success' => true, 'data' => array_merge(array(
			'status' => 'running',
			'percent' => $this->calc_percent($job),
			'message' => $progress_message,
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Ajax start generation.
	 *
	 * @since    1.0.0
	 * @return   void
	 */
	public function ajax_start_generation(): void {

		$this->verify_ajax_request();

		$result = $this->start_generation_core('manual_generation');

		if (! $result['success']) {
			wp_send_json_error($result['data']);
		}

		wp_send_json_success($result['data']);
	}

	/**
	 * Process the next chunk of the index file generation job via AJAX.
	 *
	 * Advances the transient job state and writes additional sections into the temp file, returning progress for the UI.
	 *
	 * @since      1.0.0
	 * @access     public
	 * @return     void
	 */
	public function ajax_process_generation(): void {

		$this->verify_ajax_request();

		$result = $this->process_generation_core();

		if (! $result['success']) {
			wp_send_json_error($result['data']);
		}

		wp_send_json_success($result['data']);
	}

	/**
	 * Starts an automated (scheduled) generation run without browser nonce/capability checks.
	 *
	 * This is intended to be called by the scheduled runner which validates a signed request.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Result structure: ['success' => bool, 'data' => array]
	 */
	public function scheduled_start_generation(): array {

		return $this->start_generation_core('scheduled_generation');
	}

	/**
	 * Processes the next generation step for an automated (scheduled) run.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Result structure: ['success' => bool, 'data' => array]
	 */
	public function scheduled_process_generation(): array {

		return $this->process_generation_core();
	}

	/**
	 * Ajax get generation status.
	 *
	 * @since    1.0.0
	 * @return   void
	 */
	public function ajax_get_generation_status(): void {

		$this->verify_ajax_request();

		$job_key = self::get_job_key();
		$job = get_transient($job_key);

		if (! is_array($job) || empty($job)) {
			wp_send_json_success(array_merge(array(
				'status' => 'idle',
				'percent' => 0,
				'message' => esc_html__('No active generation job.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}

		wp_send_json_success(array_merge(array(
			'status' => isset($job['status']) ? (string)$job['status'] : 'idle',
			'percent' => $this->calc_percent($job),
			'message' => $this->get_progress_message($job),
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Start an index file deletion job via AJAX.
	 *
	 * Creates a transient job structure so the admin UI can show progressive logs.
	 *
	 * @since  1.0.0
	 * @see    Mop_Ai_Indexer_Defaults
	 * @return void
	 */
	public function ajax_start_deletion(): void {

		$this->verify_ajax_request();

		/**
		 * Do not allow deletion while a generation job is running.
		 */
		$gen_job = get_transient(self::get_job_key());
		if (is_array($gen_job) && isset($gen_job['status']) && $gen_job['status'] === 'running') {

			$logs_payload = $this->get_generation_logs_payload();

			wp_send_json_success(array_merge(array(
				'status' => 'running',
				'percent' => $this->calc_percent($gen_job),
				'message' => esc_html__('File generation is already running.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $logs_payload));
		}

		$job_key = self::get_delete_job_key();

		/**
		 * Prevent starting a new delete job while one is still running.
		 */
		$existing_job = get_transient($job_key);
		if (is_array($existing_job) && isset($existing_job['status']) && $existing_job['status'] === 'running') {

			$logs_payload = $this->get_generation_logs_payload();

			wp_send_json_success(array_merge(array(
				'status' => 'running',
				'percent' => $this->calc_deletion_percent($existing_job),
				'message' => esc_html__('File deletion is already running.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $logs_payload));
		}

		/**
		 * Start a fresh manual deletion (AJAX) log set.
		 */
		$this->reset_generation_logs('manual_deletion');
		$this->append_generation_log('STATUS', esc_html__('Manual index file deletion started.', 'mop-ai-indexer'));

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = (is_array($mop_ai_indexer_iset)) ? $mop_ai_indexer_iset : array();

		$last_file = get_option('mop_ai_indexer_last_generated_file', '');
		$last_file = is_string($last_file) ? sanitize_text_field($last_file) : '';
		$last_file = $this->sanitize_file_name($last_file);

		$current_file = isset($mop_ai_indexer_iset['iset_file_name_format']) ? sanitize_text_field((string)$mop_ai_indexer_iset['iset_file_name_format']) : 'llms.txt';
		$current_file = $this->sanitize_file_name($current_file);

		$files_to_delete = array();
		if ($last_file !== '') $files_to_delete[] = $last_file;
		if ($current_file !== '' && $current_file !== $last_file) $files_to_delete[] = $current_file;

		/**
		 * Uploads base directory.
		 */
		$upload_dir = wp_upload_dir();
		$upload_basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$upload_basedir = wp_normalize_path($upload_basedir);

		if ($upload_basedir === '' || ! is_dir($upload_basedir)) {
			$this->append_generation_log('ERROR', esc_html__('Unable to access the uploads directory.', 'mop-ai-indexer'));
			wp_send_json_error(array_merge(array(
				'message' => esc_html__('Unable to access the uploads directory.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		$plugin_uploads_path = wp_normalize_path(trailingslashit($upload_basedir) . 'mop-ai-indexer/');

		$job = array(
			'status' => 'running',
			'started_at' => current_time('timestamp'),
			'cursor' => 0,
			'files' => $files_to_delete,
			'plugin_uploads_path' => $plugin_uploads_path,
			'deleted_any' => false,
		);
		set_transient($job_key, $job, HOUR_IN_SECONDS);

		wp_send_json_success(array_merge(array(
			'status' => 'running',
			'percent' => 0,
			'message' => esc_html__('Manual index file deletion started.', 'mop-ai-indexer'),
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Process the next deletion step of the index file deletion job via AJAX.
	 *
	 * @since  1.0.0
	 * @see    Mop_Ai_Indexer_Cache_Manager
	 * @return void
	 */
	public function ajax_process_deletion(): void {

		$this->verify_ajax_request();

		$job_key = self::get_delete_job_key();
		$job = get_transient($job_key);

		if (! is_array($job) || empty($job)) {
			wp_send_json_error(array_merge(array(
				'message' => esc_html__('No active deletion job found.', 'mop-ai-indexer'),
			), $this->get_generation_logs_payload()));
		}

		if (! isset($job['status']) || $job['status'] !== 'running') {
			wp_send_json_success(array_merge(array(
				'status' => 'completed',
				'percent' => 100,
				'message' => esc_html__('No active deletion job found.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}

		$files = (isset($job['files']) && is_array($job['files'])) ? $job['files'] : array();
		$cursor = isset($job['cursor']) ? absint($job['cursor']) : 0;
		$plugin_uploads_path = isset($job['plugin_uploads_path']) ? (string)$job['plugin_uploads_path'] : '';
		$plugin_uploads_path = wp_normalize_path($plugin_uploads_path);

		/**
		 * Write "Deleting..." once, at the start of processing.
		 */
		if ($cursor === 0) {
			$this->append_generation_log('STATUS', esc_html__('Deleting...', 'mop-ai-indexer'));
		}

		if ($plugin_uploads_path === '' || ! is_dir($plugin_uploads_path)) {
			$err_msg = esc_html__('Unable to access the plugin uploads directory.', 'mop-ai-indexer');
			$this->append_generation_log('ERROR', $err_msg);
			$job['status'] = 'error';
			$job['error_message'] = $err_msg;
			set_transient($job_key, $job, HOUR_IN_SECONDS);
			wp_send_json_error(array_merge(array(
				'message' => $err_msg,
			), $this->get_generation_logs_payload()));
		}

		if (empty($files)) {

			delete_option('mop_ai_indexer_last_generated');
			delete_option('mop_ai_indexer_last_generated_file');
			$this->delete_index_file_meta();

			delete_transient($job_key);

			$this->append_generation_log('STATUS', esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'));

			$purge_result = Mop_Ai_Indexer_Cache_Manager::maybe_purge_after_success('manual_deletion_ajax', true);
			if (is_array($purge_result)) {
				$purged_items = (isset($purge_result['purged']) && is_array($purge_result['purged'])) ? $purge_result['purged'] : array();
				if (! empty($purged_items)) {
					/* translators: %s: comma-separated list of cache layers purged. */
					$this->append_generation_log('STATUS', sprintf(esc_html__('Caches purged: %s.', 'mop-ai-indexer'), implode(', ', $purged_items)));
				} else {
					$this->append_generation_log('STATUS', esc_html__('Cache purge attempted, but no supported caches were detected.', 'mop-ai-indexer'));
				}
				$purge_errors = (isset($purge_result['errors']) && is_array($purge_result['errors'])) ? $purge_result['errors'] : array();
				foreach ($purge_errors as $purge_error) {
					$this->append_generation_log('ERROR', (string)$purge_error);
				}
			}
			wp_send_json_success(array_merge(array(
				'status' => 'completed',
				'percent' => 100,
				'message' => esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}

		/**
		 * Delete one file per request to allow progressive UI updates.
		 */
		if (isset($files[$cursor])) {

			$file_name = $this->sanitize_file_name((string)$files[$cursor]);

			if ($file_name !== '') {

				$file_path = wp_normalize_path(trailingslashit($plugin_uploads_path) . $file_name);

				if (is_file($file_path) && is_writable($file_path)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.
					$deleted = @unlink($file_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting plugin-owned allowlisted uploads files is required for generation/deletion.
					if ($deleted) {
						$job['deleted_any'] = true;
					}
				}
			}

			$cursor++;
			$job['cursor'] = $cursor;
		}

		$total = count($files);
		if ($total <= 0) $total = 1;

		if ($cursor >= count($files)) {

			delete_option('mop_ai_indexer_last_generated');
			delete_option('mop_ai_indexer_last_generated_file');
			$this->delete_index_file_meta();

			delete_transient($job_key);

			$this->append_generation_log('STATUS', esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'));

			$purge_result = Mop_Ai_Indexer_Cache_Manager::maybe_purge_after_success('manual_deletion_ajax', true);
			if (is_array($purge_result)) {
				$purged_items = (isset($purge_result['purged']) && is_array($purge_result['purged'])) ? $purge_result['purged'] : array();
				if (! empty($purged_items)) {
					/* translators: %s: comma-separated list of cache layers purged. */
					$this->append_generation_log('STATUS', sprintf(esc_html__('Caches purged: %s.', 'mop-ai-indexer'), implode(', ', $purged_items)));
				} else {
					$this->append_generation_log('STATUS', esc_html__('Cache purge attempted, but no supported caches were detected.', 'mop-ai-indexer'));
				}
				$purge_errors = (isset($purge_result['errors']) && is_array($purge_result['errors'])) ? $purge_result['errors'] : array();
				foreach ($purge_errors as $purge_error) {
					$this->append_generation_log('ERROR', (string)$purge_error);
				}
			}
			wp_send_json_success(array_merge(array(
				'status' => 'completed',
				'percent' => 100,
				'message' => esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'),
				'file_status_html' => $this->get_file_status_html(),
			), $this->get_generation_logs_payload()));
		}
		set_transient($job_key, $job, HOUR_IN_SECONDS);

		wp_send_json_success(array_merge(array(
			'status' => 'running',
			'percent' => $this->calc_deletion_percent($job),
			'message' => esc_html__('Deleting...', 'mop-ai-indexer'),
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Calculate a simple deletion progress percent.
	 *
	 * @since  1.0.0
	 * @param  array $job Deletion job array.
	 * @return int
	 */
	private function calc_deletion_percent(array $job): int {

		$files = (isset($job['files']) && is_array($job['files'])) ? $job['files'] : array();
		$total = count($files);
		if ($total <= 0) $total = 1;

		$cursor = isset($job['cursor']) ? absint($job['cursor']) : 0;

		$percent = (int)floor(($cursor / $total) * 100);
		if ($percent > 100) $percent = 100;
		if ($percent < 0) $percent = 0;

		return $percent;
	}

	/**
	 * Delete the generated index file via AJAX.
	 *
	 * Removes the generated file from the uploads directory and clears last-generated metadata.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Cache_Manager
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     public
	 * @return     void
	 */
	public function ajax_delete_index_file(): void {

		$this->verify_ajax_request();

		/**
		 * Start a fresh manual deletion (AJAX) log set.
		 */
		$this->reset_generation_logs('manual_deletion');
		$this->append_generation_log('STATUS', esc_html__('Manual index file deletion started.', 'mop-ai-indexer'));
		$this->append_generation_log('STATUS', esc_html__('Deleting...', 'mop-ai-indexer'));

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = (is_array($mop_ai_indexer_iset)) ? $mop_ai_indexer_iset : array();

		/**
		 * Determine files to delete:
		 * - The last generated file name stored in mop_ai_indexer_last_generated_file (if any)
		 * - The currently configured file name (in case the user changed the setting after generation)
		 */
		$last_file = get_option('mop_ai_indexer_last_generated_file', '');
		$last_file = is_string($last_file) ? sanitize_text_field($last_file) : '';
		$last_file = $this->sanitize_file_name($last_file);

		$current_file = isset($mop_ai_indexer_iset['iset_file_name_format']) ? sanitize_text_field((string)$mop_ai_indexer_iset['iset_file_name_format']) : 'llms.txt';
		$current_file = $this->sanitize_file_name($current_file);

		$files_to_delete = array();
		if ($last_file !== '') $files_to_delete[] = $last_file;
		if ($current_file !== '' && $current_file !== $last_file) $files_to_delete[] = $current_file;

		$deleted_any = false;

		foreach ($files_to_delete as $file_name) {

			$file_name = $this->sanitize_file_name($file_name);
			if ($file_name === '') continue;

			$upload_dir = wp_upload_dir();
			$upload_basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
			$upload_basedir = wp_normalize_path($upload_basedir);

			if ($upload_basedir === '') continue;

			$plugin_uploads_path = wp_normalize_path(trailingslashit($upload_basedir) . 'mop-ai-indexer/');

			$upload_file = wp_normalize_path(trailingslashit($plugin_uploads_path) . $file_name);

			if (is_file($upload_file) && is_writable($upload_file)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.
				$deleted = @unlink($upload_file); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting plugin-owned allowlisted uploads files is required for generation/deletion.
				if ($deleted) {
					$deleted_any = true;
				}
			}
		}

		/**
		 * Clear last-generated metadata when a delete is requested.
		 */
		delete_option('mop_ai_indexer_last_generated');
		delete_option('mop_ai_indexer_last_generated_file');
		$this->delete_index_file_meta();

		$this->append_generation_log('STATUS', esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'));

		$purge_result = Mop_Ai_Indexer_Cache_Manager::maybe_purge_after_success('manual_deletion_ajax', true);
		if (is_array($purge_result)) {
			$purged_items = (isset($purge_result['purged']) && is_array($purge_result['purged'])) ? $purge_result['purged'] : array();
			if (! empty($purged_items)) {
				/* translators: %s: comma-separated list of cache layers purged. */
				$this->append_generation_log('STATUS', sprintf(esc_html__('Caches purged: %s.', 'mop-ai-indexer'), implode(', ', $purged_items)));
			} else {
				$this->append_generation_log('STATUS', esc_html__('Cache purge attempted, but no supported caches were detected.', 'mop-ai-indexer'));
			}
			$purge_errors = (isset($purge_result['errors']) && is_array($purge_result['errors'])) ? $purge_result['errors'] : array();
			foreach ($purge_errors as $purge_error) {
				$this->append_generation_log('ERROR', (string)$purge_error);
			}
		}
		wp_send_json_success(array_merge(array(
			'deleted' => $deleted_any ? '1' : '0',
			'message' => esc_html__('Manual index file deletion is complete.', 'mop-ai-indexer'),
			'file_status_html' => $this->get_file_status_html(),
		), $this->get_generation_logs_payload()));
	}

	/**
	 * Generate overview sections for the configured post types.
	 *
	 * Writes headings and bullet lines into the temp file, chunked by offset to avoid timeouts.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     array Updated job data.
	 */
	private function process_overview_phase(array $job): array {

		$post_types = isset($job['post_types']) && is_array($job['post_types']) ? $job['post_types'] : array();
		$pt_cursor = isset($job['pt_cursor']) ? absint($job['pt_cursor']) : 0;

		if (! isset($post_types[$pt_cursor])) {

			if ($this->should_write_detailed_section($post_types)) {
				$job = $this->start_detailed_phase($job);
				return $job;
			}

			$job['status'] = 'completed';
			return $job;
		}

		$pt = $post_types[$pt_cursor];
		$slug = isset($pt['slug']) ? (string)$pt['slug'] : '';
		$label = isset($pt['label']) ? (string)$pt['label'] : $slug;

		/**
		 * Resolve the max content length for this post type.
		 *
		 * A value of 0 means post content should not be written in the detailed section and
		 * post content should not be used as a description fallback.
		 */
		$max_words = isset($pt['max_content_length']) ? absint($pt['max_content_length']) : 1000;

		if ($slug === '') {
			$job['status'] = 'error';
			$job['error_message'] = esc_html__('Invalid post type configuration.', 'mop-ai-indexer');
			return $job;
		}

		if (! isset($job['overview_headers_written'][$slug])) {
			$written = $this->append_to_file($job['temp_file'], '## ' . $label . "\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}
			$job['overview_headers_written'][$slug] = 1;
		}

		$offset = isset($pt['overview_offset']) ? absint($pt['overview_offset']) : 0;
		$limit = isset($pt['num_of_latest_posts']) ? absint($pt['num_of_latest_posts']) : 0;
		$chunk_size = isset($job['chunk_size']) ? absint($job['chunk_size']) : 25;

		if ($limit <= 0) $limit = 100;

		$remaining = $limit - $offset;
		$take = ($remaining > $chunk_size) ? $chunk_size : $remaining;

		if ($take <= 0) {

			$written = $this->append_to_file($job['temp_file'], "\n---\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}

			$job['pt_cursor'] = $pt_cursor + 1;
			return $job;
		}

		$q = new \WP_Query(array(
			'post_type' => $slug,
			'post_status' => 'publish',
			'posts_per_page' => $take,
			'offset' => $offset,
			'orderby' => 'date',
			'order' => 'DESC',
			'no_found_rows' => true,
			'fields' => 'ids',
		));

		$post_ids = (isset($q->posts) && is_array($q->posts)) ? $q->posts : array();

		/**
		 * IMPORTANT:
		 * If there are fewer published posts than the configured limit, WP_Query will
		 * return an empty list once the offset exceeds available results.
		 *
		 * If we do not detect this case, the generator can get stuck in an endless loop
		 * (offset never changes, remaining never becomes 0), which freezes progress.
		 */
		if (empty($post_ids)) {

			$written = $this->append_to_file($job['temp_file'], "\n---\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}

			$pt['overview_offset'] = $limit;
			$post_types[$pt_cursor] = $pt;
			$job['post_types'] = $post_types;
			$job['pt_cursor'] = $pt_cursor + 1;
			return $job;
		}

		foreach ($post_ids as $post_id) {

			$post_id = absint($post_id);

			/**
			 * Respect SEO config: skip noindex items.
			 */
			if ($this->should_skip_post($post_id)) {
				$job['processed'] = isset($job['processed']) ? absint($job['processed']) + 1 : 1;
				$offset++;
				continue;
			}

			$title = get_the_title($post_id);
			$title = $title !== '' ? $title : esc_html__('(No title)', 'mop-ai-indexer');
			$url = get_permalink($post_id);

			$line = '- [' . $this->sanitize_md_text($title) . '](' . esc_url_raw($url) . ')';

			$include_desc = isset($pt['include_exc_meta_desc']) && $pt['include_exc_meta_desc'] === '1';

			if ($include_desc) {
				$desc = $this->get_post_description($post_id, ($max_words > 0));
				if ($desc !== '') {
					$line .= ': ' . $this->sanitize_md_text($desc);
				}
			}

			$line .= "\n";

			$written = $this->append_to_file($job['temp_file'], $line);
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}

			$job['processed'] = isset($job['processed']) ? absint($job['processed']) + 1 : 1;
			$offset++;
		}

		$pt['overview_offset'] = $offset;
		$post_types[$pt_cursor] = $pt;
		$job['post_types'] = $post_types;

		return $job;
	}

	/**
	 * Transition the job into the detailed content phase.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     array Updated job data.
	 */
	private function start_detailed_phase(array $job): array {

		$written = $this->append_to_file($job['temp_file'], "\n#\n# Detailed Content\n\n");
		if (! $written) {
			$job['status'] = 'error';
			$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
			return $job;
		}

		$job['phase'] = 'detailed';
		$job['pt_cursor'] = 0;

		$post_types = isset($job['post_types']) && is_array($job['post_types']) ? $job['post_types'] : array();
		foreach ($post_types as $i => $pt) {
			$post_types[$i]['detailed_offset'] = 0;
		}
		$job['post_types'] = $post_types;

		return $job;
	}

	/**
	 * Generate detailed content sections for the configured post types.
	 *
	 * Writes per-post detail blocks (meta/taxonomies/content) into the temp file, chunked by offset.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     array Updated job data.
	 */
	private function process_detailed_phase(array $job): array {

		$post_types = isset($job['post_types']) && is_array($job['post_types']) ? $job['post_types'] : array();
		$pt_cursor = isset($job['pt_cursor']) ? absint($job['pt_cursor']) : 0;

		if (! isset($post_types[$pt_cursor])) {
			$job['status'] = 'completed';
			return $job;
		}

		$pt = $post_types[$pt_cursor];

		$slug = isset($pt['slug']) ? (string)$pt['slug'] : '';
		$label = isset($pt['label']) ? (string)$pt['label'] : $slug;

		$include_meta   = isset($pt['include_meta_info']) && $pt['include_meta_info'] === '1';
		$include_tax    = isset($pt['include_taxonomies']) && $pt['include_taxonomies'] === '1';
		$include_custom = isset($pt['include_custom_fields']) && $pt['include_custom_fields'] === '1';
		$max_words      = isset($pt['max_content_length']) ? absint($pt['max_content_length']) : 1000;

		/**
		 * Skip post types which do not require detailed content.
		 */
		if (! ($include_meta || $include_tax || $include_custom || $max_words > 0)) {
			$job['pt_cursor'] = $pt_cursor + 1;
			return $job;
		}

		if (! isset($job['detailed_headers_written'][$slug])) {
			$written = $this->append_to_file($job['temp_file'], '## ' . $label . "\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}
			$job['detailed_headers_written'][$slug] = 1;
		}

		$offset = isset($pt['detailed_offset']) ? absint($pt['detailed_offset']) : 0;
		$limit = isset($pt['num_of_latest_posts']) ? absint($pt['num_of_latest_posts']) : 0;
		$chunk_size = isset($job['chunk_size']) ? absint($job['chunk_size']) : 25;

		$max_words = isset($pt['max_content_length']) ? absint($pt['max_content_length']) : 1000;

		if ($limit <= 0) $limit = 100;

		$remaining = $limit - $offset;
		$take = ($remaining > $chunk_size) ? $chunk_size : $remaining;

		if ($take <= 0) {

			$written = $this->append_to_file($job['temp_file'], "\n---\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}

			$job['pt_cursor'] = $pt_cursor + 1;
			return $job;
		}

		$q = new \WP_Query(array(
			'post_type' => $slug,
			'post_status' => 'publish',
			'posts_per_page' => $take,
			'offset' => $offset,
			'orderby' => 'date',
			'order' => 'DESC',
			'no_found_rows' => true,
			'fields' => 'ids',
		));

		$post_ids = (isset($q->posts) && is_array($q->posts)) ? $q->posts : array();

		/**
		 * Prevent endless loops when offset exceeds available results.
		 */
		if (empty($post_ids)) {

			$written = $this->append_to_file($job['temp_file'], "\n---\n\n");
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}

			$pt['detailed_offset'] = $limit;
			$post_types[$pt_cursor] = $pt;
			$job['post_types'] = $post_types;
			$job['pt_cursor'] = $pt_cursor + 1;
			return $job;
		}

		foreach ($post_ids as $post_id) {

			$post_id = absint($post_id);

			if ($this->should_skip_post($post_id)) {
				$job['processed'] = isset($job['processed']) ? absint($job['processed']) + 1 : 1;
				$offset++;
				continue;
			}

			/**
			 * Write a per-item title line for the detailed section.
			 *
			 * This keeps each detailed block readable and uniquely identifiable, even when
			 * descriptions and/or content are disabled.
			 */
			$title = get_the_title($post_id);
			$title = $title !== '' ? $title : esc_html__('(No title)', 'mop-ai-indexer');
			$url = get_permalink($post_id);

			$written = $this->append_to_file(
				$job['temp_file'],
				'### [' . $this->sanitize_md_text((string)$title) . '](' . esc_url_raw((string)$url) . ")\n\n"
			);
			if (! $written) {
				$job['status'] = 'error';
				$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
				return $job;
			}


			$include_desc = isset($pt['include_exc_meta_desc']) && $pt['include_exc_meta_desc'] === '1';
			if ($include_desc) {
				$desc = $this->get_post_description($post_id, ($max_words > 0));
				if ($desc !== '') {
					$this->append_to_file($job['temp_file'], '> ' . $this->sanitize_md_text($desc) . "\n\n");
				}
			}

			if ($include_meta) {

				$published = get_post_time('Y-m-d', false, $post_id, true);
				$modified = get_post_modified_time('Y-m-d', false, $post_id, true);

				$this->append_to_file($job['temp_file'], '- Published: ' . $published . "\n");
				$this->append_to_file($job['temp_file'], '- Modified: ' . $modified . "\n");
			}

			if ($include_tax) {

				$tax_lines = $this->get_taxonomy_lines($post_id, $slug);
				if (! empty($tax_lines)) {
					foreach ($tax_lines as $tax_line) {
						$this->append_to_file($job['temp_file'], $tax_line . "\n");
					}
				}
			}



			/**
			 * Optionally write custom fields (post meta) when enabled for this post type.
			 *
			 * Custom fields are written after taxonomy output and before the post content,
			 * so crawlers see taxonomy context first and then structured metadata, followed
			 * by the main content body.
			 *
			 * IMPORTANT:
			 * - Only human-readable scalar values are written.
			 * - Private/internal keys (prefixed with "_") are skipped to avoid dumping builder
			 *   payloads such as Elementor's "_elementor_data".
			 */
			if ($include_custom) {

				$cf_lines = $this->get_custom_field_lines($post_id);
				if (! empty($cf_lines)) {
					foreach ($cf_lines as $cf_line) {

						$written = $this->append_to_file($job['temp_file'], $cf_line . "\n");
						if (! $written) {
							$job['status'] = 'error';
							$job['error_message'] = esc_html__('Unable to write to the temporary file.', 'mop-ai-indexer');
							return $job;
						}
					}
				}
			}

			/**
			 * Write post content only when the configured max content length is greater than 0.
			 *
			 * A value of 0 means content is intentionally suppressed for this post type.
			 */
			if ($max_words > 0) {
				$content = $this->get_post_content_text($post_id);
				if ($content !== '') {
					$content = $this->trim_words($content, $max_words);
					if ($content !== '') {
						$this->append_to_file($job['temp_file'], "\n" . $content . "\n\n");
					}
				}
			}

			$this->append_to_file($job['temp_file'], "---\n\n");

			$job['processed'] = isset($job['processed']) ? absint($job['processed']) + 1 : 1;
			$offset++;
		}

		$pt['detailed_offset'] = $offset;
		$post_types[$pt_cursor] = $pt;
		$job['post_types'] = $post_types;

		return $job;
	}

	/**
	 * Finalize the generation job by replacing the final file in the uploads directory.
	 *
	 * This plugin now serves the endpoint via a WordPress rewrite route, so a physical
	 * file in the site root is no longer required.
	 *
	 * Workflow:
	 * 1) Generate to a temp file in uploads (e.g., llms-temp.txt)
	 * 2) Delete the old final file (if exists)
	 * 3) Rename the temp file to the final name (e.g., llms.txt)
	 *
	 * This avoids partial writes and keeps the file location consistent.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     bool True on success, false on failure.
	 */
	private function finalize_file(array $job): bool {

		$temp_file = isset($job['temp_file']) ? wp_normalize_path((string)$job['temp_file']) : '';
		$final_file = isset($job['final_file']) ? wp_normalize_path((string)$job['final_file']) : '';
		$file_name = isset($job['file_name']) ? (string)$job['file_name'] : '';

		if ($temp_file === '' || $final_file === '') return false;
		if (! is_file($temp_file)) return false;

		$final_dir = wp_normalize_path(dirname($final_file));

		if (! is_dir($final_dir) || ! is_writable($final_dir)) return false; // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.

		/**
		 * Remove any existing final file first to avoid rename failures on some hosts.
		 */
		if (is_file($final_file) && is_writable($final_file)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.
			@unlink($final_file); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting plugin-owned allowlisted uploads files is required for generation/deletion.
		}

		/**
		 * Prefer rename for an atomic swap on the same filesystem.
		 */
		if (@rename($temp_file, $final_file)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.rename_rename -- Atomic swap of temp file into final location within allowlisted uploads directory.
			return true;
		}

		/**
		 * Fallback: copy + delete temp.
		 */
		if (@copy($temp_file, $final_file)) {
			@unlink($temp_file); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Deleting plugin-owned allowlisted uploads files is required for generation/deletion.
			return true;
		}

		return false;
	}


	/**
	 * Build a user-friendly error message for finalize failures.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     string Error message.
	 */
	private function get_finalize_error_message(array $job): string {

		$final_file = isset($job['final_file']) ? (string)$job['final_file'] : '';
		$final_file = wp_normalize_path($final_file);

		if ($final_file === '') {
			return esc_html__('Unable to finalize the generated file in the uploads directory.', 'mop-ai-indexer');
		}

		$final_dir = wp_normalize_path(dirname($final_file));

		if (is_file($final_file) && ! is_writable($final_file)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.
			return esc_html__('The index file exists in the uploads directory, but it is not writable. Please check file permissions.', 'mop-ai-indexer');
		}

		if (is_dir($final_dir) && ! is_writable($final_dir)) { // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable -- Permission check is required before mutating allowlisted uploads files.
			return esc_html__('The uploads directory is not writable. Please check your server permissions to allow creating files in uploads.', 'mop-ai-indexer');
		}

		return esc_html__('Unable to finalize the generated file in the uploads directory. Please check server permissions and try again.', 'mop-ai-indexer');
	}

	/**
	 * Write the initial header block into the temp file.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     bool True on success, false on failure.
	 */
	private function write_initial_header(array $job): bool {

		$temp_file = isset($job['temp_file']) ? (string)$job['temp_file'] : '';
		if ($temp_file === '') return false;

		/**
		 * Create/overwrite temp file.
		 */
		$created = @file_put_contents($temp_file, '');
		if ($created === false) return false;

		$content = "\xEF\xBB\xBF";

		$ai_sitemap_page = get_page_by_path('ai-sitemap');
		if ($ai_sitemap_page instanceof \WP_Post) {
			$content .= '# Learn more:' . get_permalink($ai_sitemap_page) . "\n\n";
		}

		$site_title = get_bloginfo('name');
		$content .= '# ' . $this->sanitize_md_text((string)$site_title) . "\n\n";

		$site_desc = get_bloginfo('description');
		$site_desc = is_string($site_desc) ? trim($site_desc) : '';
		if ($site_desc !== '') {
			$content .= '> ' . $this->sanitize_md_text($site_desc) . "\n\n";
		}

		$file_name = isset($job['file_name']) ? $this->sanitize_file_name((string)$job['file_name']) : 'llms.txt';
		$endpoint_url = home_url('/' . $file_name);

		$dt = date_i18n('d/m/Y g:i A', current_time('timestamp'));

		/**
		 * AI-focused preamble (clarifies purpose for LLM crawlers).
		 * Keep this compact and token-efficient.
		 */
		$content .= 'Index file dedicated for AI agents and LLM crawlers. It lists selected public URLs and short summaries for fast consumption.' . "\n";
		$content .= 'Generated by MOP AI Indexer on ' . $dt . '.' . "\n";
		$content .= 'Endpoint: ' . $endpoint_url . "\n";
		$content .= 'Policy: Respect robots.txt and noindex/nofollow directives.' . "\n\n";

		$content .= "---\n\n";

		return $this->append_to_file($temp_file, $content);
	}

	/**
	 * Verify the AJAX request nonce and capabilities.
	 *
	 * Ensures the caller has permission and provides a valid nonce; terminates the request on failure.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @return     void
	 */
	private function verify_ajax_request(): void {

		if (! current_user_can('manage_options')) {
			wp_send_json_error(array(
				'message' => esc_html__('You are not allowed to do this action.', 'mop-ai-indexer'),
			));
		}

		check_ajax_referer('mop_ai_indexer_file_manager_admin_nonce');
	}

	/**
	 * Sanitize and normalize the index filename requested by the user.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $file_name Requested file name/format.
	 * @return     string Sanitized file name.
	 */
	private function sanitize_file_name(string $file_name): string {

		$file_name = sanitize_text_field((string)$file_name);
		$file_name = wp_basename($file_name);

		$allowed = array('llms.txt', 'llms-full.txt');

		if (! in_array($file_name, $allowed, true)) {
			$file_name = 'llms.txt';
		}

		return $file_name;
	}

	/**
	 * Append content to a file using a safe write strategy.
	 *
	 * Writes using append mode to keep memory usage low while generating large files.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $file_path Absolute file path.
	 * @param      string $content Content to append.
	 * @return     bool True on success, false on failure.
	 */
	private function append_to_file(string $file_path, string $content): bool {

		$file_path = wp_normalize_path((string)$file_path);
		$content = (string)$content;

		if ($file_path === '') return false;

		$fp = @fopen($file_path, 'ab'); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Append log content to plugin-owned allowlisted file.
		if ($fp) {
			$written = @fwrite($fp, $content); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fwrite -- Write log content to plugin-owned allowlisted file.
			@fclose($fp); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Close file handle for plugin-owned allowlisted file.
			return ($written !== false);
		}

		/**
		 * Fallback: use file_put_contents append.
		 */
		$written = @file_put_contents($file_path, $content, FILE_APPEND);
		return ($written !== false);
	}

	/**
	 * Return configured post types, filtered and ordered by priority.
	 *
	 * Filters out non-included post types and sorts the remaining by order_priority_in_index ascending.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $post_type_configs Post type configuration array from options.
	 * @return     array Ordered post type data for job processing.
	 */
	private function get_post_types_in_order(array $post_type_configs): array {

		if (! is_array($post_type_configs)) return array();

		$selected = array();
		foreach ($post_type_configs as $slug => $cfg) {

			if (! is_array($cfg)) continue;

			$include = isset($cfg['include_in_mop_ai_indexer']) ? (string)$cfg['include_in_mop_ai_indexer'] : '0';
			if ($include !== '1') continue;

			$prio = isset($cfg['order_priority_in_index']) ? absint($cfg['order_priority_in_index']) : 9999;
			if ($prio <= 0) $prio = 9999;

			$label = $slug;
			$obj = get_post_type_object($slug);
			if ($obj && isset($obj->labels) && isset($obj->labels->singular_name)) {
				$label = (string)$obj->labels->singular_name;
			}

			$selected[] = array(
				'slug' => $slug,
				'label' => $label,
				'order_priority_in_index' => (string)$prio,
				'num_of_latest_posts' => isset($cfg['num_of_latest_posts']) ? (string)$cfg['num_of_latest_posts'] : '100',
				'max_content_length' => isset($cfg['max_content_length']) ? (string)$cfg['max_content_length'] : '200',
				'include_meta_info' => isset($cfg['include_meta_info']) ? (string)$cfg['include_meta_info'] : '0',
				'include_exc_meta_desc' => isset($cfg['include_exc_meta_desc']) ? (string)$cfg['include_exc_meta_desc'] : '0',
				'include_taxonomies' => isset($cfg['include_taxonomies']) ? (string)$cfg['include_taxonomies'] : '0',
				'include_custom_fields'   => isset($cfg['include_custom_fields']) ? (string)$cfg['include_custom_fields'] : '0',
				'overview_offset' => 0,
				'detailed_offset' => 0,
			);
		}

		usort($selected, function($a, $b) {
				$ap = isset($a['order_priority_in_index']) ? absint($a['order_priority_in_index']) : 9999;
				$bp = isset($b['order_priority_in_index']) ? absint($b['order_priority_in_index']) : 9999;
				if ($ap === $bp) return 0;
				return ($ap < $bp) ? -1 : 1;
			});

		return $selected;
	}

	/**
	 * Estimate total work units for progress tracking.
	 *
	 * Counts publishable items up to each post type limit, and optionally doubles work when detailed sections are enabled.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $post_types Ordered post type data.
	 * @return     int Estimated total units.
	 */
	private function calc_total_units(array $post_types): int {

		$total = 0;
		if (! is_array($post_types)) return $total;

		foreach ($post_types as $pt) {
			$slug = isset($pt['slug']) ? (string)$pt['slug'] : '';
			$limit = isset($pt['num_of_latest_posts']) ? absint($pt['num_of_latest_posts']) : 0;
			if ($limit <= 0) $limit = 100;

			$published_count = 0;
			if ($slug !== '') {
				$c = wp_count_posts($slug);
				if ($c && isset($c->publish)) {
					$published_count = absint($c->publish);
				}
			}

			$units = ($published_count > 0) ? min($limit, $published_count) : 0;
			if ($units <= 0) $units = 1;
			$total += $units;

			$include_meta   = isset($pt['include_meta_info']) && $pt['include_meta_info'] === '1';
			$include_tax    = isset($pt['include_taxonomies']) && $pt['include_taxonomies'] === '1';
			$include_custom = isset($pt['include_custom_fields']) && $pt['include_custom_fields'] === '1';
			$max_words      = isset($pt['max_content_length']) ? absint($pt['max_content_length']) : 1000;
			if ($include_meta || $include_tax || $include_custom || $max_words > 0) {
				$total += $units;
			}
		}

		if ($total <= 0) $total = 1;

		return $total;
	}

	/**
	 * Determine whether the detailed section should be generated.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $post_types Ordered post type data.
	 * @return     bool True if detailed section should be generated.
	 */
	private function should_write_detailed_section(array $post_types): bool {

		if (! is_array($post_types)) return false;

		foreach ($post_types as $pt) {
			$include_meta   = isset($pt['include_meta_info']) && $pt['include_meta_info'] === '1';
			$include_tax    = isset($pt['include_taxonomies']) && $pt['include_taxonomies'] === '1';
			$include_custom = isset($pt['include_custom_fields']) && $pt['include_custom_fields'] === '1';
			$max_words      = isset($pt['max_content_length']) ? absint($pt['max_content_length']) : 1000;
			if ($include_meta || $include_tax || $include_custom || $max_words > 0) return true;
		}

		return false;
	}

	/**
	 * Calculate the current progress percentage for the job.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     int Percent complete.
	 */
	private function calc_percent(array $job): int {

		$processed = isset($job['processed']) ? absint($job['processed']) : 0;
		$total = isset($job['total']) ? absint($job['total']) : 0;

		if ($total <= 0) return 0;

		$percent = (int) floor(($processed / $total) * 100);
		if ($percent < 0) $percent = 0;
		if ($percent > 99 && isset($job['status']) && $job['status'] === 'running') $percent = 99;

		return $percent;
	}

	/**
	 * Build a user-facing progress message for the current job state.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      array $job Generation job data.
	 * @return     string User-facing progress message.
	 */
	private function get_progress_message(array $job): string {

		$percent = $this->calc_percent($job);

		$phase = isset($job['phase']) ? (string)$job['phase'] : 'overview';

		if ($phase === 'overview') {
			return sprintf(
				/* translators: %s percent value */
				esc_html__('Generating overview (%s%%)...', 'mop-ai-indexer'),
				(string)$percent
			);
		}

		if ($phase === 'detailed') {
			return sprintf(
				/* translators: %s percent value */
				esc_html__('Generating detailed content (%s%%)...', 'mop-ai-indexer'),
				(string)$percent
			);
		}

		return sprintf(
			/* translators: %s percent value */
			esc_html__('Generating (%s%%)...', 'mop-ai-indexer'),
			(string)$percent
		);
	}

	/**
	 * Determine whether a post should be skipped based on SEO settings.
	 *
	 * When 'respect-seo' is enabled, skips posts marked as noindex by popular SEO plugins.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Indexability_Resolver
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @param      int $post_id Post ID.
	 * @return     bool True to skip, false to include.
	 */
	private function should_skip_post(int $post_id): bool {

		$post_id = absint($post_id);
		if ($post_id <= 0) return true;

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = is_array($mop_ai_indexer_iset) ? $mop_ai_indexer_iset : array();

		$verdict = Mop_Ai_Indexer_Indexability_Resolver::get_verdict($post_id, $mop_ai_indexer_iset);

		return isset($verdict['excluded']) ? (bool)$verdict['excluded'] : false;
	}


	/**
	 * Resolve a short description for a post (SEO meta/excerpt fallback).
	 *
	 * Prefers SEO meta descriptions, then excerpt, then a trimmed version of content.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @param      int $post_id Post ID.
	 * @param      bool  $allow_content_fallback  Whether post content is allowed as a fallback description source when excerpt/meta is empty.
	 * @return     string Resolved description.
	 */
	private function get_post_description(int $post_id, bool $allow_content_fallback = true): string {

		$post_id = absint($post_id);
		if ($post_id <= 0) return '';

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = is_array($mop_ai_indexer_iset) ? $mop_ai_indexer_iset : array();

		$respect_seo = isset($mop_ai_indexer_iset['iset_respect_seo_config']) ? (string)$mop_ai_indexer_iset['iset_respect_seo_config'] : 'respect-seo';

		$desc = '';

		if ($respect_seo === 'respect-seo') {
			$desc = (string)get_post_meta($post_id, '_yoast_wpseo_metadesc', true);
			if ($desc === '') {
				$desc = (string)get_post_meta($post_id, 'rank_math_description', true);
			}
		}

		if ($desc === '') {
			$excerpt = get_post_field('post_excerpt', $post_id);
			$desc = is_string($excerpt) ? $excerpt : '';
		}

		if ($desc === '' && $allow_content_fallback) {
			$content = $this->get_post_content_text($post_id);
			$desc = $this->trim_words($content, 25);
		}

		$desc = $this->clean_text($desc);

		return $desc;
	}

	/**
	 * Extract and clean the plain-text content for a post.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      int $post_id Post ID.
	 * @return     string Plain-text content.
	 */
	private function get_post_content_text(int $post_id): string {

		$post_id = absint($post_id);
		if ($post_id <= 0) return '';

		$content = get_post_field('post_content', $post_id);
		$content = is_string($content) ? $content : '';

		$content = strip_shortcodes($content);
		$content = wp_strip_all_tags($content, true);
		$content = html_entity_decode($content, ENT_QUOTES, get_bloginfo('charset'));

		$content = $this->clean_text($content);

		return $content;
	}

	/**
	 * Normalize whitespace and clean text for safe output.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $text Text to clean.
	 * @return     string Cleaned text.
	 */
	private function clean_text(string $text): string {

		$text = is_string($text) ? $text : '';
		$text = preg_replace('/\s+/u', ' ', $text);
		$text = trim((string)$text);

		return $text;
	}

	/**
	 * Trim a string to a maximum number of words.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $text Text to trim.
	 * @param      int $max_words Maximum words.
	 * @return     string Trimmed text.
	 */
	private function trim_words(string $text, int $max_words): string {

		$max_words = absint($max_words);
		if ($max_words <= 0) return '';

		$text = is_string($text) ? $text : '';

		return trim(wp_trim_words($text, $max_words, ''));
	}

	/**
	 * Sanitize text intended to be embedded into Markdown-like output.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $text Text to sanitize.
	 * @return     string Sanitized text.
	 */
	private function sanitize_md_text(string $text): string {

		$text = is_string($text) ? $text : '';
		$text = wp_strip_all_tags($text, true);
		$text = html_entity_decode($text, ENT_QUOTES, get_bloginfo('charset'));
		$text = preg_replace('/\s+/u', ' ', $text);
		$text = trim((string)$text);

		return $text;
	}

	/**
	 * Build formatted taxonomy lines for a post.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      int $post_id Post ID.
	 * @param      string $post_type Post type slug.
	 * @return     array Array of formatted taxonomy lines.
	 */
	private function get_taxonomy_lines(int $post_id, string $post_type): array {

		$out = array();

		$post_id = absint($post_id);
		if ($post_id <= 0) return $out;

		$taxes = get_object_taxonomies((string)$post_type, 'objects');
		if (! is_array($taxes) || empty($taxes)) return $out;

		foreach ($taxes as $tax_obj) {

			if (! isset($tax_obj->name)) continue;

			$terms = get_the_terms($post_id, $tax_obj->name);
			if (! is_array($terms) || empty($terms)) continue;

			$names = array();
			foreach ($terms as $t) {
				if (isset($t->name) && $t->name !== '') $names[] = (string)$t->name;
			}

			if (empty($names)) continue;

			$label = isset($tax_obj->labels) && isset($tax_obj->labels->singular_name) ? (string)$tax_obj->labels->singular_name : (string)$tax_obj->label;
			$label = $label !== '' ? $label : (string)$tax_obj->name;

			$out[] = '- ' . $this->sanitize_md_text($label) . ': ' . $this->sanitize_md_text(implode(', ', $names));
		}

		return $out;
	}

	/**
	 * Build formatted custom field (post meta) lines for a post.
	 *
	 * This method reads post meta via get_post_meta() and returns a list of lines in the format:
	 * - Label: Value
	 *
	 * IMPORTANT:
	 * - Only human-readable scalar values are included.
	 * - Private/internal keys (prefixed with "_") are excluded.
	 * - Obvious machine payloads (JSON/serialized/base64 blobs) are excluded.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      int $post_id Post ID.
	 * @return     array Array of formatted custom field lines.
	 */
	private function get_custom_field_lines(int $post_id): array {

		$out = array();

		$post_id = absint($post_id);
		if ($post_id <= 0) return $out;

		$raw_meta = get_post_meta($post_id);
		if (! is_array($raw_meta) || empty($raw_meta)) return $out;

		/**
		 * Sort keys to keep output stable across runs.
		 *
		 * This makes diffs cleaner when the file is regenerated.
		 */
		$keys = array_keys($raw_meta);
		sort($keys, SORT_STRING);

		foreach ($keys as $meta_key) {

			$meta_key = is_string($meta_key) ? $meta_key : '';
			if ($meta_key === '') continue;

			/**
			 * Skip internal/private keys and known builder/system payloads.
			 */
			if ($this->should_skip_custom_meta_key($meta_key)) continue;

			$values = isset($raw_meta[$meta_key]) ? $raw_meta[$meta_key] : array();
			if (! is_array($values) || empty($values)) continue;

			/**
			 * A single meta key can have multiple values. We collect readable values,
			 * de-duplicate them, and join with a comma for a compact representation.
			 */
			$usable = array();

			foreach ($values as $v) {

				if (! $this->is_meta_value_human_readable($v)) {
					continue;
				}

				$txt = $this->clean_text((string)$v);
				if ($txt === '') continue;

				$usable[] = $txt;
			}

			if (empty($usable)) continue;

			$usable = array_values(array_unique($usable));

			/**
			 * Limit the number of values to prevent noise and excessive output.
			 */
			if (count($usable) > 5) {
				$usable = array_slice($usable, 0, 5);
			}

			$value_out = implode(', ', $usable);

			/**
			 * Hard length guard: even readable strings can be too large for a compact index.
			 */
			if (function_exists('mb_strlen') && mb_strlen($value_out) > 300) {
				$value_out = mb_substr($value_out, 0, 297) . '...';
			} elseif (strlen($value_out) > 300) {
				$value_out = substr($value_out, 0, 297) . '...';
			}

			$label = $this->format_custom_field_label($meta_key);

			$out[] = '- ' . $this->sanitize_md_text($label) . ': ' . $this->sanitize_md_text($value_out);
		}

		return $out;
	}

	/**
	 * Determine whether a meta key should be excluded from custom-field output.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $meta_key Meta key name.
	 * @return     bool True if the key should be skipped.
	 */
	private function should_skip_custom_meta_key(string $meta_key): bool {

		$meta_key = is_string($meta_key) ? trim($meta_key) : '';
		if ($meta_key === '') return true;

		/**
		 * Private/internal meta keys are prefixed with "_" by convention.
		 *
		 * This filters out common builder payloads such as "_elementor_data" as well as
		 * WordPress internal housekeeping keys.
		 */
		if (strpos($meta_key, '_') === 0) return true;

		/**
		 * Additional safety guard for known builder keywords in non-private keys.
		 */
		if (stripos($meta_key, 'elementor') !== false) return true;

		return false;
	}

	/**
	 * Determine whether a meta value is safe and human-readable for inclusion.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      mixed $value Raw meta value.
	 * @return     bool True if the value can be included.
	 */
	private function is_meta_value_human_readable($value): bool {

		/**
		 * Arrays/objects usually represent structured or serialized payloads (e.g., builders),
		 * which are not suitable for a compact index file.
		 */
		if (is_array($value) || is_object($value)) return false;

		/**
		 * Numeric values are fine (IDs, prices, etc.).
		 */
		if (is_int($value) || is_float($value)) return true;

		$value = is_string($value) ? trim($value) : (string)$value;
		if ($value === '') return false;

		/**
		 * Hard size guard: avoid dumping large blobs.
		 */
		if (strlen($value) > 500) return false;

		/**
		 * Reject obvious serialized strings (PHP serialize()).
		 */
		if (preg_match('/^(a|O|s|i|b|d):/i', $value)) return false;

		/**
		 * Reject obvious JSON payloads.
		 */
		$first = substr($value, 0, 1);
		if (($first === '{' || $first === '[') && strlen($value) > 200) return false;

		/**
		 * Reject likely base64 blobs.
		 */
		if (strlen($value) > 200 && preg_match('/^[A-Za-z0-9+\/\n\r=]+$/', $value)) return false;

		return true;
	}

	/**
	 * Convert a meta key into a human-readable label.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @param      string $meta_key Meta key name.
	 * @return     string Human-readable label.
	 */
	private function format_custom_field_label(string $meta_key): string {

		$meta_key = is_string($meta_key) ? $meta_key : '';
		$label = str_replace(array('_', '-'), ' ', $meta_key);
		$label = trim($label);

		if ($label === '') return $meta_key;

		return ucwords($label);
	}

	/**
	 * Reset the index file meta option for a new generation run.
	 *
	 * This stores start timestamps so we can later calculate a high-precision duration
	 * when the generation completes.
	 *
	 * @since  1.0.0
	 * @param  string $process Process key.
	 * @param  string $file_name Target file name.
	 * @param  int    $started_at Start timestamp (local WP time).
	 * @param  float  $started_micro Start timestamp (microtime).
	 * @return void
	 */
	private function reset_index_file_meta(string $process, string $file_name, int $started_at, float $started_micro): void {

		$process = $this->sanitize_log_process($process);
		$file_name = $this->sanitize_file_name((string)$file_name);

		$started_at = absint($started_at);
		if ($started_at <= 0) {
			$started_at = current_time('timestamp');
		}

		$started_micro = (float)$started_micro;
		if ($started_micro <= 0) {
			$started_micro = microtime(true);
		}

		$data = array(
			'process' => $process,
			'updated_at' => current_time('timestamp'),
			'file_name' => $file_name,
			'file_size_bytes' => 0,
			'file_size_mb' => '',
			'generation_started_at' => $started_at,
			'generation_started_micro' => $started_micro,
			'generation_ended_at' => 0,
			'generation_ended_micro' => 0,
			'generation_duration_seconds' => 0,
		);

		update_option(self::INDEX_FILE_META_OPTION, $data, false);
	}

	/**
	 * Finalize the index file meta option for a completed generation run.
	 *
	 * @since  1.0.0
	 * @param  array $job Generation job array.
	 * @return void
	 */
	private function finalize_index_file_meta(array $job): void {

		$file_name = isset($job['file_name']) ? (string)$job['file_name'] : '';
		$file_name = $this->sanitize_file_name($file_name);

		$final_file = isset($job['final_file']) ? (string)$job['final_file'] : '';
		$final_file = wp_normalize_path($final_file);

		$ended_at = current_time('timestamp');
		$ended_micro = microtime(true);

		$file_size_bytes = 0;
		if ($final_file !== '' && is_file($final_file)) {
			$fs = @filesize($final_file);
			$file_size_bytes = is_int($fs) ? (int)$fs : 0;
			if ($file_size_bytes < 0) $file_size_bytes = 0;
		}

		$meta = get_option(self::INDEX_FILE_META_OPTION, array());
		$meta = is_array($meta) ? $meta : array();

		$started_at = isset($meta['generation_started_at']) ? absint($meta['generation_started_at']) : 0;
		$started_micro = isset($meta['generation_started_micro']) ? (float)$meta['generation_started_micro'] : 0;

		/**
		 * Prefer the job start timestamps if present.
		 */
		if (isset($job['started_at'])) {
			$job_started_at = absint($job['started_at']);
			if ($job_started_at > 0) $started_at = $job_started_at;
		}

		if (isset($job['started_micro'])) {
			$job_started_micro = (float)$job['started_micro'];
			if ($job_started_micro > 0) $started_micro = $job_started_micro;
		}

		$duration = 0.0;
		if ($started_micro > 0 && $ended_micro > 0 && $ended_micro >= $started_micro) {
			$duration = (float)($ended_micro - $started_micro);
		}

		$mb_str = '';
		if ($file_size_bytes > 0) {
			$mb_str = number_format(((float)$file_size_bytes / 1048576), 2, '.', '');
		}

		$data = array(
			'process' => 'manual_generation',
			'updated_at' => current_time('timestamp'),
			'file_name' => $file_name,
			'file_size_bytes' => absint($file_size_bytes),
			'file_size_mb' => $mb_str,
			'generation_started_at' => $started_at,
			'generation_started_micro' => (float)$started_micro,
			'generation_ended_at' => absint($ended_at),
			'generation_ended_micro' => (float)$ended_micro,
			'generation_duration_seconds' => ($duration > 0) ? (float)$duration : 0,
		);

		update_option(self::INDEX_FILE_META_OPTION, $data, false);
	}

	/**
	 * Delete the stored index file generation meta option.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	private function delete_index_file_meta(): void {

		delete_option(self::INDEX_FILE_META_OPTION);
	}

	/**
	 * Persist last-generated metadata into plugin options.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @param      string $file_name Generated file name.
	 * @param      int $timestamp Generation timestamp.
	 * @return     void
	 */
	private function store_last_generated(string $file_name, int $timestamp): void {

		$file_name = $this->sanitize_file_name((string)$file_name);
		$timestamp = absint($timestamp);

		/**
		 * Persist last-generated metadata as standalone options.
		 *
		 * This keeps mop_ai_indexer_iset focused on user-configurable settings only, while
		 * still allowing the Index Manager UI to display a generated timestamp + file link.
		 */
		update_option('mop_ai_indexer_last_generated', (string)$timestamp);
		update_option('mop_ai_indexer_last_generated_file', $file_name);
	}

	/**
	 * Build the file status HTML for the Index Manager admin UI.
	 *
	 * Outputs the same status line format used by the placeholder: '{file} file is generated at ...'.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     public
	 * @return     string HTML output.
	 */
	public function get_file_status_html(): string {

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = is_array($mop_ai_indexer_iset) ? $mop_ai_indexer_iset : array();

		/**
		 * Determine the current configured output file name and the expected uploads path.
		 *
		 * The index file is stored in the plugin uploads directory and served dynamically
		 * from the site root via rewrite endpoint(s). This status line is used across
		 * the admin UI (Index Manager toolbar + AJAX responses) to clearly show whether
		 * the configured file exists right now.
		 */
		$file_name = isset($mop_ai_indexer_iset['iset_file_name_format']) ? (string)$mop_ai_indexer_iset['iset_file_name_format'] : 'llms.txt';
		$file_name = $this->sanitize_file_name($file_name);
		if ($file_name === '') {
			$file_name = 'llms.txt';
		}

		$upload_dir = wp_upload_dir();
		$basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$basedir = wp_normalize_path($basedir);

		$plugin_uploads_path = ($basedir !== '') ? wp_normalize_path(trailingslashit($basedir) . 'mop-ai-indexer/') : '';

		$upload_file = ($plugin_uploads_path !== '') ? wp_normalize_path(trailingslashit($plugin_uploads_path) . $file_name) : '';

		/**
		 * Load last-generated metadata from standalone options.
		 *
		 * These options are written by the generator when the index file is finalized.
		 * We use them only as a fallback if the filesystem timestamp cannot be read.
		 */
		$last_ts = (int)get_option('mop_ai_indexer_last_generated', 0);
		$last_file = get_option('mop_ai_indexer_last_generated_file', '');
		$last_file = is_string($last_file) ? sanitize_text_field($last_file) : '';
		$last_file = $this->sanitize_file_name($last_file);

		$meta = get_option(self::INDEX_FILE_META_OPTION, array());
		$meta = is_array($meta) ? $meta : array();

		$meta_file = isset($meta['file_name']) ? (string)$meta['file_name'] : '';
		$meta_file = $this->sanitize_file_name($meta_file);
		$meta_mb = isset($meta['file_size_mb']) ? (string)$meta['file_size_mb'] : '';
		$meta_mb = is_string($meta_mb) ? trim($meta_mb) : '';
		$meta_end_at = isset($meta['generation_ended_at']) ? absint($meta['generation_ended_at']) : 0;
		$meta_duration = isset($meta['generation_duration_seconds']) ? (float)$meta['generation_duration_seconds'] : 0;


		$url = home_url('/' . $file_name);

		if ($upload_file !== '' && is_file($upload_file)) {
			$ts = @filemtime($upload_file);
			$ts = is_int($ts) ? (int)$ts : 0;

			if ($ts <= 0) {
				$ts = ($last_file === $file_name && $last_ts > 0) ? $last_ts : time();
			}

			$dt = date_i18n('d/m/Y g:i A', $ts);

			$size_bytes = @filesize($upload_file);
			$size_bytes = is_int($size_bytes) ? (int)$size_bytes : 0;
			if ($size_bytes < 0) $size_bytes = 0;

			$size_mb = '';
			if ($size_bytes > 0) {
				$size_mb = number_format(((float)$size_bytes / 1048576), 2, '.', '');
			} elseif ($meta_file === $file_name && $meta_mb !== '') {
				$size_mb = $meta_mb;
			}

			$size_part = ($size_mb !== '') ? ' (' . esc_html($size_mb) . ' MB)' : '';

			$duration_part = '';
			if ($meta_file === $file_name && $meta_end_at > 0 && $meta_duration > 0) {
				$duration_str = number_format((float)$meta_duration, 4, '.', '');
				$duration_part = ' ' . esc_html__('Generation time', 'mop-ai-indexer') . ' ' . esc_html($duration_str) . ' ' . esc_html__('seconds.', 'mop-ai-indexer');
			}

			return '<a href="' . esc_url($url) . '" target="_blank" rel="noopener noreferrer">' . esc_html($file_name) . '</a> ' .
			esc_html__('file', 'mop-ai-indexer') . $size_part . ' ' . esc_html__('is generated at', 'mop-ai-indexer') . ' ' . esc_html($dt) . '.' . $duration_part;
		}

		/**
		 * If the configured file does not exist in uploads, clear last-generated metadata and show not-generated.
		 *
		 * Do not clear meta/options while a generation job is actively running.
		 */
		$gen_job = get_transient(self::get_job_key());
		$gen_running = (is_array($gen_job) && isset($gen_job['status']) && $gen_job['status'] === 'running');

		if (! $gen_running) {
			delete_option('mop_ai_indexer_last_generated');
			delete_option('mop_ai_indexer_last_generated_file');
			$this->delete_index_file_meta();
		}

		return '<a href="#" aria-disabled="true">' . esc_html($file_name) . '</a> ' . esc_html__('file is not generated yet.', 'mop-ai-indexer');
	}

	/**
	 * Retrieves index settings option defaults.
	 *
	 * These defaults must align with the values displayed on the Index Settings page
	 * when the plugin is installed but settings have not been saved yet.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @return     array Default index settings array.
	 */
	private static function get_iset_defaults() {
		return Mop_Ai_Indexer_Defaults::get_iset_defaults();

	}

	/**
	 * Retrieves public post types in the same order used by the Index Manager UI.
	 *
	 * This is required so default index configuration matches what the user sees in
	 * the admin form before any configuration is saved.
	 *
	 * Rules:
	 * - 'page' and 'post' should appear first (if available).
	 * - Remaining public post types are sorted alphabetically by label.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @return     array Map of post type slug => singular label.
	 */
	private static function get_public_post_types_in_order() {
		return Mop_Ai_Indexer_Defaults::get_public_post_types_in_order();

	}

	/**
	 * Builds the default index configuration structure.
	 *
	 * The defaults are used when mop_ai_indexer_config does not exist in the options table
	 * (fresh install) so that index generation can run immediately.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @param      array  $post_types Map of post type slug => label.
	 * @return     array  Default mop_ai_indexer_config array.
	 */
	private static function get_default_index_config($post_types) {
		return Mop_Ai_Indexer_Defaults::get_default_index_config($post_types);

	}

	/**
	 * Merges a saved configuration array with defaults.
	 *
	 * Ensures that:
	 * - all expected post types exist in the config,
	 * - each post type includes all expected fields.
	 *
	 * @since      1.0.0
	 * @see        Mop_Ai_Indexer_Defaults
	 * @access     private
	 * @param      mixed  $saved Saved mop_ai_indexer_config option (may be invalid or incomplete).
	 * @param      array  $defaults Default config structure.
	 * @return     array  Merged configuration array.
	 */
	private static function merge_config_with_defaults($saved, $defaults) {

		return Mop_Ai_Indexer_Defaults::merge_config_with_defaults(
			(is_array($saved) ? $saved : array()),
			(is_array($defaults) ? $defaults : array())
		);

	}

	/**
	 * Get the transient key used to store the generation job state.
	 *
	 * Centralizes the transient key so all AJAX handlers refer to the same job storage.
	 *
	 * @since      1.0.0
	 * @access     private
	 * @return     string Transient job key.
	 */
	private static function get_delete_job_key() {
		return 'mop_ai_indexer_file_delete_job';
	}

	/**
	 * Determine whether a generation job is currently running.
	 *
	 * Used to prevent full cache flushes from breaking in-progress jobs.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public static function is_generation_job_running(): bool {

		$job = get_transient(self::get_job_key());
		return is_array($job) && isset($job['status']) && $job['status'] === 'running';
	}

	/**
	 * Determine whether a deletion job is currently running.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public static function is_deletion_job_running(): bool {

		$job = get_transient(self::get_delete_job_key());
		return is_array($job) && isset($job['status']) && $job['status'] === 'running';
	}

	/**
	 * Get job key.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @return   string
	 */
	private static function get_job_key() {
		return 'mop_ai_indexer_file_gen_job';
	}
}
