<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Provides default option values for the MOP AI Indexer plugin.
 *
 * This class is the single source of truth for default values used by:
 * - Index Settings page (mop_ai_indexer_iset)
 * - Index Manager page (mop_ai_indexer_config)
 * - File generation logic (Mop_Ai_Indexer_File_Manager)
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * MOP AI Indexer defaults provider.
 *
 * @since 1.0.0
 */
class Mop_Ai_Indexer_Defaults {

	/**
	 * Returns default Index Settings (mop_ai_indexer_iset) values.
	 *
	 * @since  1.0.0
	 * @return array Default index settings array.
	 */
	public static function get_iset_defaults(): array {

		return array(
			'iset_file_name_format'          => 'llms.txt',
			'iset_update_frequency'          => 'once-a-week',
			'iset_enable_mop_ai_indexer_post_meta_box' => 'enable-meta-box',
			'iset_respect_seo_config'        => 'respect-seo',
			'iset_strict_indexability_check' => '',
			'iset_purge_caches_after_generation_deletion' => 'purge-caches',
		);
	}

	/**
	 * Returns default Index Configuration (mop_ai_indexer_config) values.
	 *
	 * Defaults are derived from the site's public post types, ordered as:
	 * - 'page' and 'post' first (if available)
	 * - remaining post types alphabetically by singular label
	 *
	 * @since  1.0.0
	 * @return array Default index configuration array.
	 */
	public static function get_config_defaults(): array {

		$post_types = self::get_public_post_types_in_order();

		return self::get_default_index_config($post_types);
	}

	/**
	 * Returns public post types in a stable display/generation order.
	 *
	 * Rules:
	 * - 'page' and 'post' should appear first (if available).
	 * - Remaining public post types are sorted alphabetically by label.
	 *
	 * @since  1.0.0
	 * @return array Map of post type slug => singular label.
	 */
	public static function get_public_post_types_in_order(): array {

		/**
		 * Retrieve post type objects in a broad, stable way.
		 *
		 * We intentionally do NOT require `show_ui` here because some sites can
		 * customize built-in post types and hide them from UI while still keeping
		 * them publicly queryable (e.g. 'page'). We still want 'page' and 'post'
		 * to be available for Index generation and configuration.
		 */
		$args = array(
			'public'             => true,
			'publicly_queryable' => true,
		);

		$objects = get_post_types($args, 'objects');

		$out = array();

		/**
		 * Always attempt to include core 'page' and 'post' first if available.
		 */
		$page_obj = isset($objects['page']) ? $objects['page'] : get_post_type_object('page');
		if ($page_obj instanceof \WP_Post_Type) {
			$out['page'] = isset($page_obj->labels->singular_name) ? (string)$page_obj->labels->singular_name : 'Page';
		}

		$post_obj = isset($objects['post']) ? $objects['post'] : get_post_type_object('post');
		if ($post_obj instanceof \WP_Post_Type) {
			$out['post'] = isset($post_obj->labels->singular_name) ? (string)$post_obj->labels->singular_name : 'Post';
		}

		$others = array();

		foreach ($objects as $slug => $obj) {

			$slug = is_string($slug) ? $slug : '';

			/**
			 * Exclude empty slugs and always skip core types we handle separately.
			 * Also exclude 'attachment' (Media), which should not be shown as a config card
			 * or included in Index generation.
			 */
			if ($slug === '' || $slug === 'page' || $slug === 'post' || $slug === 'attachment') {
				continue;
			}

			$label = isset($obj->labels->singular_name) ? (string)$obj->labels->singular_name : $slug;
			$others[$slug] = $label;
		}

		if (! empty($others)) {
			asort($others, SORT_NATURAL | SORT_FLAG_CASE);
			foreach ($others as $slug => $label) {
				$out[$slug] = $label;
			}
		}

		return $out;
	}


	/**
	 * Builds the default index configuration array for a given post type map.
	 *
	 * @since  1.0.0
	 * @param  array $post_types Map of post type slug => singular label.
	 * @return array Default configuration array.
	 */
	public static function get_default_index_config(array $post_types): array {

		$defaults = array('post_type_config' => array());
		$priority = 1;

		foreach ($post_types as $slug => $label) {

			$slug = is_string($slug) ? $slug : '';
			$label = is_string($label) ? $label : $slug;

			if ($slug === '') {
				continue;
			}

			$defaults['post_type_config'][$slug] = array(
				'include_in_mop_ai_indexer'     => ($slug === 'page' || $slug === 'post') ? '1' : '0',
				'order_priority_in_index' => (string)$priority,
				'num_of_latest_posts'     => '100',
				'max_content_length'      => '1000',
				'include_meta_info'       => '1',
				'include_exc_meta_desc'   => '0',
				'include_taxonomies'      => '1',
				'include_custom_fields'   => '1',
			);

			$priority++;
		}

		return $defaults;
	}

	/**
	 * Merges saved mop_ai_indexer_config values with defaults.
	 *
	 * Ensures:
	 * - missing post types are added
	 * - missing keys per post type are filled
	 * - saved values override defaults when present
	 *
	 * @since  1.0.0
	 * @param  array $saved    Saved configuration array.
	 * @param  array $defaults Default configuration array.
	 * @return array Merged configuration array.
	 */
	public static function merge_config_with_defaults(array $saved, array $defaults): array {

		$out = $defaults;

		if (! isset($out['post_type_config']) || ! is_array($out['post_type_config'])) {
			$out['post_type_config'] = array();
		}

		if (! isset($saved['post_type_config']) || ! is_array($saved['post_type_config'])) {
			return $out;
		}

		foreach ($saved['post_type_config'] as $slug => $cfg) {

			$slug = is_string($slug) ? $slug : '';
			if ($slug === '' || ! is_array($cfg)) {
				continue;
			}

			if (! isset($out['post_type_config'][$slug]) || ! is_array($out['post_type_config'][$slug])) {
				$out['post_type_config'][$slug] = array();
			}

			$out['post_type_config'][$slug] = array_merge($out['post_type_config'][$slug], $cfg);
		}

		return $out;
	}
}
