<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Manages WP-Cron schedules for MOP AI Indexer.
 *
 * Note:
 * - This class only registers schedules and schedules/unschedules the cron event.
 * - It does NOT connect cron events to index file generation (that will be added later).
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * WP-Cron schedule manager.
 *
 * @since 1.0.0
 */
class Mop_Ai_Indexer_Cron_Manager {

	/**
	 * Cron hook name used by MOP AI Indexer.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	public const CRON_HOOK = 'mop_ai_indexer_cron_generation_event';

	/**
	 * Returns a map of frequency slugs to custom schedule keys.
	 *
	 * @since 1.0.0
	 * @return array<string,string>
	 */
	public static function get_frequency_to_schedule_key_map(): array {

		return array(
			'twice-a-day'    => 'mop_ai_indexer_twice_a_day',
			'once-a-day'     => 'mop_ai_indexer_once_a_day',
			'every-2-days'   => 'mop_ai_indexer_every_2_days',
			'every-3-days'   => 'mop_ai_indexer_every_3_days',
			'every-4-days'   => 'mop_ai_indexer_every_4_days',
			'every-5-days'   => 'mop_ai_indexer_every_5_days',
			'every-6-days'   => 'mop_ai_indexer_every_6_days',
			'once-a-week'    => 'mop_ai_indexer_once_a_week',
			'every-2-weeks'  => 'mop_ai_indexer_every_2_weeks',
			'every-3-weeks'  => 'mop_ai_indexer_every_3_weeks',
			'every-4-weeks'  => 'mop_ai_indexer_every_4_weeks',
		);
	}

	/**
	 * Registers MOP AI Indexer custom schedules.
	 *
	 * @since 1.0.0
	 * @param array<string,array<string,mixed>> $schedules Existing schedules.
	 * @return array<string,array<string,mixed>> Updated schedules.
	 */
	public function register_custom_schedules(array $schedules): array {

		$schedules['mop_ai_indexer_twice_a_day'] = array(
			'interval' => 12 * HOUR_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Twice a day',
		);

		$schedules['mop_ai_indexer_once_a_day'] = array(
			'interval' => DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Once a day',
		);

		$schedules['mop_ai_indexer_every_2_days'] = array(
			'interval' => 2 * DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 2 days',
		);

		$schedules['mop_ai_indexer_every_3_days'] = array(
			'interval' => 3 * DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 3 days',
		);

		$schedules['mop_ai_indexer_every_4_days'] = array(
			'interval' => 4 * DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 4 days',
		);

		$schedules['mop_ai_indexer_every_5_days'] = array(
			'interval' => 5 * DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 5 days',
		);

		$schedules['mop_ai_indexer_every_6_days'] = array(
			'interval' => 6 * DAY_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 6 days',
		);

		$schedules['mop_ai_indexer_once_a_week'] = array(
			'interval' => WEEK_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Once a week',
		);

		$schedules['mop_ai_indexer_every_2_weeks'] = array(
			'interval' => 2 * WEEK_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 2 weeks',
		);

		$schedules['mop_ai_indexer_every_3_weeks'] = array(
			'interval' => 3 * WEEK_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 3 weeks',
		);

		$schedules['mop_ai_indexer_every_4_weeks'] = array(
			'interval' => 4 * WEEK_IN_SECONDS,
			'display'  => 'MOP AI Indexer - Every 4 weeks',
		);

		return $schedules;
	}

	/**
	 * Ensures the cron schedule matches the saved Update Frequency setting.
	 *
	 * This is a lightweight guard that:
	 * - schedules the event if missing (and not manual)
	 * - unschedules the event if manual
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function maybe_sync_schedule(): void {

		self::sync_schedule_from_settings();
	}

	/**
	 * Sync schedule from saved settings (mop_ai_indexer_iset).
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function sync_schedule_from_settings(): void {

		$iset = get_option('mop_ai_indexer_iset', array());
		if (! is_array($iset)) $iset = array();

		$frequency = isset($iset['iset_update_frequency']) ? (string)$iset['iset_update_frequency'] : '';

		if ($frequency === 'manual' || $frequency === '') {
			self::unschedule_all_events();
			return;
		}

		$map = self::get_frequency_to_schedule_key_map();
		if (! isset($map[$frequency])) {
			self::unschedule_all_events();
			return;
		}

		$schedule_key = (string)$map[$frequency];

		/**
		 * Check existing schedule and avoid rescheduling if already correct.
		 */
		$timestamp        = wp_next_scheduled(self::CRON_HOOK);
		$current_schedule = $timestamp ? wp_get_schedule(self::CRON_HOOK) : '';

		if ($current_schedule === $schedule_key) {
			return;
		}

		/**
		 * Ensure only one scheduled event exists for this hook.
		 */
		self::unschedule_all_events();

		/**
		 * Schedule the event based on the defined interval for the schedule key.
		 *
		 * If the schedule definition is missing for some reason, fall back to 60 seconds.
		 */
		$schedules = wp_get_schedules();
		$interval  = isset($schedules[$schedule_key]['interval']) ? (int)$schedules[$schedule_key]['interval'] : 0;

		$first_run = time() + ($interval > 0 ? $interval : 60);

		wp_schedule_event($first_run, $schedule_key, self::CRON_HOOK);
	}

	/**
	 * Unschedules all MOP AI Indexer cron events for the hook.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function unschedule_all_events(): void {

		$timestamp = wp_next_scheduled(self::CRON_HOOK);
		while ($timestamp) {
			wp_unschedule_event($timestamp, self::CRON_HOOK);
			$timestamp = wp_next_scheduled(self::CRON_HOOK);
		}
	}
}
