<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Cache manager for MOP AI Indexer.
 *
 * Provides a centralized, best-effort cache purge routine that can be used:
 * - Manually (admin action)
 * - Automatically (after successful generation/deletion), when enabled
 *
 * Notes:
 * - Cache purging is intentionally best-effort and must never break the generation flow.
 * - WordPress object cache flushing is only executed when it is safe to do so.
 *
 * @since 1.0.0
 * @package Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Provides cache purge utilities and integration helpers for common caching layers.
 *
 * This class is to provide cache purge utilities and integration helpers for common caching layers..
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Cache_Manager {

	/**
	 * Settings key used to enable automatic cache purging after generation/deletion.
	 *
	 * @since 1.0.0
	 */
	private const ISET_KEY_PURGE_AFTER = 'iset_purge_caches_after_generation_deletion';

	/**
	 * Settings value indicating the checkbox is enabled.
	 *
	 * @since 1.0.0
	 */
	private const ISET_VALUE_PURGE_AFTER = 'purge-caches';

	/**
	 * Determine whether automatic cache purge is enabled.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Defaults
	 * @return   bool
	 */
	public static function is_purge_after_enabled(): bool {

		$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		$mop_ai_indexer_iset = is_array($mop_ai_indexer_iset) ? $mop_ai_indexer_iset : array();

		$value = isset($mop_ai_indexer_iset[self::ISET_KEY_PURGE_AFTER]) ? (string)$mop_ai_indexer_iset[self::ISET_KEY_PURGE_AFTER] : '';

		return ($value === self::ISET_VALUE_PURGE_AFTER);
	}

	/**
	 * Purge supported caches.
	 *
	 * @since 1.0.0
	 * @param string $context Context identifier (for logging / debugging).
	 * @param bool   $allow_wp_cache_flush Whether wp_cache_flush() may be called.
	 * @return array{
	 *	status:string,
	 *	purged:array<int,string>,
	 *	warnings:array<int,string>,
	 *	errors:array<int,string>
	 * }
	 */
	public static function purge_all(string $context, bool $allow_wp_cache_flush = true): array {

		$context = sanitize_key($context);

		$result = array(
			'status' => 'ok',
			'purged' => array(),
			'warnings' => array(),
			'errors' => array(),
		);

		/**
		 * WordPress object cache (best effort).
		 *
		 * IMPORTANT: This can clear transients. Only call when explicitly allowed.
		 */
		if ($allow_wp_cache_flush && function_exists('wp_cache_flush')) {
			try {
				wp_cache_flush();
				$result['purged'][] = 'WordPress object cache';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'WordPress object cache flush failed.';
			}
		} elseif (! $allow_wp_cache_flush) {
			$result['warnings'][] = 'WordPress object cache flush skipped (a generation/deletion job may be running).';
		}

		/**
		 * Autoptimize.
		 */
		if (class_exists('autoptimizeCache') && method_exists('autoptimizeCache', 'clearall')) {
			try {
				\autoptimizeCache::clearall();
				$result['purged'][] = 'Autoptimize';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'Autoptimize cache purge failed.';
			}
		}

		/**
		 * LiteSpeed Cache.
		 */
		if (class_exists('LiteSpeed\\Purge') && method_exists('LiteSpeed\\Purge', 'purge_all')) {
			try {
				\LiteSpeed\Purge::purge_all();
				$result['purged'][] = 'LiteSpeed Cache';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'LiteSpeed cache purge failed.';
			}
		}

		/**
		 * WP Rocket.
		 */
		if (function_exists('rocket_clean_domain')) {
			try {
				rocket_clean_domain();
				$result['purged'][] = 'WP Rocket';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'WP Rocket cache purge failed.';
			}
		}

		/**
		 * W3 Total Cache.
		 */
		if (function_exists('w3tc_pgcache_flush')) {
			try {
				w3tc_pgcache_flush();
				$result['purged'][] = 'W3 Total Cache';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'W3 Total Cache purge failed.';
			}
		}

		/**
		 * WP Super Cache.
		 */
		if (function_exists('wp_cache_clear_cache')) {
			try {
				wp_cache_clear_cache();
				$result['purged'][] = 'WP Super Cache';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'WP Super Cache purge failed.';
			}
		}

		/**
		 * WP Fastest Cache.
		 */
		if (class_exists('WpFastestCache') && method_exists('WpFastestCache', 'deleteCache')) {
			try {
				$wpfc = new \WpFastestCache();
				$wpfc->deleteCache(true);
				$result['purged'][] = 'WP Fastest Cache';
			} catch (\Throwable $e) {
				$result['warnings'][] = 'WP Fastest Cache purge failed.';
			}
		}

		if (empty($result['purged']) && empty($result['warnings']) && empty($result['errors'])) {
			$result['status'] = 'noop';
			$result['warnings'][] = 'No supported caching plugins were detected.';
		}

		return $result;
	}

	/**
	 * Purge caches after a successful generation/deletion run, only when enabled.
	 *
	 * @since 1.0.0
	 * @param string $context One of: manual_generation_ajax, manual_deletion_ajax, scheduled_generation_cron.
	 * @param bool   $allow_wp_cache_flush Whether wp_cache_flush() may be called.
	 * @return array|null Purge result if executed, otherwise null.
	 */
	public static function maybe_purge_after_success(string $context, bool $allow_wp_cache_flush = true): ?array {

		if (! self::is_purge_after_enabled()) {
			return null;
		}

		return self::purge_all($context, $allow_wp_cache_flush);
	}
}
