<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes\Logic;

/**
 * Provides a future-proof framework for agent allow/deny policies and endpoint hit logging.
 *
 * Current behaviour (v1.1.6 scope):
 * - allow all requests by default
 * - expose a filter hook so developers can implement allow/deny logic
 * - optionally log endpoint hits when enabled by constant
 *
 * This class is designed to scale for a future UI where admins can select which
 * AI bots are permitted to access the endpoint.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Evaluates user-agent policies for MOP AI indexer endpoint access and optional logging.
 *
 * This class is to evaluate user-agent policies for MOP AI indexer endpoint access and optional logging.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Agent_Policy_Manager {

	/**
	 * Read and sanitize the request user-agent.
	 *
	 * @since  1.0.0
	 * @return string
	 */
	public function get_user_agent(): string {

		$ua = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash((string)$_SERVER['HTTP_USER_AGENT'])) : '';
		$ua = trim($ua);

		/**
		 * Hard length guard to avoid log abuse / overly large headers.
		 */
		if (strlen($ua) > 500) {
			$ua = substr($ua, 0, 500);
		}

		return $ua;
	}

	/**
	 * Determine whether a given user-agent should be allowed to access the endpoint.
	 *
	 * Default is "allow". Developers can override via filter.
	 *
	 * Filter:
	 * - mop_ai_indexer_is_agent_allowed
	 *
	 * @since  1.0.0
	 * @param  string $user_agent Raw user-agent header value.
	 * @param  string $file_name  Endpoint file name being requested.
	 * @return bool
	 */
	public function is_allowed(string $user_agent, string $file_name): bool {

		$allowed = true;

		/**
		 * Allow third parties to implement their own policy rules.
		 *
		 * Example usage in theme/plugin:
		 * add_filter('mop_ai_indexer_is_agent_allowed', function($allowed, $ua, $file) {
		 *     if (stripos($ua, 'SomeBot') !== false) return true;
		 *     return $allowed;
		 * }, 10, 3);
		 */
		$allowed = apply_filters('mop_ai_indexer_is_agent_allowed', $allowed, $user_agent, $file_name);

		return (bool)$allowed;
	}

	/**
	 * Optionally log endpoint hits.
	 *
	 * Controlled via constant:
	 * - MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS (default false)
	 *
	 * @since  1.0.0
	 * @param  array $data Log data.
	 * @return void
	 */
	public function maybe_log_hit(array $data): void {

		$enabled = false;
		if (defined('MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS')) {
			$enabled = (bool)MOP_AI_INDEXER_ENDPOINT_LOG_AGENT_HITS;
		}

		if (! $enabled) return;

		/**
		 * Keep logging light-weight:
		 * - single line per request
		 * - stored in uploads directory to avoid touching site root
		 *
		 * If uploads is not writable, logging is silently skipped (no fatal errors).
		 */
		$upload_dir = wp_upload_dir();
		$basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$basedir = wp_normalize_path($basedir);
		if ($basedir === '' || ! is_dir($basedir)) return;

		$plugin_uploads_path = wp_normalize_path(trailingslashit($basedir) . 'mop-ai-indexer/');
		if (! is_dir($plugin_uploads_path)) {
			wp_mkdir_p($plugin_uploads_path);
		}
		if (! is_dir($plugin_uploads_path)) return;

		$log_file = wp_normalize_path(trailingslashit($plugin_uploads_path) . 'mop-ai-indexer-endpoint-access.log');

		$ts = date_i18n('Y-m-d H:i:s', current_time('timestamp'));
		$file_name = isset($data['file_name']) ? (string)$data['file_name'] : '';
		$allowed = isset($data['allowed']) ? (string)$data['allowed'] : '1';
		$ua = isset($data['user_agent']) ? (string)$data['user_agent'] : '';
		$ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash((string)$_SERVER['REMOTE_ADDR'])) : '';

		$line = $ts . "\t" . $ip . "\t" . $allowed . "\t" . $file_name . "\t" . $ua . "\n";

		@file_put_contents($log_file, $line, FILE_APPEND);
	}
}
