<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes;

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * common-area of the site and the admin area.
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Admin\{Mop_Ai_Indexer_Admin_Enqueue, Mop_Ai_Indexer_Admin_Menu, Mop_Ai_Indexer_Admin_Notice_Manager};
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_File_Manager, Mop_Ai_Indexer_Endpoint_Controller, Mop_Ai_Indexer_Index_Discovery, Mop_Ai_Indexer_Sitemap_Integration, Mop_Ai_Indexer_Cron_Manager, Mop_Ai_Indexer_Scheduled_Runner};
use Mop_Ai_Indexer\Admin\Logic\{Mop_Ai_Indexer_Post_Control_Manager, Mop_Ai_Indexer_Cache_Admin_Actions};

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * common-area site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Mop_Ai_Indexer_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected Mop_Ai_Indexer_Loader $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_slug    The string used to uniquely identify this plugin.
	 */
	protected string $plugin_slug;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected string $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin slug and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the common-area of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		if (defined('MOP_AI_INDEXER_VERSION')) {
			$this->version = MOP_AI_INDEXER_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_slug = MOP_AI_INDEXER_SLUG;

		$this->load_plugin_loader();
		$this->define_general_hooks();
		$this->define_admin_hooks();
	}

	/**
	 * Load the plugin loader for this plugin.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @see      Mop_Ai_Indexer_Loader
	 */
	private function load_plugin_loader(): void {

		$this->loader = new Mop_Ai_Indexer_Loader();
	}

	/**
	 * Register all of the hooks related to general functionalities
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Cron_Manager
	 * @see      Mop_Ai_Indexer_Endpoint_Controller
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @see      Mop_Ai_Indexer_Index_Discovery
	 * @see      Mop_Ai_Indexer_Sitemap_Integration
	 * @access   private
	 */
	private function define_general_hooks(): void {

		/**
		 * Endpoint router: registers rewrite rules + query vars.
		 */
		$router_obj = new Mop_Ai_Indexer_Endpoint_Router();
		$this->loader->add_action('init', $router_obj, 'register_rewrite_rules');
		$this->loader->add_filter('query_vars', $router_obj, 'register_query_vars');

		/**
		 * Endpoint controller: serves the generated file via WordPress routing.
		 *
		 * Hooked with priority 0 so it executes before redirect_canonical and before theme output.
		 */
		$endpoint_controller_obj = new Mop_Ai_Indexer_Endpoint_Controller();
		$this->loader->add_action('template_redirect', $endpoint_controller_obj, 'maybe_serve_endpoint', 0);


		/**
		 * Cron manager: registers custom schedules and ensures the cron hook is scheduled.
		 */
		$cron_manager_obj = new Mop_Ai_Indexer_Cron_Manager();
		$this->loader->add_filter('cron_schedules', $cron_manager_obj, 'register_custom_schedules');
		$this->loader->add_action('init', $cron_manager_obj, 'maybe_sync_schedule');

		/**
		 * Cache/minification guards for common optimization plugins.
		 *
		 * These filters are only used if the respective optimization plugin is installed.
		 */
		$this->loader->add_filter('autoptimize_filter_noptimize', $endpoint_controller_obj, 'autoptimize_disable_optimization');
		$this->loader->add_filter('litespeed_cache_is_cacheable', $endpoint_controller_obj, 'litespeed_disable_cache');
		$this->loader->add_filter('rocket_cache_reject_uri', $endpoint_controller_obj, 'wp_rocket_reject_uri');

		/**
		 * Discovery hints: optional <link ...> in wp_head and optional robots.txt comment hint.
		 */
		$discovery_obj = new Mop_Ai_Indexer_Index_Discovery();
		$this->loader->add_action('wp_head', $discovery_obj, 'output_head_hint');
		$this->loader->add_filter('robots_txt', $discovery_obj, 'add_robots_txt_hint', 10, 2);

		/**
		 * Optional core sitemap integration, gated by constants.
		 */
		$sitemap_integration_obj = new Mop_Ai_Indexer_Sitemap_Integration();
		$this->loader->add_action('wp_sitemaps_init', $sitemap_integration_obj, 'maybe_register_sitemap_provider', 20);
	}
	
	/**
	 * Register all of the hooks related to the admin area functionalities
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Cache_Admin_Actions
	 * @see      Mop_Ai_Indexer_Post_Control_Manager
	 * @see      Mop_Ai_Indexer_Scheduled_Runner
	 * @see      Mop_Ai_Indexer_Admin_Enqueue
	 * @see      Mop_Ai_Indexer_Admin_Menu
	 * @see      Mop_Ai_Indexer_Admin_Notice_Manager
	 * @see      Mop_Ai_Indexer_File_Manager
	 * @access   private
	 */
	private function define_admin_hooks(): void {

		/**
		 * Enqueue style and script files for admin area.
		 */
		$admin_enqueue_obj = new Mop_Ai_Indexer_Admin_Enqueue($this->get_plugin_slug(), $this->get_version());
		$this->loader->add_action('admin_enqueue_scripts', $admin_enqueue_obj, 'enqueue_styles');
		$this->loader->add_action('admin_enqueue_scripts', $admin_enqueue_obj, 'enqueue_scripts');

		/**
		 * Add top-level administrative menu page and sub-level menu pages.
		 */
		$admin_menu_obj = new Mop_Ai_Indexer_Admin_Menu();
		$this->loader->add_action('admin_menu', $admin_menu_obj, 'add_menu_pages');

		/**
		 * Filter third-party/admin notice clutter on MOP AI Indexer admin pages only.
		 */
		$notice_manager_obj = new Mop_Ai_Indexer_Admin_Notice_Manager();
		$this->loader->add_action('in_admin_header', $notice_manager_obj, 'notice_filter_for_mop_ai_indexer', 10, 0);

		/**
		 * Manage per-post controls.
		 */
		$post_control_obj = new Mop_Ai_Indexer_Post_Control_Manager();
		$this->loader->add_action('add_meta_boxes', $post_control_obj, 'register_meta_box', 10, 0);
		$this->loader->add_action('save_post', $post_control_obj, 'save_post_control_meta');
		$this->loader->add_action('admin_init', $post_control_obj, 'register_list_table_hooks');
		$this->loader->add_action('quick_edit_custom_box', $post_control_obj, 'render_quick_edit', 10, 2);
		$this->loader->add_action('admin_enqueue_scripts', $post_control_obj, 'output_quick_edit_js', 10, 0);

		/**
		 * Manage index file via AJAX.
		 */
		$file_manager_obj = Mop_Ai_Indexer_File_Manager::get_instance();
		$this->loader->add_action('wp_ajax_mop_ai_indexer_start_generation', $file_manager_obj, 'ajax_start_generation');
		$this->loader->add_action('wp_ajax_mop_ai_indexer_process_generation', $file_manager_obj, 'ajax_process_generation');
		$this->loader->add_action('wp_ajax_mop_ai_indexer_get_generation_status', $file_manager_obj, 'ajax_get_generation_status');
		$this->loader->add_action('wp_ajax_mop_ai_indexer_start_deletion', $file_manager_obj, 'ajax_start_deletion');
		$this->loader->add_action('wp_ajax_mop_ai_indexer_process_deletion', $file_manager_obj, 'ajax_process_deletion');
		$this->loader->add_action('wp_ajax_mop_ai_indexer_delete_index_file', $file_manager_obj, 'ajax_delete_index_file');

		/**
		 * Manual cache maintenance action (admin-post).
		 */
		$cache_actions_obj = new Mop_Ai_Indexer_Cache_Admin_Actions();
		$this->loader->add_action('admin_post_mop_ai_indexer_clear_caches', $cache_actions_obj, 'handle_clear_caches_action');

		/**
		 * Scheduled generation runner (WP-Cron -> signed loopback steps).
		 */
		$scheduled_runner_obj = new Mop_Ai_Indexer_Scheduled_Runner();
		$this->loader->add_action('wp_ajax_mop_ai_indexer_scheduled_step_runner', $scheduled_runner_obj, 'ajax_step_runner');
		$this->loader->add_action('wp_ajax_nopriv_mop_ai_indexer_scheduled_step_runner', $scheduled_runner_obj, 'ajax_step_runner');
		$this->loader->add_action('mop_ai_indexer_cron_generation_event', $scheduled_runner_obj, 'cron_start_scheduled_generation');
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run(): void {
		$this->loader->run();
	}

	/**
	 * The slug of the plugin used to uniquely identify it within the context of WordPress.
	 *
	 * @since     1.0.0
	 * @return    string    The slug of the plugin.
	 */
	public function get_plugin_slug(): string {
		return $this->plugin_slug;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version(): string {
		return $this->version;
	}
}
