<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes;

/**
 * Checks required system requirements
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Checks required system requirements.
 *
 * This class is to check the system for required minimum PHP Version and WordPress Version
 * for this plugin and to create an array of errors accordingly.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 * @author     Anjana Hemachandra
 */
final class Mop_Ai_Indexer_Requirement_Checker {

	/**
	 * Errors associative array.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $requirement_errors    Contains errors associative array.
	 */
	private static array $requirement_errors = array();

	/**
	 * Checks the system for required minimum PHP Version.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @return   bool   True on system PHP version is higher than the version required by the plugin. Otherwise false.
	 */
	private static function php_version_match(): bool {

		if (version_compare(PHP_VERSION, MOP_AI_INDEXER_MIN_PHP_VERSION, '>=')) return true;
		return false;
	}

	/**
	 * Checks the system for required minimum WordPress Version.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @return   bool   True on WordPress version is higher than the version required by the plugin. Otherwise false.
	 */
	private static function wp_version_match(): bool {

		global $wp_version;

		if (version_compare($wp_version, MOP_AI_INDEXER_MIN_WP_VERSION, '>=')) return true;
		return false;
	}

	/**
	 * Creates an array of errors accordingly.
	 *
	 * @since     1.0.0
	 * @return    array    Errors associative array.
	 */
	public static function get_requirement_errors(): array {

		if (! self::php_version_match()) self::$requirement_errors['php_version_error'] = 'This plugin requires a minmum PHP Version of ' . MOP_AI_INDEXER_MIN_PHP_VERSION . '.';
		if (! self::wp_version_match()) self::$requirement_errors['wp_version_error'] = 'This plugin requires a minmum WordPress Version of ' . MOP_AI_INDEXER_MIN_WP_VERSION . '.';
		return self::$requirement_errors;
	}
}
