<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes;

/**
 * Generates message string
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_Allowed_Html};

/**
 * Generates message string.
 *
 * This class is to generate message string.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Message_Generator {

	/**
	 * The array of messages.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $message_arr    Contains associative array of messages.
	 */
	private array $message_arr = array();

	/**
	 * The output template data associative array.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $template_arr    Contains the output template data associative array.
	 */
	private array $template_arr = array();

	/**
	 * The availability of wp_kses() filtering mechanism.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      bool    $kses_enabled    Contains the availability of wp_kses() filtering mechanism.
	 */
	private bool $kses_enabled = true;

	/**
	 * Initialize the array of messages, output template data associative array
	 * and the availability of wp_kses() filtering mechanism.
	 *
	 * @since    1.0.0
	 * @param    array     $args     Optional. The arguments array.
	 */
	public function __construct(array $args = array()) {

		$this->message_arr  = $args['message_arr'] ?? array();
		$this->template_arr = $args['template_arr'] ?? array('type' => 'plain');
		$this->kses_enabled = $args['kses_enabled'] ?? true;
	}

	/**
	 * Generates a message string from $message_arr.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Allowed_Html
	 * @return   string    The message string. Can be an empty string.
	 */
	public function get_message(): string {

		$message = '';

		if (! empty($this->message_arr) && $this->template_arr['type'] === 'plain') {

			foreach ($this->message_arr as $message_key => $message_value) {
				$message .= $message_value;
				$message .= '<br>';
			}

		} elseif (! empty($this->message_arr) && $this->template_arr['type'] === 'notice') {

			if ($this->template_arr['sub_type'] === 'notice_success') {

				$message .= '<div class="notice notice-success is-dismissible">';
				$message .= '<p>';
				$message .= '<strong>';
				foreach ($this->message_arr as $message_key => $message_value) {
					$message .= $message_value;
					$message .= '<br>';
				}
				$message .= '</strong>';
				$message .= '</p>';
				$message .= '</div>';

			} elseif ($this->template_arr['sub_type'] === 'notice_warning') {

				$message .= '<div class="notice notice-warning is-dismissible">';
				$message .= '<p>';
				$message .= '<strong>';
				foreach ($this->message_arr as $message_key => $message_value) {
					$message .= $message_value;
					$message .= '<br>';
				}
				$message .= '</strong>';
				$message .= '</p>';
				$message .= '</div>';

			} elseif ($this->template_arr['sub_type'] === 'notice_error') {

				$message .= '<div class="notice notice-error is-dismissible">';
				$message .= '<p>';
				$message .= '<strong>';
				foreach ($this->message_arr as $message_key => $message_value) {
					$message .= $message_value;
					$message .= '<br>';
				}
				$message .= '</strong>';
				$message .= '</p>';
				$message .= '</div>';

			} elseif ($this->template_arr['sub_type'] === 'notice_info') {

				$message .= '<div class="notice notice-info is-dismissible">';
				$message .= '<p>';
				$message .= '<strong>';
				foreach ($this->message_arr as $message_key => $message_value) {
					$message .= $message_value;
					$message .= '<br>';
				}
				$message .= '</strong>';
				$message .= '</p>';
				$message .= '</div>';
			}
		}

		if ($this->kses_enabled) {
			return wp_kses($message, Mop_Ai_Indexer_Allowed_Html::get_allowed_html());
		}else{
			return $message;
		}
	}
}
