<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Includes;

/**
 * Fires during plugin activation
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_Allowed_Html, Mop_Ai_Indexer_Defaults, Mop_Ai_Indexer_Cron_Manager};

/**
 * Fires during plugin activation.
 *
 * This class is to execute all necessary actions on plugin activation.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/includes
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Activator {

	/**
	 * Executes all necessary actions on plugin activation.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Cron_Manager
	 * @see      Mop_Ai_Indexer_Defaults
	 * @see      Mop_Ai_Indexer_Requirement_Checker
	 * @see      Mop_Ai_Indexer_Message_Generator
	 * @see      Mop_Ai_Indexer_Allowed_Html
	 */
	public static function activate(): void {

		/**
		 * Check PHP Version and WordPress Version, then deactivate and die
		 * if it doesn't meet minimum requirements.
		 */
		$requirement_errors = Mop_Ai_Indexer_Requirement_Checker::get_requirement_errors();
		if (! empty($requirement_errors)) {
			$args = array('message_arr' => $requirement_errors);
			$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
			deactivate_plugins(MOP_AI_INDEXER_BASENAME);
			wp_die(wp_kses($message_generator_obj->get_message(), Mop_Ai_Indexer_Allowed_Html::get_allowed_html()));
		}

		/**
		 * If the current user is not allowed to activate plugins, then die.
		 */
		if (! current_user_can('activate_plugins')) {
			deactivate_plugins(MOP_AI_INDEXER_BASENAME);
			wp_die('Something is not right.');
		}

		/**
		 * Initialize default options if not already set.
		 */
		if (false === get_option('mop_ai_indexer_iset', false)) {
			add_option('mop_ai_indexer_iset', Mop_Ai_Indexer_Defaults::get_iset_defaults());
		}
		if (false === get_option('mop_ai_indexer_config', false)) {
			add_option('mop_ai_indexer_config', Mop_Ai_Indexer_Defaults::get_config_defaults());
		}

		/**
		 * Ensure the cron event is scheduled based on saved/default frequency.
		 */
		$cron_manager_obj = new Mop_Ai_Indexer_Cron_Manager();
		add_filter('cron_schedules', array($cron_manager_obj, 'register_custom_schedules'));
		Mop_Ai_Indexer_Cron_Manager::sync_schedule_from_settings();
		remove_filter('cron_schedules', array($cron_manager_obj, 'register_custom_schedules'));

		/**
		 * All other function calls to be fired on plugin activation.
		 */
		self::flush_all_rewrite_rules();
	}

	/**
	 * Flushes all rewrite rules by deleting rewrite_rules options.
	 * 
	 * The inbuilt flush_rewrite_rules() function is not used
	 * because it has some technical problems when executing
	 * plugin activation and deactivation.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private static function flush_all_rewrite_rules(): void {

		delete_option('rewrite_rules');
	}
}
