<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin\Logic;

/**
 * Adds MOP AI Indexer per-post controls.
 *
 * This provides a plugin-specific "escape hatch" so site owners can control
 * how individual posts/pages appear in Index output, regardless of what SEO
 * plugins are installed or active.
 *
 * Features:
 * - Checkbox meta box on the post edit screen.
 * - Optional checkbox inside Quick Edit (post list table).
 * - A list-table column that shows the current excluded state.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 * @author     Anjana Hemachandra
 */
if (! defined('ABSPATH')) exit;

/**
 * Manages per-post MOP AI Indexer controls, including meta boxes and list-table integration.
 *
 * This class is to manage per-post MOP AI Indexer controls, including meta boxes and list-table integration..
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Post_Control_Manager {

	/**
	 * Post meta key used to store exclusion state.
	 *
	 * @var string
	 */
	private $meta_key = 'mop_ai_indexer_exclude';

	/**
	 * Returns whether the Post Meta Box feature is enabled in settings.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	private function is_post_meta_box_enabled(): bool {

		$iset = get_option('mop_ai_indexer_iset', array());
		$iset = is_array($iset) ? $iset : array();

		$val = isset($iset['iset_enable_mop_ai_indexer_post_meta_box']) ? (string)$iset['iset_enable_mop_ai_indexer_post_meta_box'] : '';

		return ($val === 'enable-meta-box');
	}

	/**
	 * Returns post types that are included in the index file via Index Configurations.
	 *
	 * @since 1.0.0
	 * @return array Array of post type slugs.
	 */
	private function get_included_post_types(): array {

		$config = get_option('mop_ai_indexer_config', array());
		$config = is_array($config) ? $config : array();

		$post_type_config = isset($config['post_type_config']) && is_array($config['post_type_config']) ? $config['post_type_config'] : array();

		$included = array();

		foreach ($post_type_config as $post_type => $data) {

			if (! is_string($post_type) || $post_type === '') continue;
			if (! is_array($data)) continue;

			$flag = isset($data['include_in_mop_ai_indexer']) ? (string)$data['include_in_mop_ai_indexer'] : '0';
			if ($flag === '1') {
				$included[] = $post_type;
			}
		}

		return $included;
	}

	/**
	 * Register meta box for supported post types.
	 *
	 * @since 1.0.0
	 */
	public function register_meta_box(): void {

		if (! $this->is_post_meta_box_enabled()) return;

		$post_types = $this->get_included_post_types();
		if (empty($post_types)) return;

		$excluded = array('attachment', 'revision', 'nav_menu_item');

		foreach ($post_types as $post_type) {

			if (! is_string($post_type) || $post_type === '') continue;
			if (! post_type_exists($post_type)) continue;

			if (in_array($post_type, $excluded, true)) continue;

			add_meta_box(
				'mop-ai-indexer-post-control-meta-box',
				esc_html__('MOP AI Indexer', 'mop-ai-indexer'),
				array($this, 'render_meta_box'),
				$post_type,
				'side',
				'default'
			);
		}
	}

	/**
	 * Render the meta box UI.
	 *
	 * @since  1.0.0
	 * @param  \WP_Post $post Post object.
	 */
	public function render_meta_box($post): void {

		$post_id = isset($post->ID) ? absint($post->ID) : 0;
		$value = ($post_id > 0) ? get_post_meta($post_id, $this->meta_key, true) : '';
		$checked = ((string)$value === '1') ? 'checked' : '';

		wp_nonce_field('mop_ai_indexer_exclude_meta_box', 'mop_ai_indexer_exclude_meta_nonce');

		echo '<p>';
		echo '<label for="mop-ai-indexer-exclude-checkbox">';
		echo '<input type="checkbox" id="mop-ai-indexer-exclude-checkbox" name="' . esc_attr($this->meta_key) . '" value="1" ' . esc_attr($checked) . ' />';
		echo ' ' . esc_html__('Exclude this content from Index output', 'mop-ai-indexer');
		echo '</label>';
		echo '</p>';

		echo '<p class="description">' . esc_html__('When enabled, this post/page will not appear in the generated llms.txt or llms-full.txt file.', 'mop-ai-indexer') . '</p>';
	}

	/**
	 * Save meta box / quick edit value.
	 *
	 * @since  1.0.0
	 * @param  int $post_id Post ID.
	 */
	public function save_post_control_meta(int $post_id): void {

		$post_id = absint($post_id);
		if ($post_id <= 0) return;

		// Ignore autosaves, revisions, and non-edit contexts.
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
		if (wp_is_post_revision($post_id)) return;

		if (! current_user_can('edit_post', $post_id)) return;

		/**
		 * Nonce checks.
		 *
		 * The meta box and quick edit use separate nonce fields. Accept either.
		 */
		$meta_nonce = isset($_POST['mop_ai_indexer_exclude_meta_nonce']) ? sanitize_text_field(wp_unslash($_POST['mop_ai_indexer_exclude_meta_nonce'])) : '';
		$quick_nonce = isset($_POST['mop_ai_indexer_exclude_quick_nonce']) ? sanitize_text_field(wp_unslash($_POST['mop_ai_indexer_exclude_quick_nonce'])) : '';

		$meta_ok = ($meta_nonce !== '' && wp_verify_nonce($meta_nonce, 'mop_ai_indexer_exclude_meta_box'));
		$quick_ok = ($quick_nonce !== '' && wp_verify_nonce($quick_nonce, 'mop_ai_indexer_exclude_quick_edit'));

		if (! $meta_ok && ! $quick_ok) return;

		$enabled = isset($_POST[$this->meta_key]) ? sanitize_text_field(wp_unslash($_POST[$this->meta_key])) : '';

		if ((string)$enabled === '1') {
			update_post_meta($post_id, $this->meta_key, '1');
		} else {
			delete_post_meta($post_id, $this->meta_key);
		}
	}

	/**
	 * Register list-table column and rendering hooks for all public post types.
	 *
	 * This is executed on admin_init so it can dynamically cover custom post types.
	 *
	 * @since 1.0.0
	 */
	public function register_list_table_hooks(): void {

		if (! $this->is_post_meta_box_enabled()) return;

		$post_types = $this->get_included_post_types();
		if (empty($post_types)) return;

		$excluded = array('attachment', 'revision', 'nav_menu_item');

		foreach ($post_types as $post_type) {

			if (! is_string($post_type) || $post_type === '') continue;

			if (in_array($post_type, $excluded, true)) continue;

			add_filter("manage_edit-{$post_type}_columns", array($this, 'add_list_column'));
			add_action("manage_{$post_type}_posts_custom_column", array($this, 'render_list_column'), 10, 2);
		}
	}

	/**
	 * Adds the "MOP AI Indexer Exclude" column to the post list table.
	 *
	 * @since  1.0.0
	 * @param  array $columns Columns.
	 * @return array
	 */
	public function add_list_column(array $columns): array {

		$new = array();

		/**
		 * Place the column near the title for better visibility, when possible.
		 */
		foreach ($columns as $key => $label) {

			$new[$key] = $label;

			if ($key === 'title') {
				$new[$this->meta_key] = esc_html__('MOP AI Indexer', 'mop-ai-indexer');
			}
		}

		if (! isset($new[$this->meta_key])) {
			$new[$this->meta_key] = esc_html__('MOP AI Indexer', 'mop-ai-indexer');
		}

		return $new;
	}

	/**
	 * Renders the exclusion value in the list-table column.
	 *
	 * @since  1.0.0
	 * @param  string $column  Column key.
	 * @param  int    $post_id Post ID.
	 */
	public function render_list_column(string $column, int $post_id): void {

		if ($column !== $this->meta_key) return;

		$value = get_post_meta($post_id, $this->meta_key, true);
		$is_excluded = ((string)$value === '1');

		echo $is_excluded ? esc_html__('Excluded', 'mop-ai-indexer') : esc_html__('Included', 'mop-ai-indexer');

		/**
		 * Data marker used by Quick Edit script to pre-fill the checkbox state.
		 */
		echo '<span class="mop-ai-indexer-exclude-flag" data-mop-ai-indexer-exclude="' . esc_attr($is_excluded ? '1' : '0') . '" style="display:none;"></span>';
	}

	/**
	 * Renders a checkbox in the Quick Edit panel.
	 *
	 * @since  1.0.0
	 * @param  string $column_name Column key.
	 * @param  string $post_type   Post type.
	 */
	public function render_quick_edit(string $column_name, string $post_type): void {

		if (! $this->is_post_meta_box_enabled()) return;
		if (! in_array($post_type, $this->get_included_post_types(), true)) return;

		if ($column_name !== $this->meta_key) return;

		wp_nonce_field('mop_ai_indexer_exclude_quick_edit', 'mop_ai_indexer_exclude_quick_nonce');

		echo '<fieldset class="inline-edit-col-right">';
		echo '<div class="inline-edit-col">';
		echo '<label class="alignleft">';
		echo '<span class="title">' . esc_html__('MOP AI Indexer', 'mop-ai-indexer') . '</span>';
		echo '<span class="input-text-wrap">';
		echo '<label class="mop-ai-indexer-quick-edit-exclude">';
		echo '<input type="checkbox" name="' . esc_attr($this->meta_key) . '" value="1" />';
		echo ' ' . esc_html__('Exclude from Index output', 'mop-ai-indexer');
		echo '</label>';
		echo '</span>';
		echo '</label>';
		echo '</div>';
		echo '</fieldset>';
	}

	/**
	 * Outputs the Quick Edit JavaScript required to pre-fill the checkbox state.
	 *
	 * This is printed only on edit.php screens via the admin_enqueue_scripts hook.
	 *
	 * @since 1.0.0
	 */
	public function output_quick_edit_js(): void {

		if (! $this->is_post_meta_box_enabled()) return;

		$screen = function_exists('get_current_screen') ? get_current_screen() : null;
		$post_type = ($screen && isset($screen->post_type)) ? (string)$screen->post_type : '';
		if ($post_type === '' || ! in_array($post_type, $this->get_included_post_types(), true)) return;

		global $pagenow;
		if ($pagenow !== 'edit.php') return;

		wp_enqueue_script('inline-edit-post');

		$meta_key_js = wp_json_encode((string)$this->meta_key);

		$js_output = '';
		$js_output .= '(function($){';
		$js_output .= 'if(typeof inlineEditPost === "undefined") return;';
		$js_output .= 'var metaKey = ' . $meta_key_js . ';';
		$js_output .= 'var $wp_inline_edit = inlineEditPost.edit;';
		$js_output .= 'inlineEditPost.edit = function(id){';
		$js_output .= '$wp_inline_edit.apply(this, arguments);';
		$js_output .= 'var post_id = 0;';
		$js_output .= 'if(typeof(id) === "object"){ post_id = parseInt(this.getId(id), 10); }';
		$js_output .= 'if(post_id <= 0) return;';
		$js_output .= 'var $row = $("#post-" + post_id);';
		$js_output .= 'var flag = $row.find(".mop-ai-indexer-exclude-flag").data("mop-ai-indexer-exclude");';
		$js_output .= 'var excluded = (String(flag) === "1");';
		$js_output .= 'var $edit = $("#edit-" + post_id);';
		$js_output .= '$edit.find(\'input[name="\' + metaKey + \'"]\').prop("checked", excluded);';
		$js_output .= '};';
		$js_output .= '})(jQuery);';

		wp_add_inline_script('inline-edit-post', $js_output, 'after');
	}
}
