<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin\Logic;

/**
 * Displays index settings page of the plugin in admin area
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer_Message_Generator, Mop_Ai_Indexer_Endpoint_Router};
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_Allowed_Html, Mop_Ai_Indexer_Defaults, Mop_Ai_Indexer_Cron_Manager, Mop_Ai_Indexer_Cache_Manager, Mop_Ai_Indexer_File_Manager};

/**
 * Displays index settings page of the plugin in admin area.
 *
 * This class is to display index settings page of the plugin in admin area.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Index_Settings_Page {

	/**
	 * Displays index settings page of the plugin in admin area.
	 *
	 * @since   1.0.0
	 * @see     Mop_Ai_Indexer_Defaults
	 */
	public function display_index_settings_page(): void {

		/**
		 * If the current user is not allowed to manage options, then die.
		 */
		if (! current_user_can('manage_options')) wp_die('Something is not right.');

		/**
		 * Handles save index settings form data.
		 */
		if (isset($_POST['iset_save'])) {

			/**
			 * Nonce check.
			 */
			check_admin_referer('mop_ai_indexer_save_iset_action', 'mop_ai_indexer_save_iset_nonce_field');

			$this->save_index_settings();

		/**
		 * Handles save index settings view.
		 */	
		}else{

			$this->display_save_index_settings_view();
		}
	}

	/**
	 * Displays the save index settings view.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @access   private
	 * @see      Mop_Ai_Indexer_Allowed_Html
	 * @see      Mop_Ai_Indexer_Message_Generator
	 */
	private function display_save_index_settings_view(): void {

		/**
		 * Retrieves index settings default values.
		 */
		$iset_defaults = self::get_iset_defaults();

		/**
		 * Retrieves $allowed_html array for wp_kses.
		 */
		$allowed_html = Mop_Ai_Indexer_Allowed_Html::get_allowed_html();

		/**
		 * Starts output variables generation.
		 * 
		 * Starts notice output.
		 */
		$notice_output = '';

		if (isset($_GET['iset-saved'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			$is_iset_saved = isset($_GET['iset-saved']) ? sanitize_text_field(wp_unslash($_GET['iset-saved'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			if ($is_iset_saved === 'true') {

				$message_arr  = array('iset_save_suc' => esc_html__('Index Settings have been saved.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_success');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			} elseif ($is_iset_saved === 'false') {

				$message_arr  = array('iset_save_war' => esc_html__('Index Settings have not been changed.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_warning');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			}else{

				$message_arr  = array('iset_save_err_1' => esc_html__('There has been an unknown error.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();
			}

		} elseif (isset($_GET['form_data'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			$form_data = isset($_GET['form_data']) ? sanitize_text_field(wp_unslash($_GET['form_data'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			/**
			 * Last save index settings view data management logic, if available.
			 */
			if ($form_data === 'available') {

				$session_token = wp_get_session_token();

				if (! empty($session_token)) {

					/**
					 * Retrieves last save index settings view data, if available.
					 */
					$transient_key = 'mop_ai_indexer_save_iset_form_data_' . $session_token;
					$form_data_arr = get_transient($transient_key);

					if (! empty($form_data_arr)) {

						$new_data = $form_data_arr['new_data'];
						$error_data = $form_data_arr['error_data'];

						if (! empty($error_data)) {

							$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
							$args         = array('message_arr' => $error_data, 'template_arr' => $template_arr);
							$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
							$notice_output .= $message_generator_obj->get_message();
						}

						delete_transient($transient_key);

					}else{

						$message_arr  = array('iset_save_err_4' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
						$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
						$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
						$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
						$notice_output .= $message_generator_obj->get_message();
					}

				}else{

					$message_arr  = array('iset_save_err_3' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
					$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
					$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
					$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
					$notice_output .= $message_generator_obj->get_message();
				}

			} elseif ($form_data === 'unavailable') {

				$message_arr  = array('iset_save_err_2' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			}else{

				$notice_output .= '';
			}

		}else{

			$notice_output .= '';
		}

		/**
		 * Retrieves saved mop_ai_indexer_iset data.
		 * or
		 * Handles last save index settings view data.
		 */
		if (! isset($new_data) || empty($new_data)) {
			$mop_ai_indexer_iset = get_option('mop_ai_indexer_iset', $iset_defaults);
			$mop_ai_indexer_iset = (! empty($mop_ai_indexer_iset) && is_array($mop_ai_indexer_iset)) ? $mop_ai_indexer_iset : array();
		}else{
			$mop_ai_indexer_iset = $new_data;
		}

		/**
		 * Starts saved form field outputs.
		 */
		$iset_file_name_format = isset($mop_ai_indexer_iset['iset_file_name_format']) ? $mop_ai_indexer_iset['iset_file_name_format'] : '';
		$iset_file_name_format = Mop_Ai_Indexer_Endpoint_Router::sanitize_file_name((string)$iset_file_name_format);
		$iset_update_frequency = isset($mop_ai_indexer_iset['iset_update_frequency']) ? $mop_ai_indexer_iset['iset_update_frequency'] : '';
		$iset_enable_mop_ai_indexer_post_meta_box = isset($mop_ai_indexer_iset['iset_enable_mop_ai_indexer_post_meta_box']) ? $mop_ai_indexer_iset['iset_enable_mop_ai_indexer_post_meta_box'] : '';
		$iset_respect_seo_config = isset($mop_ai_indexer_iset['iset_respect_seo_config']) ? $mop_ai_indexer_iset['iset_respect_seo_config'] : '';
		$iset_strict_indexability_check = isset($mop_ai_indexer_iset['iset_strict_indexability_check']) ? $mop_ai_indexer_iset['iset_strict_indexability_check'] : '';
		$iset_purge_caches_after_generation_deletion = isset($mop_ai_indexer_iset['iset_purge_caches_after_generation_deletion']) ? $mop_ai_indexer_iset['iset_purge_caches_after_generation_deletion'] : '';

		/**
		 * Template output with output buffering.
		 */
		ob_start();
		require_once MOP_AI_INDEXER_DIR_PATH . 'admin/partials/mop-ai-indexer-index-settings-template.php';
		echo wp_kses(ob_get_clean(), $allowed_html);
	}

	/**
	 * Save index settings.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Cache_Manager
	 * @see      Mop_Ai_Indexer_Cron_Manager
	 * @see      Mop_Ai_Indexer_Endpoint_Router
	 * @see      Mop_Ai_Indexer_File_Manager
	 * @access   private
	 */
	private function save_index_settings(): void {

		/**
		 * Retrieves form data.
		 */
		$iset_file_name_format = isset($_POST['iset_file_name_format']) ? sanitize_text_field(wp_unslash($_POST['iset_file_name_format'])) : '';
		$iset_update_frequency = isset($_POST['iset_update_frequency']) ? sanitize_text_field(wp_unslash($_POST['iset_update_frequency'])) : '';
		$iset_enable_mop_ai_indexer_post_meta_box = isset($_POST['iset_enable_mop_ai_indexer_post_meta_box']) ? sanitize_text_field(wp_unslash($_POST['iset_enable_mop_ai_indexer_post_meta_box'])) : '';
		$iset_respect_seo_config = isset($_POST['iset_respect_seo_config']) ? sanitize_text_field(wp_unslash($_POST['iset_respect_seo_config'])) : '';
		$iset_strict_indexability_check = isset($_POST['iset_strict_indexability_check']) ? sanitize_text_field(wp_unslash($_POST['iset_strict_indexability_check'])) : '';
		$iset_purge_caches_after_generation_deletion = isset($_POST['iset_purge_caches_after_generation_deletion']) ? sanitize_text_field(wp_unslash($_POST['iset_purge_caches_after_generation_deletion'])) : '';
		$iset_back_to_default = isset($_POST['iset_back_to_default']) ? sanitize_text_field(wp_unslash($_POST['iset_back_to_default'])) : '';

		/**
		 * Starts validation.
		 * 
		 * Validates nonce.
		 */
		$nonce = isset($_POST['mop_ai_indexer_save_iset_nonce_field']) ? sanitize_text_field(wp_unslash($_POST['mop_ai_indexer_save_iset_nonce_field'])) : false;
		if (! wp_verify_nonce($nonce, 'mop_ai_indexer_save_iset_action')) wp_die('Something is not right.');

		/**
		 * Validates form fields.
		 */
		$error_data = array();

		if (! ($iset_file_name_format === 'llms.txt'
			|| $iset_file_name_format === 'llms-full.txt')) {
			$error_data['iset_save_file_name_format_err_1'] = esc_html__('Undefined form error in "Update Frequency" field.', 'mop-ai-indexer');
		};

		
		if (! (
			$iset_update_frequency === 'manual'
			|| $iset_update_frequency === 'twice-a-day'
			|| $iset_update_frequency === 'once-a-day'
			|| $iset_update_frequency === 'every-2-days'
			|| $iset_update_frequency === 'every-3-days'
			|| $iset_update_frequency === 'every-4-days'
			|| $iset_update_frequency === 'every-5-days'
			|| $iset_update_frequency === 'every-6-days'
			|| $iset_update_frequency === 'once-a-week'
			|| $iset_update_frequency === 'every-2-weeks'
			|| $iset_update_frequency === 'every-3-weeks'
			|| $iset_update_frequency === 'every-4-weeks'
		)) {
			$error_data['iset_save_update_frequency_err_1'] = esc_html__('Undefined form error in "Update Frequency" field.', 'mop-ai-indexer');
		};

		if (! ($iset_enable_mop_ai_indexer_post_meta_box === 'enable-meta-box'
			|| $iset_enable_mop_ai_indexer_post_meta_box === '')) {
			$error_data['iset_save_enable_mop_ai_indexer_post_meta_box_err_1'] = esc_html__('Form error in "Enable MOP AI Indexer Post Meta Box" field.', 'mop-ai-indexer');
		};


		if (! ($iset_respect_seo_config === 'respect-seo'
			|| $iset_respect_seo_config === '')) {
			$error_data['iset_save_respect_seo_config_err_1'] = esc_html__('Undefined form error in "Respect SEO Configurations" field.', 'mop-ai-indexer');
		};

		if (! ($iset_strict_indexability_check === 'strict-check'
			|| $iset_strict_indexability_check === '')) {
			$error_data['iset_save_strict_indexability_check_err_1'] = esc_html__('Form error in "Strict indexability check (HTTP)" field.', 'mop-ai-indexer');
		};

		if (! ($iset_purge_caches_after_generation_deletion === 'purge-caches'
			|| $iset_purge_caches_after_generation_deletion === '')) {
			$error_data['iset_save_purge_caches_after_generation_deletion_err_1'] = esc_html__('Form error in "Purge caches after generation/deletion" field.', 'mop-ai-indexer');
		};


		/**
		 * Determine whether the file name/format setting is changing.
		 *
		 * When this setting changes, we must:
		 * - delete the previously generated file (stored in plugin uploads directory) to avoid stale content
		 * - flush rewrite rules so endpoint routing stays consistent
		 *
		 * This is done only after a successful save.
		 */
		$old_iset = get_option('mop_ai_indexer_iset', self::get_iset_defaults());
		$old_iset = is_array($old_iset) ? $old_iset : array();

		$old_file_name = isset($old_iset['iset_file_name_format']) ? (string)$old_iset['iset_file_name_format'] : 'llms.txt';
		$old_file_name = Mop_Ai_Indexer_Endpoint_Router::sanitize_file_name($old_file_name);

		$new_file_name = Mop_Ai_Indexer_Endpoint_Router::sanitize_file_name((string)$iset_file_name_format);

		$file_name_changed = ($old_file_name !== '' && $new_file_name !== '' && $old_file_name !== $new_file_name);

		$new_data = array('iset_file_name_format' => $iset_file_name_format,
			'iset_update_frequency' => $iset_update_frequency,
			'iset_enable_mop_ai_indexer_post_meta_box' => $iset_enable_mop_ai_indexer_post_meta_box,
			'iset_respect_seo_config' => $iset_respect_seo_config,
			'iset_strict_indexability_check' => $iset_strict_indexability_check,
			'iset_purge_caches_after_generation_deletion' => $iset_purge_caches_after_generation_deletion);

		$iset_path = admin_url('admin.php?page=mop-ai-indexer-index-settings');

		/**
		 * Form data management logic.
		 */
		if (empty($error_data)) {

			/**
			 * Saves index settings data.
			 */
			if ($iset_back_to_default === 'true') {

				$is_added = update_option('mop_ai_indexer_iset', self::get_iset_defaults());
				Mop_Ai_Indexer_Cron_Manager::sync_schedule_from_settings();
			}else{

				$is_added = update_option('mop_ai_indexer_iset', $new_data);
				Mop_Ai_Indexer_Cron_Manager::sync_schedule_from_settings();
			}

			/**
			 * Redirection logic.
			 */
			if ($is_added) {

				/**
				 * If the file name/format setting has changed, delete the old generated file.
				 *
				 * This keeps the system honest: changing the endpoint name implies the output must be regenerated.
				 */
				if ($file_name_changed) {
					$this->maybe_delete_generated_upload_file($old_file_name);

					/**
					 * Clear last-generated metadata so admin UI reflects the deletion.
					 */
					delete_option('mop_ai_indexer_last_generated');
					delete_option('mop_ai_indexer_last_generated_file');

					/**
					 * Refresh rewrite rules (soft flush) so endpoint routing stays correct.
					 */
					flush_rewrite_rules(false);

					/**
					 * If enabled, purge caches after deletion triggered by a file name/format change.
					 *
					 * Uses the same cache purge routine as the manual "Clear Caches" action.
					 */
					$allow_wp_cache_flush = ! (Mop_Ai_Indexer_File_Manager::is_generation_job_running() || Mop_Ai_Indexer_File_Manager::is_deletion_job_running());
					Mop_Ai_Indexer_Cache_Manager::maybe_purge_after_success('manual_deletion_ajax', $allow_wp_cache_flush);
				}


				$iset_path = add_query_arg('iset-saved', 'true', $iset_path);
				wp_safe_redirect(esc_url_raw($iset_path));
				exit;

			}else{

				$iset_path = add_query_arg('iset-saved', 'false', $iset_path);
				wp_safe_redirect(esc_url_raw($iset_path));
				exit;
			}

		}else{

			/**
			 * Temporarily saves last index settings view data.
			 * Redirection logic.
			 */
			$session_token = wp_get_session_token();

			if (! empty($session_token)) {

				$form_data_arr = array('new_data' => $new_data, 'error_data' => $error_data);
				$transient_key = 'mop_ai_indexer_save_iset_form_data_' . $session_token;
				set_transient($transient_key, $form_data_arr, 10);

				$iset_path = add_query_arg('form_data', 'available', $iset_path);
				wp_safe_redirect(esc_url_raw($iset_path));
				exit;

			}else{

				$iset_path = add_query_arg('form_data', 'unavailable', $iset_path);
				wp_safe_redirect(esc_url_raw($iset_path));
				exit;
			}
		}	
	}

	/**
	 * Provides index settings options defaults.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Defaults
	 * @return   array    An array of index settings options defaults.
	 */
	public static function get_iset_defaults(): array {

		/**
		 * Returning index settings options defaults.
		 */
		return Mop_Ai_Indexer_Defaults::get_iset_defaults();
	}


	/**
	 * Delete a generated index file from the plugin uploads directory.
	 *
	 * The generator stores the final file in plugin uploads directory and writes to a temp file first.
	 * When a user changes the endpoint file name/format, the previous file becomes stale
	 * and must be removed.
	 *
	 * @since  1.0.0
	 * @see    Mop_Ai_Indexer_Endpoint_Router
	 * @param  string $file_name The previously configured endpoint file name.
	 * @return void
	 */
	private function maybe_delete_generated_upload_file(string $file_name): void {

		$file_name = Mop_Ai_Indexer_Endpoint_Router::sanitize_file_name($file_name);
		if ($file_name === '') return;

		$upload_dir = wp_upload_dir();
		$basedir = isset($upload_dir['basedir']) ? (string)$upload_dir['basedir'] : '';
		$basedir = wp_normalize_path($basedir);
		if ($basedir === '') return;

		$plugin_uploads_path = wp_normalize_path(trailingslashit($basedir) . 'mop-ai-indexer/');

		$file_path = wp_normalize_path(trailingslashit($plugin_uploads_path) . $file_name);
		if (is_file($file_path)) {
			@unlink($file_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleanup of plugin-owned generated files in allowlisted uploads directory.
		}

		/**
		 * Also remove a temp file if one exists.
		 *
		 * Example:
		 * - llms-temp.txt
		 */
		$base_name = preg_replace('/\.txt$/', '', $file_name);
		$base_name = $base_name ? $base_name : '';
		if ($base_name !== '') {
			$temp_path = wp_normalize_path(trailingslashit($plugin_uploads_path) . $base_name . '-temp.txt');
			if (is_file($temp_path)) {
				@unlink($temp_path); // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Cleanup of plugin-owned generated files in allowlisted uploads directory.
			}
		}
	}
}
