<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin\Logic;

/**
 * Displays index manager page of the plugin in admin area
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 */

/**
 * If this index is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Import classes from sub-namespaces.
 */
use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer_Message_Generator};
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_File_Manager, Mop_Ai_Indexer_Allowed_Html, Mop_Ai_Indexer_Defaults};

/**
 * Displays index manager page of the plugin in admin area.
 *
 * This class is to display index manager page of the plugin in admin area.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Index_Manager_Page {

	/**
	 * Displays Index Manager page of the plugin in admin area.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Defaults
	 * @return   void
	 */
	public function display_index_manager_page(): void {

		/**
		 * If the current user is not allowed to manage options, then die.
		 */
		if (! current_user_can('manage_options')) wp_die('Something is not right.');

		/**
		 * Handles save index config form data.
		 */
		if (isset($_POST['ic_save'])) {

			/**
			 * Nonce check.
			 */
			check_admin_referer('mop_ai_indexer_save_ic_action', 'mop_ai_indexer_save_ic_nonce_field');

			$this->save_index_config();

		/**
		 * Handles save index config view.
		 */	
		}else{

			$this->display_save_index_config_view();
		}
	}


	/**
	 * Displays the save index config view.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @see      Mop_Ai_Indexer_Defaults
	 * @see      Mop_Ai_Indexer_Allowed_Html
	 * @see      Mop_Ai_Indexer_Message_Generator
	 * @see      Mop_Ai_Indexer_File_Manager
	 * @return   void
	 */
	private function display_save_index_config_view(): void {

		/**
		 * Retrieves public post types in order (page, post, then others A to Z).
		 */
		$post_types_to_index = Mop_Ai_Indexer_Defaults::get_public_post_types_in_order();

		/**
		 * Retrieves $allowed_html array for wp_kses.
		 */
		$allowed_html = Mop_Ai_Indexer_Allowed_Html::get_allowed_html();

		/**
		 * Starts output variables generation.
		 * 
		 * Starts notice output.
		 */
		$notice_output = '';

		if (isset($_GET['ic-saved'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			$is_ic_saved = isset($_GET['ic-saved']) ? sanitize_text_field(wp_unslash($_GET['ic-saved'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			if ($is_ic_saved === 'true') {

				$message_arr  = array('ic_save_suc' => esc_html__('Index configurations have been saved.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_success');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			} elseif ($is_ic_saved === 'false') {

				$message_arr  = array('ic_save_war' => esc_html__('Index configurations have not been changed.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_warning');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			}else{

				$message_arr  = array('ic_save_err_1' => esc_html__('There has been an unknown error.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();
			}

		} elseif (isset($_GET['cache-cleared'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			$cache_cleared = isset($_GET['cache-cleared']) ? sanitize_text_field(wp_unslash($_GET['cache-cleared'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			if ($cache_cleared === 'available') {

				$session_token = wp_get_session_token();

				if (! empty($session_token)) {

					$transient_key = 'mop_ai_indexer_clear_caches_notice_' . $session_token;
					$notice_data = get_transient($transient_key);

					if (is_array($notice_data) && isset($notice_data['template_arr'], $notice_data['message_arr'])) {

						$args = array(
							'message_arr' => (array)$notice_data['message_arr'],
							'template_arr' => (array)$notice_data['template_arr'],
						);
						$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
						$notice_output .= $message_generator_obj->get_message();

					} else {

						$message_arr  = array('cache_clear_err_1' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
						$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
						$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
						$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
						$notice_output .= $message_generator_obj->get_message();
					}

					delete_transient($transient_key);

				} else {

					$message_arr  = array('cache_clear_err_2' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
					$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
					$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
					$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
					$notice_output .= $message_generator_obj->get_message();
				}

			} elseif ($cache_cleared === 'unavailable') {

				$message_arr  = array('cache_clear_err_3' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			} else {

				$notice_output .= '';
			}

		} elseif (isset($_GET['form_data'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			$form_data = isset($_GET['form_data']) ? sanitize_text_field(wp_unslash($_GET['form_data'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag; no state change.

			/**
			 * Last save index config view data management logic, if available.
			 */
			if ($form_data === 'available') {

				$session_token = wp_get_session_token();

				if (! empty($session_token)) {

					/**
					 * Retrieves last save index config view data, if available.
					 */
					$transient_key = 'mop_ai_indexer_save_ic_form_data_' . $session_token;
					$form_data_arr = get_transient($transient_key);

					if (! empty($form_data_arr)) {

						$new_data   = $form_data_arr['new_data'];
						$error_data = $form_data_arr['error_data'];

						if (! empty($error_data)) {

							$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
							$args         = array('message_arr' => $error_data, 'template_arr' => $template_arr);
							$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
							$notice_output .= $message_generator_obj->get_message();
						}

						delete_transient($transient_key);

					}else{

						$message_arr  = array('ic_save_err_4' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
						$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
						$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
						$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
						$notice_output .= $message_generator_obj->get_message();
					}

				}else{

					$message_arr  = array('ic_save_err_3' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
					$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
					$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
					$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
					$notice_output .= $message_generator_obj->get_message();
				}

			} elseif ($form_data === 'unavailable') {

				$message_arr  = array('ic_save_err_2' => esc_html__('Undefined form error.', 'mop-ai-indexer'));
				$template_arr = array('type' => 'notice', 'sub_type' => 'notice_error');
				$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
				$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
				$notice_output .= $message_generator_obj->get_message();

			}else{

				$notice_output .= '';
			}

		}else{

			$notice_output .= '';
		}

		/**
		 * Starts discourage search engines warning output.
		 
		 * Warn if WordPress is set to discourage search engines.
		 * When enabled, post data won’t be added to the index file.
		 */
		$discourage_search_warn_output = '';

		if ((int) get_option('blog_public', 1) === 0) {

			$message_arr  = array(
				'discourage_search_engines_warning' => esc_html__('Disable “Discourage search engines from indexing this site” in Settings → Reading, otherwise post data won’t be added to the index file (llms.txt / llms-full.txt).', 'mop-ai-indexer'),
			);
			$template_arr = array('type' => 'notice', 'sub_type' => 'notice_warning');
			$args         = array('message_arr' => $message_arr, 'template_arr' => $template_arr);
			$message_generator_obj = new Mop_Ai_Indexer_Message_Generator($args);
			$discourage_search_warn_output .= $message_generator_obj->get_message();
		}

		/**
		 * Index file status output.
		 */
		$file_manager_obj = Mop_Ai_Indexer_File_Manager::get_instance();
		$file_status_html = $file_manager_obj->get_file_status_html();

		/**
		 * Generation/deletion logs output (latest set).
		 */
		$generation_logs_payload = $file_manager_obj->get_generation_logs_display_data();
		$generation_logs_title = (isset($generation_logs_payload['logs_title'])) ? (string)$generation_logs_payload['logs_title'] : '';
		$generation_logs_lines = (isset($generation_logs_payload['logs']) && is_array($generation_logs_payload['logs'])) ? $generation_logs_payload['logs'] : array();

		/**
		 * Retrieves saved mop_ai_indexer_config data OR uses last form data.
		 */
		if (! isset($new_data) || empty($new_data)) {
			$defaults         = Mop_Ai_Indexer_Defaults::get_default_index_config($post_types_to_index);
			$mop_ai_indexer_config  = get_option('mop_ai_indexer_config', $defaults);
			$mop_ai_indexer_config  = Mop_Ai_Indexer_Defaults::merge_config_with_defaults($mop_ai_indexer_config, $defaults);
		}else{
			$mop_ai_indexer_config  = $new_data;
		}

		/**
		 * Order post types by 'order_priority_in_index' for display.
		 */
		if (isset($mop_ai_indexer_config['post_type_config']) && is_array($mop_ai_indexer_config['post_type_config'])) {
			$order = array();
			foreach ($post_types_to_index as $slug => $label) {
				$prio = isset($mop_ai_indexer_config['post_type_config'][$slug]['order_priority_in_index']) ? absint($mop_ai_indexer_config['post_type_config'][$slug]['order_priority_in_index']) : 0;
				$order[$slug] = ($prio > 0) ? $prio : 9999;
			}
			asort($order, SORT_NUMERIC);
			$ordered = array();
			foreach ($order as $slug => $prio) {
				$ordered[$slug] = $post_types_to_index[$slug];
			}
			$post_types_to_index = $ordered;
		}

		/**
		 * Template output with output buffering.
		 */
		ob_start();
		require_once MOP_AI_INDEXER_DIR_PATH . 'admin/partials/mop-ai-indexer-index-manager-template.php';
		echo wp_kses(ob_get_clean(), $allowed_html);
	}


	/**
	 * Save index configurations.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @return   void
	 */
	private function save_index_config(): void {

		/**
		 * Retrieves form data.
		 */
		$raw_ic_post = $_POST['ic'] ?? null; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Value is unslashed and sanitized below via map_deep().
		$raw_ic = is_array($raw_ic_post) ? wp_unslash($raw_ic_post) : array();
		$raw_ic = map_deep($raw_ic, 'sanitize_text_field');
		$ic_back_to_default = isset($_POST['ic_back_to_default']) ? sanitize_text_field(wp_unslash($_POST['ic_back_to_default'])) : '';

		/**
		 * Validates nonce.
		 */
		$nonce = isset($_POST['mop_ai_indexer_save_ic_nonce_field']) ? sanitize_text_field(wp_unslash($_POST['mop_ai_indexer_save_ic_nonce_field'])) : false;
		if (! wp_verify_nonce($nonce, 'mop_ai_indexer_save_ic_action')) wp_die('Something is not right.');

		/**
		 * Validates form fields.
		 */
		$error_data = array();

		$post_types = self::get_public_post_types_in_order();

		$clean_ic = array('post_type_config' => array());

		foreach ($post_types as $slug => $label) {

			$numbers_regex = '/^[0-9]+$/';

			$pt_raw_ic = isset($raw_ic[$slug]) ? $raw_ic[$slug] : array();

			$include_in_mop_ai_indexer = isset($pt_raw_ic['include_in_mop_ai_indexer']) ? $pt_raw_ic['include_in_mop_ai_indexer'] : '0';
			if (! ($include_in_mop_ai_indexer === '1' || $include_in_mop_ai_indexer === '0')) {
				$error_data['ic_save_include_in_mop_ai_indexer_err_1'] = esc_html__('Undefined form error in "Include in index file" field.', 'mop-ai-indexer');
			}

			$order_priority_in_index = isset($pt_raw_ic['order_priority_in_index']) ? $pt_raw_ic['order_priority_in_index'] : '0';
			if ($order_priority_in_index !== '' && ! preg_match($numbers_regex, $order_priority_in_index)) {
				$error_data['ic_save_order_priority_in_index_err_1'] = esc_html__('Only numbers (0–9) are allowed in the "Order priority in index file" field.', 'mop-ai-indexer');
			}
			$order_priority_in_index = absint($order_priority_in_index);
			if ($order_priority_in_index > 1000) {
				$error_data['ic_save_order_priority_in_index_err_2'] = esc_html__('The "Order priority in index file" field value must not be greater than 1000.', 'mop-ai-indexer');
			}
			if ($order_priority_in_index <= 0) {
				$error_data['ic_save_order_priority_in_index_err_3'] = esc_html__('The "Order priority in index file" field value must be greater than 0.', 'mop-ai-indexer');
			}

			$num_of_latest_posts = isset($pt_raw_ic['num_of_latest_posts']) ? $pt_raw_ic['num_of_latest_posts'] : '0';
			if ($num_of_latest_posts !== '' && ! preg_match($numbers_regex, $num_of_latest_posts)) {
				$error_data['ic_save_num_of_latest_posts_err_1'] = esc_html__('Only numbers (0–9) are allowed in the "Number of latest posts to include" field.', 'mop-ai-indexer');
			}
			$num_of_latest_posts = absint($num_of_latest_posts);
			if ($num_of_latest_posts > 1000) {
				$error_data['ic_save_num_of_latest_posts_err_2'] = esc_html__('The "Number of latest posts to include" field value must not be greater than 1000.', 'mop-ai-indexer');
			}
			if ($num_of_latest_posts <= 0) $num_of_latest_posts = 100;

			$max_content_length = isset($pt_raw_ic['max_content_length']) ? $pt_raw_ic['max_content_length'] : '0';
			if ($max_content_length !== '' && ! preg_match($numbers_regex, $max_content_length)) {
				$error_data['ic_save_max_content_length_err_1'] = esc_html__('Only numbers (0–9) are allowed in the "Max content length per post (words)" field.', 'mop-ai-indexer');
			}
			$max_content_length = absint($max_content_length);
			if ($max_content_length > 10000) {
				$error_data['ic_save_max_content_length_err_2'] = esc_html__('The "Max content length per post (words)" field value must not be greater than 10000.', 'mop-ai-indexer');
			}
			
			$include_meta_info = isset($pt_raw_ic['include_meta_info']) ? $pt_raw_ic['include_meta_info'] : '0';
			if (! ($include_meta_info === '1' || $include_meta_info === '0')) {
				$error_data['ic_save_include_meta_info_err_1'] = esc_html__('Undefined form error in "Include meta info" field.', 'mop-ai-indexer');
			}

			$include_exc_meta_desc = isset($pt_raw_ic['include_exc_meta_desc']) ? $pt_raw_ic['include_exc_meta_desc'] : '0';
			if (! ($include_exc_meta_desc === '1' || $include_exc_meta_desc === '0')) {
				$error_data['ic_save_include_exc_meta_desc_err_1'] = esc_html__('Undefined form error in "Include post excerpts or meta descriptions" field.', 'mop-ai-indexer');
			}

			$include_taxonomies = isset($pt_raw_ic['include_taxonomies']) ? $pt_raw_ic['include_taxonomies'] : '0';
			if (! ($include_taxonomies === '1' || $include_taxonomies === '0')) {
				$error_data['ic_save_include_taxonomies_err_1'] = esc_html__('Undefined form error in "	Include taxonomies" field.', 'mop-ai-indexer');
			}

			$include_custom_fields = isset($pt_raw_ic['include_custom_fields']) ? $pt_raw_ic['include_custom_fields'] : '0';
			if (! ($include_custom_fields === '1' || $include_custom_fields === '0')) {
				$error_data['ic_save_include_custom_fields_err_1'] = esc_html__('Undefined form error in "Include custom fields" field.', 'mop-ai-indexer');
			}

			/**
			 * Build cleaned config per post type.
			 */
			$clean_ic['post_type_config'][$slug] = array(
				'include_in_mop_ai_indexer'     => ($include_in_mop_ai_indexer === '1') ? '1' : '0',
				'order_priority_in_index' => (string)$order_priority_in_index,
				'num_of_latest_posts'     => (string)$num_of_latest_posts,
				'max_content_length'      => (string)$max_content_length,
				'include_meta_info'       => ($include_meta_info === '1') ? '1' : '0',
				'include_exc_meta_desc'   => ($include_exc_meta_desc === '1') ? '1' : '0',
				'include_taxonomies'      => ($include_taxonomies === '1') ? '1' : '0',
				'include_custom_fields'  => ($include_custom_fields === '1') ? '1' : '0'
			);
		}

		/**
		 * Ensure $new_data is always defined for error path storage.
		 */
		$new_data = $clean_ic;

		/**
		 * Validate unique priorities and normalize to 1..N (preserve order) when provided.
		 */
		$prio_seen  = array();
		$dupe_found = false;
		$prio_pairs = array();

		foreach ($clean_ic['post_type_config'] as $slug_k => $cfg_v) {
			$prio = isset($cfg_v['order_priority_in_index']) ? absint($cfg_v['order_priority_in_index']) : 0;
			if ($prio > 0) {
				if (isset($prio_seen[$prio])) $dupe_found = true;
				$prio_seen[$prio] = true;
				$prio_pairs[] = array('slug' => $slug_k, 'prio' => $prio);
			}else{
				$prio_pairs[] = array('slug' => $slug_k, 'prio' => 9999);
			}
		}

		if ($dupe_found) {
			$error_data['ic_save_order_priority_in_index_err_4'] = esc_html__('Each post type must have a unique "Order priority in index file" value.', 'mop-ai-indexer');
		}

		/**
		 * If no errors so far, normalize priorities to 1..N based on current order.
		 */
		if (empty($error_data)) {
			usort($prio_pairs, function($a, $b){ return $a['prio'] <=> $b['prio']; });
			$rank = 1;
			foreach ($prio_pairs as $row) {
				$clean_ic['post_type_config'][$row['slug']]['order_priority_in_index'] = (string)$rank;
				$rank++;
			}
			$new_data = $clean_ic;
		}

		$ic_path = admin_url('admin.php?page=mop-ai-indexer');

		/**
		 * Form data management logic.
		 */
		if (empty($error_data)) {

			/**
			 * Saves index config data.
			 */
			if ($ic_back_to_default === 'true') {

				$is_added = update_option('mop_ai_indexer_config', self::get_default_index_config($post_types));

			}else{

				$is_added = update_option('mop_ai_indexer_config', $new_data);
			}

			/**
			 * Redirection logic.
			 */
			if ($is_added) {

				$ic_path = add_query_arg('ic-saved', 'true', $ic_path);
				wp_safe_redirect(esc_url_raw($ic_path));
				exit;

			}else{

				$ic_path = add_query_arg('ic-saved', 'false', $ic_path);
				wp_safe_redirect(esc_url_raw($ic_path));
				exit;
			}

		}else{

			/**
			 * Temporarily saves last index config view data.
			 * Redirection logic.
			 */
			$session_token = wp_get_session_token();

			if (! empty($session_token)) {

				$form_data_arr = array('new_data' => $new_data, 'error_data' => $error_data);
				$transient_key = 'mop_ai_indexer_save_ic_form_data_' . $session_token;
				set_transient($transient_key, $form_data_arr, 10);

				$ic_path = add_query_arg('form_data', 'available', $ic_path);
				wp_safe_redirect(esc_url_raw($ic_path));
				exit;

			}else{

				$ic_path = add_query_arg('form_data', 'unavailable', $ic_path);
				wp_safe_redirect(esc_url_raw($ic_path));
				exit;
			}
		}
	}


	/**
	 * Gets the default index configuration array for the provided post types.
	 * This delegates default configuration generation to Mop_Ai_Indexer_Defaults.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Defaults
	 * @param    array $post_types Post type data used for default configuration.
	 * @return   array Default index configuration array.
	 */

	public static function get_default_index_config(array $post_types): array {
		return Mop_Ai_Indexer_Defaults::get_default_index_config($post_types);

	}


	/**
	 * Merges a saved index configuration with defaults.
	 * Ensures all required keys and post types exist, even on fresh installs.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @see      Mop_Ai_Indexer_Defaults
	 * @param    array $saved    Saved configuration array.
	 * @param    array $defaults Default configuration array.
	 * @return   array Merged configuration array.
	 */

	private static function merge_config_with_defaults($saved, array $defaults): array {

		return Mop_Ai_Indexer_Defaults::merge_config_with_defaults(
			(is_array($saved) ? $saved : array()),
			(is_array($defaults) ? $defaults : array())
		);
	}


	/**
	 * Gets public post types in the preferred display order.
	 * The order is: page, post, then remaining public post types A–Z.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @see      Mop_Ai_Indexer_Defaults
	 * @return   array Post type map used to render the Index Manager cards.
	 */

	private static function get_public_post_types_in_order(): array {
		return Mop_Ai_Indexer_Defaults::get_public_post_types_in_order();

	}
}
