<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin\Logic;

/**
 * Admin actions for cache clearing.
 *
 * @since 1.0.0
 * @package Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 */

if (! defined('ABSPATH')) exit;

use Mop_Ai_Indexer\Includes\{Mop_Ai_Indexer_Message_Generator};
use Mop_Ai_Indexer\Includes\Logic\{Mop_Ai_Indexer_Cache_Manager, Mop_Ai_Indexer_File_Manager};

/**
 * Handles admin actions for cache purging and related notices.
 *
 * This class is to handle admin actions for cache purging and related notices..
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin/logic
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Cache_Admin_Actions {

	/**
	 * Handle the manual "Clear Caches" toolbar action.
	 *
	 * - Always best-effort.
	 * - Reloads the Index Manager page with a traditional WP admin notice.
	 *
	 * @since    1.0.0
	 * @see      Mop_Ai_Indexer_Cache_Manager
	 * @see      Mop_Ai_Indexer_File_Manager
	 * @return   void
	 */
	public function handle_clear_caches_action(): void {

		if (! current_user_can('manage_options')) {
			wp_die('Something is not right.');
		}

		$nonce = isset($_POST['mop_ai_indexer_clear_caches_nonce_field']) ? sanitize_text_field(wp_unslash($_POST['mop_ai_indexer_clear_caches_nonce_field'])) : false;
		if (! wp_verify_nonce($nonce, 'mop_ai_indexer_clear_caches_action')) {
			wp_die('Something is not right.');
		}

		$allow_wp_cache_flush = true;

		/**
		 * IMPORTANT:
		 * wp_cache_flush() can clear transients, which can interfere with an active job.
		 * Therefore, skip wp_cache_flush() if a generation/deletion job is currently running.
		 */
		if (Mop_Ai_Indexer_File_Manager::is_generation_job_running() || Mop_Ai_Indexer_File_Manager::is_deletion_job_running()) {
			$allow_wp_cache_flush = false;
		}

		$result = Mop_Ai_Indexer_Cache_Manager::purge_all('manual_clear_caches_admin_action', $allow_wp_cache_flush);

		$notice = $this->build_notice_payload($result);

		$redirect = admin_url('admin.php?page=mop-ai-indexer');
		$session_token = wp_get_session_token();

		if (! empty($session_token)) {

			$transient_key = 'mop_ai_indexer_clear_caches_notice_' . $session_token;
			set_transient($transient_key, $notice, 20);

			$redirect = add_query_arg('cache-cleared', 'available', $redirect);
			wp_safe_redirect(esc_url_raw($redirect));
			exit;
		}

		$redirect = add_query_arg('cache-cleared', 'unavailable', $redirect);
		wp_safe_redirect(esc_url_raw($redirect));
		exit;
	}

	/**
	 * Build a notice payload for transient storage.
	 *
	 * @since 1.0.0
	 * @param array $result Purge result.
	 * @return array
	 */
	private function build_notice_payload(array $result): array {

		$purged = (isset($result['purged']) && is_array($result['purged'])) ? $result['purged'] : array();
		$warnings = (isset($result['warnings']) && is_array($result['warnings'])) ? $result['warnings'] : array();
		$errors = (isset($result['errors']) && is_array($result['errors'])) ? $result['errors'] : array();

		$sub_type = 'notice_success';
		$message_arr = array();

		if (! empty($errors)) {
			$sub_type = 'notice_error';
			$i = 1;
			foreach ($errors as $msg) {
				$message_arr['cache_clear_err_' . $i] = sanitize_text_field((string)$msg);
				$i++;
			}
		} elseif (empty($purged)) {
			$sub_type = 'notice_warning';
			$message_arr['cache_clear_war_1'] = esc_html__('No supported cache plugins were detected. No caches were cleared.', 'mop-ai-indexer');
		} else {
			$sub_type = 'notice_success';
			$message_arr['cache_clear_suc_1'] = esc_html__('Caches have been cleared.', 'mop-ai-indexer');
			$message_arr['cache_clear_suc_2'] = esc_html__('Purged:', 'mop-ai-indexer') . ' ' . implode(', ', array_map('sanitize_text_field', $purged));
		}

		if (! empty($warnings)) {
			$i = 1;
			foreach ($warnings as $msg) {
				$message_arr['cache_clear_war_' . (10 + $i)] = sanitize_text_field((string)$msg);
				$i++;
			}
		}

		$message_arr['cache_clear_note_1'] = esc_html__('If your site is behind a CDN (e.g., Cloudflare), you may need to purge CDN caches manually.', 'mop-ai-indexer');

		return array(
			'template_arr' => array('type' => 'notice', 'sub_type' => $sub_type),
			'message_arr' => $message_arr,
		);
	}
}
