<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin;

/**
 * Manages admin notice filtering for MOP AI Indexer pages
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.1
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * Manages admin notice filtering for MOP AI Indexer pages.
 *
 * Reduces admin notice clutter from other plugins/themes on MOP AI Indexer admin pages
 * so there is enough visible top space for MOP AI Indexer plugin controls and notices,
 * including the Generate Index File button, Delete Index File button, and Clear Caches
 * button. The filtering is scoped to MOP AI Indexer plugin pages only.
 *
 * @since      1.0.1
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Admin_Notice_Manager {

	/**
	 * Filters admin notices on MOP AI Indexer pages only.
	 *
	 * This removes notices added by other sources on MOP AI Indexer admin pages so the
	 * top area stays clear for MOP AI Indexer notices and action controls (Generate Index
	 * File, Delete Index File, and Clear Caches). Plugin-specific notices can be added
	 * back here later in a controlled order.
	 *
	 * @since    1.0.1
	 */
	public function notice_filter_for_mop_ai_indexer(): void {

		if (! self::is_plugin_page()) return;

		remove_all_actions('network_admin_notices');
		remove_all_actions('admin_notices');
		remove_all_actions('user_admin_notices');
		remove_all_actions('all_admin_notices');

		/**
		 * Placeholder: Add MOP AI Indexer dedicated admin notices here when notice
		 * callbacks are introduced for plugin pages.
		 */
	}

	/**
	 * Checks if the current admin page belongs to MOP AI Indexer.
	 *
	 * Uses the admin page slug prefix check so notice filtering only affects MOP AI
	 * Indexer pages and does not impact notices on other WordPress admin screens.
	 *
	 * @since    1.0.1
	 * @return   bool    True if the current page is a MOP AI Indexer admin page.
	 */
	// phpcs:disable WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	// Reason: Simple admin page parameter check only. No data is modified or stored here.
	public static function is_plugin_page(): bool {

		if (! is_admin() || ! isset($_REQUEST['page'])) return false;

		return 0 === strpos((string) wp_unslash($_REQUEST['page']), 'mop-ai-indexer');
	}
	// phpcs:enable WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
}
