<?php
declare(strict_types=1);
namespace Mop_Ai_Indexer\Admin;

/**
 * The admin-area enqueue functionality of the plugin
 *
 * @link       https://ministryofplugins.com/anjana-hemachandra
 * @since      1.0.0
 *
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin
 */

/**
 * If this file is called directly, then exit.
 */
if (! defined('ABSPATH')) exit;

/**
 * The admin-area enqueue functionality of the plugin.
 *
 * Defines the plugin slug, version, and hooks to
 * enqueue the admin-area stylesheet and JavaScript.
 *
 * @since      1.0.0
 * @package    Mop_Ai_Indexer
 * @subpackage Mop_Ai_Indexer/admin
 * @author     Anjana Hemachandra
 */
class Mop_Ai_Indexer_Admin_Enqueue {

	/**
	 * The slug of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_slug    The slug of this plugin.
	 */
	private string $plugin_slug;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private string $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_slug       The slug of this plugin.
	 * @param    string    $version           The version of this plugin.
	 */
	public function __construct(string $plugin_slug, string $version) {

		$this->plugin_slug = $plugin_slug;
		$this->version = $version;
	}

	/**
	 * Register the Stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles(string $hook_suffix = ''): void {

		wp_enqueue_style($this->plugin_slug . '-admin', MOP_AI_INDEXER_DIR_URL . 'admin/css/mop-ai-indexer-admin.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScripts for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts(string $hook_suffix = ''): void {

		wp_enqueue_script($this->plugin_slug . '-admin', MOP_AI_INDEXER_DIR_URL . 'admin/js/mop-ai-indexer-admin.js', array('jquery'), $this->version, false);

		wp_localize_script($this->plugin_slug . '-admin', 'mop_ai_indexer_admin_data_obj', array(
			'ajaxUrl' => admin_url('admin-ajax.php'),
			'nonce' => wp_create_nonce('mop_ai_indexer_file_manager_admin_nonce'),
			'i18n' => array(
				'generating' => esc_html__('Generating...', 'mop-ai-indexer'),
				'deleting' => esc_html__('Deleting...', 'mop-ai-indexer'),
				'no_generation_logs' => esc_html__('No generation logs available yet.', 'mop-ai-indexer'),
				'error' => esc_html__('An error occurred.', 'mop-ai-indexer'),
			),
		));
	}
}