<?php
/**
 * Plugin Name:       Monzur Mailer – Dashboard Email
 * Plugin URI:        https://wordpress.org/plugins/monzurmailer
 * Description:       Compose and send styled emails with WooCommerce support directly from your WordPress dashboard. It offers reusable templates, SMTP support, making it ideal for newsletters and customer communication.
 * Version:           1.3.2
 * Author:            Monzur Rahman
 * Author URI:        https://profiles.wordpress.org/monzur/
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       monzurmailer
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

define( 'MONZURMAILER_VERSION', '1.3.2' );
define( 'MONZURMAILER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'MONZURMAILER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Main MonzurMailer Class.
 *
 * @since 1.0.0
 */
final class MonzurMailer {

    /**
     * The single instance of the class.
     * @var MonzurMailer
     */
    private static $instance;

    /**
     * Holds temporary data for email callbacks.
     * @var array
     */
    private $current_email_data = [];

    /**
     * Main MonzurMailer Instance.
     * @since 1.0.0
     * @return MonzurMailer Main instance.
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     * @since 1.0.0
     */
    private function __construct() {
        add_action( 'plugins_loaded', [ $this, 'includes' ] );
        add_action( 'admin_menu', [ $this, 'admin_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'admin_enqueue_scripts' ] );
        add_action( 'wp_ajax_monzurmailer_handle_action', [ $this, 'ajax_action_handler' ] );
        add_action( 'phpmailer_init', [ $this, 'configure_phpmailer' ] );
        add_filter( 'wp_mail_from', [ $this, 'force_from_email' ] );
        add_filter( 'wp_mail_from_name', [ $this, 'force_from_name' ] );
    }

    /**
     * Include required plugin files.
     * @since 1.0.0
     */
    public function includes() {
        require_once MONZURMAILER_PLUGIN_DIR . 'includes/helpers.php';
        require_once MONZURMAILER_PLUGIN_DIR . 'admin/admin-page.php';
        require_once MONZURMAILER_PLUGIN_DIR . 'templates/email-template.php';
    }

    /**
     * Add the admin menu page.
     * @since 1.0.0
     */
    public function admin_menu() {
        add_menu_page(
            __( 'Monzur Mailer', 'monzurmailer' ),
            __( 'Monzur Mailer', 'monzurmailer' ),
            'manage_options',
            'monzurmailer',
            'monzurmailer_render_admin_page',
            'dashicons-email-alt2',
            26
        );
    }

    /**
     * Enqueue scripts and styles for the admin page.
     * @since 1.0.0
     * @param string $hook The current admin page hook.
     */
    public function admin_enqueue_scripts( $hook ) {
        if ( 'toplevel_page_monzurmailer' !== $hook ) {
            return;
        }

        wp_enqueue_style( 'monzurmailer-admin-style', MONZURMAILER_PLUGIN_URL . 'assets/css/admin-style.css', [], MONZURMAILER_VERSION );
        wp_enqueue_script( 'monzurmailer-admin-js', MONZURMAILER_PLUGIN_URL . 'assets/js/admin.js', [ 'jquery' ], MONZURMAILER_VERSION, true );

        wp_localize_script(
            'monzurmailer-admin-js',
            'monzurmailer_data',
            [
                'ajax_url'   => admin_url( 'admin-ajax.php' ),
                'nonce'      => wp_create_nonce( 'monzurmailer_ajax_nonce' ),
                'text'       => [
                    'processing'     => __( 'Processing...', 'monzurmailer' ),
                    'confirm_delete' => __( 'Are you sure you want to delete this template?', 'monzurmailer' ),
                ],
                'templates'  => array_values( get_option( 'monzurmailer_email_templates', [] ) ),
            ]
        );
    }

    public function force_from_email( $email ) {
        return sanitize_email( get_option( 'monzurmailer_from_email', get_option( 'admin_email' ) ) );
    }

    public function force_from_name( $name ) {
        return sanitize_text_field( get_option( 'monzurmailer_from_name', get_bloginfo( 'name' ) ) );
    }

    public function configure_phpmailer( $phpmailer ) {
        $from_email = get_option( 'monzurmailer_from_email', get_option( 'admin_email' ) );
        $from_name  = get_option( 'monzurmailer_from_name', get_bloginfo( 'name' ) );
        $phpmailer->setFrom( sanitize_email( $from_email ), sanitize_text_field( $from_name ) );
        $phpmailer->Sender = sanitize_email( $from_email );
        $server_name = isset( $_SERVER['SERVER_NAME'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_NAME'] ) ) : 'localhost';
        $phpmailer->MessageID = '<' . md5( uniqid( time() ) ) . '@' . $server_name . '>';
        $phpmailer->MessageDate = gmdate( 'D, d M Y H:i:s O' );

        if ( get_option( 'monzurmailer_smtp_enable', 0 ) ) {
            $phpmailer->isSMTP();
            $phpmailer->Host       = get_option( 'monzurmailer_smtp_host', '' );
            $phpmailer->Port       = absint( get_option( 'monzurmailer_smtp_port', 587 ) );
            $phpmailer->SMTPAuth   = true;
            $phpmailer->Username   = get_option( 'monzurmailer_smtp_username', '' );
            $phpmailer->Password   = get_option( 'monzurmailer_smtp_password', '' );
            $phpmailer->SMTPSecure = get_option( 'monzurmailer_smtp_encryption', 'tls' );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
                $phpmailer->SMTPDebug  = 2;
                $phpmailer->Debugoutput = function ( $str, $level ) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log( "Monzur Mailer SMTP Debug [$level]: $str" );
                };
            } else {
                $phpmailer->SMTPDebug = 0;
            }
        } else {
            $phpmailer->isMail();
        }
    }

    public function ajax_action_handler() {
        try {
            if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'monzurmailer_ajax_nonce' ) || ! current_user_can( 'manage_options' ) ) {
                throw new Exception( __( 'Permission denied.', 'monzurmailer' ) );
            }
            $form_action = isset( $_POST['form_action'] ) ? sanitize_key( $_POST['form_action'] ) : '';
            $data        = $_POST;
            $response    = [ 'success' => false, 'message' => __( 'Invalid form action.', 'monzurmailer' ) ];
            switch ( $form_action ) {
                case 'send_email':         $response = $this->handle_composer_form( $data ); break;
                case 'save_settings':      $response = $this->handle_settings_form( $data ); break;
                case 'send_test_email':    $response = $this->handle_test_email_form( $data ); break;
                case 'save_template':      $response = $this->handle_save_template_form( $data ); break;
                case 'delete_template':    $response = $this->handle_delete_template_form( $data ); break;
                case 'clear_logs':         $response = $this->handle_clear_logs_action(); break;
                case 'save_smtp_settings': $response = $this->handle_smtp_settings_form( $data ); break;
            }
            if ( $response['success'] ) {
                wp_send_json_success( $response );
            } else {
                wp_send_json_error( $response );
            }
        } catch ( Exception $e ) {
            // translators: %s: Error message from the server.
            wp_send_json_error( [ 'success' => false, 'message' => sprintf( __( 'Unexpected server error: %s', 'monzurmailer' ), $e->getMessage() ), ], 500 );
        }
    }

    /**
     * Named callback for setting PHPMailer recipient.
     * Replaces an anonymous function to comply with WordPress.org standards.
     * @param PHPMailer $phpmailer The PHPMailer instance.
     */
    public function set_phpmailer_recipient( $phpmailer ) {
        if ( ! empty( $this->current_email_data ) ) {
            $phpmailer->clearAllRecipients();
            $phpmailer->addAddress( sanitize_email( $this->current_email_data['email'] ) );
            $phpmailer->setFrom( sanitize_email( $this->current_email_data['from_email'] ), sanitize_text_field( $this->current_email_data['from_name'] ) );
            $phpmailer->Sender = sanitize_email( $this->current_email_data['from_email'] );
        }
    }

    /**
     * Named callback for logging mail failures.
     * Replaces an anonymous function to comply with WordPress.org standards.
     * @param WP_Error $wp_error The error object.
     */
    public function log_wp_mail_failure( $wp_error ) {
        if ( ! empty( $this->current_email_data ) ) {
            $error_messages =& $this->current_email_data['error_messages_ref'];
            $email          = $this->current_email_data['email'];
            // translators: %1$s: Recipient email address, %2$s: The specific error message from the mail server.
            $error_message_format = __( 'Failed to send to %1$s: %2$s', 'monzurmailer' );
            $error_messages[] = sprintf( $error_message_format, esc_html( $email ), $wp_error->get_error_message() );
        }
    }

    private function handle_composer_form( $data ) {
        try {
            $recipient_type   = isset( $data['monzurmailer_recipient_type'] ) ? sanitize_key( $data['monzurmailer_recipient_type'] ) : 'manual';
            $manual_to        = isset( $data['monzurmailer_to_manual'] ) ? sanitize_textarea_field( wp_unslash( $data['monzurmailer_to_manual'] ) ) : '';
            $wp_role          = isset( $data['monzurmailer_wp_role'] ) ? sanitize_key( $data['monzurmailer_wp_role'] ) : '';
            $subject          = isset( $data['monzurmailer_subject'] ) ? sanitize_text_field( wp_unslash( $data['monzurmailer_subject'] ) ) : '';
            $message          = isset( $data['monzurmailer_message'] ) ? wp_kses_post( wp_unslash( $data['monzurmailer_message'] ) ) : '';
            $recipient_emails = [];

            switch ( $recipient_type ) {
                case 'manual': $emails = explode( ',', $manual_to ); foreach ( $emails as $email ) { if ( is_email( trim( $email ) ) ) $recipient_emails[] = sanitize_email( trim( $email ) ); } break;
                case 'all_users': $recipient_emails = get_users( [ 'fields' => 'user_email' ] ); break;
                case 'by_role': if ( ! empty( $wp_role ) ) $recipient_emails = get_users( [ 'role' => $wp_role, 'fields' => 'user_email' ] ); break;
                case 'wc_customer': if ( function_exists( 'monzurmailer_is_woocommerce_active' ) && monzurmailer_is_woocommerce_active() ) $recipient_emails = get_users( [ 'role' => 'customer', 'fields' => 'user_email' ] ); break;
                case 'wc_shop_manager': if ( function_exists( 'monzurmailer_is_woocommerce_active' ) && monzurmailer_is_woocommerce_active() ) $recipient_emails = get_users( [ 'role' => 'shop_manager', 'fields' => 'user_email' ] ); break;
            }
            $recipient_emails = array_unique( $recipient_emails );

            if ( empty( $recipient_emails ) || empty( $subject ) || empty( $message ) ) {
                throw new Exception( __( 'Please select valid recipients and fill out all fields.', 'monzurmailer' ) );
            }
            $from_name   = get_option( 'monzurmailer_from_name', get_bloginfo( 'name' ) );
            $from_email  = get_option( 'monzurmailer_from_email', get_option( 'admin_email' ) );
            $footer_note = get_option( 'monzurmailer_footer_note', '' );
            if ( false !== strpos( $from_email, 'noreply' ) ) {
                throw new Exception( __( 'Avoid using "noreply" in the From Email address to improve deliverability. Update in Settings.', 'monzurmailer' ) );
            }
            $html_message = monzurmailer_get_email_template( $subject, $message, $footer_note );

            $server_name = isset( $_SERVER['SERVER_NAME'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_NAME'] ) ) : 'localhost';
            $base_headers = [ 'Content-Type: text/html; charset=UTF-8', 'From: ' . esc_html( $from_name ) . ' <' . sanitize_email( $from_email ) . '>', 'Reply-To: ' . esc_html( $from_name ) . ' <' . sanitize_email( $from_email ) . '>', 'MIME-Version: 1.0', 'X-Mailer: Monzur Mailer/' . MONZURMAILER_VERSION, 'X-Priority: 3 (Normal)', 'List-Unsubscribe: <mailto:' . sanitize_email( $from_email ) . '?subject=unsubscribe>', 'Message-ID: <' . md5( uniqid( time() ) ) . '@' . $server_name . '>', 'Date: ' . gmdate( 'D, d M Y H:i:s O' ), ];
            $sent = true;
            $failed_emails = [];
            $error_messages = [];
            foreach ( $recipient_emails as $email ) {
                $this->current_email_data = [ 'email' => $email, 'from_email' => $from_email, 'from_name' => $from_name, 'error_messages_ref' => &$error_messages, ];
                add_action( 'phpmailer_init', [ $this, 'set_phpmailer_recipient' ] );
                add_filter( 'wp_mail_failed', [ $this, 'log_wp_mail_failure' ] );
                $single_headers = array_merge( $base_headers, [ 'To: ' . sanitize_email( $email ) ] );
                $sent_single = wp_mail( $email, $subject, $html_message, $single_headers );
                remove_action( 'phpmailer_init', [ $this, 'set_phpmailer_recipient' ] );
                remove_filter( 'wp_mail_failed', [ $this, 'log_wp_mail_failure' ] );
                $sent = $sent && $sent_single;
                if ( ! $sent_single ) $failed_emails[] = $email;
            }
            $this->current_email_data = [];

            $recipient_count = count( $recipient_emails );
            // translators: %d: Number of recipients (used for singular or plural form).
            $log_to = sprintf( _n( '%d recipient', '%d recipients', $recipient_count, 'monzurmailer' ), $recipient_count );
            
            // This is now the single point of logging for this action.
            monzurmailer_log_email( $log_to, $subject, $sent, [ 'from_email' => $from_email, 'headers' => $base_headers, 'failed_emails' => $failed_emails, 'to_emails' => $recipient_emails, 'errors' => $error_messages, ] );

            if ( $sent ) {
                // translators: %d: Number of recipients who received the email.
                return [ 'success' => true, 'message' => sprintf( __( 'Email successfully sent to %d recipient(s).', 'monzurmailer' ), $recipient_count ), ];
            }
            // translators: %1$s: Comma-separated list of email addresses that failed, %2$s: Error messages from the mail server.
            $error_string = sprintf( __( 'Email could not be sent to some recipients: %1$s. Errors: %2$s. Check your email settings or contact your hosting provider.', 'monzurmailer' ), implode( ', ', $failed_emails ), implode( '; ', $error_messages ) );
            return [ 'success' => false, 'message' => $error_string ];
        } catch ( Exception $e ) {
            // This catch block is for setup errors (e.g., fields not filled), not for sending failures.
            return [ 'success' => false, 'message' => $e->getMessage(), ];
        }
    }

    private function handle_test_email_form( $data ) {
        try {
            $test_email = isset( $data['monzurmailer_test_email_to'] ) ? sanitize_email( wp_unslash( $data['monzurmailer_test_email_to'] ) ) : '';
            if ( ! is_email( $test_email ) ) throw new Exception( __( 'Invalid email address provided.', 'monzurmailer' ) );
            $subject     = __( 'Monzur Mailer Test Email', 'monzurmailer' );
            $message     = __( 'This is a test email sent from the Monzur Mailer plugin. If you received this, your mail settings are working!', 'monzurmailer' );
            $from_name   = get_option( 'monzurmailer_from_name', get_bloginfo( 'name' ) );
            $from_email  = get_option( 'monzurmailer_from_email', get_option( 'admin_email' ) );
            $footer_note = get_option( 'monzurmailer_footer_note', '' );
            $html_message = monzurmailer_get_email_template( $subject, $message, $footer_note );
            $server_name = isset( $_SERVER['SERVER_NAME'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_NAME'] ) ) : 'localhost';
            $headers = [ 'Content-Type: text/html; charset=UTF-8', 'From: ' . esc_html( $from_name ) . ' <' . sanitize_email( $from_email ) . '>', 'Reply-To: ' . esc_html( $from_name ) . ' <' . sanitize_email( $from_email ) . '>', 'MIME-Version: 1.0', 'X-Mailer: Monzur Mailer/' . MONZURMAILER_VERSION, 'X-Priority: 3 (Normal)', 'List-Unsubscribe: <mailto:' . sanitize_email( $from_email ) . '?subject=unsubscribe>', 'Message-ID: <' . md5( uniqid( time() ) ) . '@' . $server_name . '>', 'Date: ' . gmdate( 'D, d M Y H:i:s O' ), 'To: ' . sanitize_email( $test_email ), ];
            $error_messages = [];
            $this->current_email_data = [ 'email' => $test_email, 'from_email' => $from_email, 'from_name' => $from_name, 'error_messages_ref' => &$error_messages, ];
            add_action( 'phpmailer_init', [ $this, 'set_phpmailer_recipient' ] );
            add_filter( 'wp_mail_failed', [ $this, 'log_wp_mail_failure' ] );
            $sent = wp_mail( $test_email, $subject, $html_message, $headers );
            remove_action( 'phpmailer_init', [ $this, 'set_phpmailer_recipient' ] );
            remove_filter( 'wp_mail_failed', [ $this, 'log_wp_mail_failure' ] );
            $this->current_email_data = [];
            
            // This is now the single point of logging for this action.
            monzurmailer_log_email( $test_email, $subject, $sent, [ 'from_email' => $from_email, 'headers' => $headers, 'failed_emails' => $sent ? [] : [ $test_email ], 'to_emails' => [ $test_email ], 'errors' => $error_messages, ] );
            
            if ( $sent ) {
                // translators: %s: Email address (formatted with HTML strong tags).
                return [ 'success' => true, 'message' => sprintf( __( 'Test email sent to %s.', 'monzurmailer' ), '<strong>' . esc_html( $test_email ) . '</strong>' ), ];
            }
            $default_error = __( 'No specific error reported. Email may have been rejected by recipient server.', 'monzurmailer' );
            // translators: %s: Error messages from the mail server or a default error message.
            $error_string = sprintf( __( 'Failed to send test email. Errors: %s. Check your email settings or contact your hosting provider.', 'monzurmailer' ), implode( '; ', $error_messages ) ?: $default_error );
            return [ 'success' => false, 'message' => $error_string ];
        } catch ( Exception $e ) {
            // This catch block is for setup errors, not for sending failures.
            return [ 'success' => false, 'message' => $e->getMessage(), ];
        }
    }

    private function handle_settings_form( $data ) {
        if ( isset( $data['monzurmailer_from_name'] ) ) update_option( 'monzurmailer_from_name', sanitize_text_field( wp_unslash( $data['monzurmailer_from_name'] ) ) );
        if ( isset( $data['monzurmailer_from_email'] ) ) update_option( 'monzurmailer_from_email', sanitize_email( wp_unslash( $data['monzurmailer_from_email'] ) ) );
        if ( isset( $data['monzurmailer_footer_note'] ) ) update_option( 'monzurmailer_footer_note', sanitize_textarea_field( wp_unslash( $data['monzurmailer_footer_note'] ) ) );
        return [ 'success' => true, 'message' => __( 'Settings saved successfully.', 'monzurmailer' ) ];
    }
    private function handle_smtp_settings_form( $data ) {
        update_option( 'monzurmailer_smtp_enable', isset( $data['monzurmailer_smtp_enable'] ) ? 1 : 0 );
        update_option( 'monzurmailer_smtp_host', sanitize_text_field( wp_unslash( $data['monzurmailer_smtp_host'] ?? '' ) ) );
        update_option( 'monzurmailer_smtp_port', absint( $data['monzurmailer_smtp_port'] ?? 0 ) );
        update_option( 'monzurmailer_smtp_username', sanitize_text_field( wp_unslash( $data['monzurmailer_smtp_username'] ?? '' ) ) );
        update_option( 'monzurmailer_smtp_password', sanitize_text_field( wp_unslash( $data['monzurmailer_smtp_password'] ?? '' ) ) );
        update_option( 'monzurmailer_smtp_encryption', sanitize_key( $data['monzurmailer_smtp_encryption'] ?? 'none' ) );
        return [ 'success' => true, 'message' => __( 'SMTP settings saved successfully.', 'monzurmailer' ) ];
    }
    private function handle_save_template_form( $data ) {
        $email_templates = get_option( 'monzurmailer_email_templates', [] );
        $is_editing      = isset( $data['template_index'] ) && '' !== $data['template_index'];
        $message_key     = $is_editing ? 'template_message_edit' : 'template_message_add';
        $template_data = [ 'title' => isset( $data['template_title'] ) ? sanitize_text_field( wp_unslash( $data['template_title'] ) ) : '', 'subject' => isset( $data['template_subject'] ) ? sanitize_text_field( wp_unslash( $data['template_subject'] ) ) : '', 'message' => isset( $data[ $message_key ] ) ? wp_kses_post( wp_unslash( $data[ $message_key ] ) ) : '', ];
        if ( $is_editing ) {
            $index = absint( $data['template_index'] );
            if ( isset( $email_templates[ $index ] ) ) {
                $email_templates[ $index ] = $template_data;
                $message                   = __( 'Template updated successfully.', 'monzurmailer' );
            } else {
                return [ 'success' => false, 'message' => __( 'Template to edit not found.', 'monzurmailer' ) ];
            }
        } else {
            if ( count( $email_templates ) >= 10 ) return [ 'success' => false, 'message' => __( 'You can only save up to 10 templates.', 'monzurmailer' ) ];
            $email_templates[] = $template_data;
            $message           = __( 'Template added successfully.', 'monzurmailer' );
        }
        update_option( 'monzurmailer_email_templates', $email_templates );
        return [ 'success' => true, 'message' => $message, 'reload' => true ];
    }
    private function handle_delete_template_form( $data ) {
        $index           = isset( $data['template_index'] ) ? absint( $data['template_index'] ) : -1;
        $email_templates = get_option( 'monzurmailer_email_templates', [] );
        if ( isset( $email_templates[ $index ] ) ) {
            array_splice( $email_templates, $index, 1 );
            update_option( 'monzurmailer_email_templates', $email_templates );
            return [ 'success' => true, 'message' => __( 'Template deleted.', 'monzurmailer' ), 'reload' => true ];
        }
        return [ 'success' => false, 'message' => __( 'Template not found.', 'monzurmailer' ) ];
    }
    private function handle_clear_logs_action() {
        delete_option( 'monzurmailer_email_logs' );
        return [ 'success' => true, 'message' => __( 'Email logs have been cleared.', 'monzurmailer' ), 'reload' => true ];
    }
}

function monzurmailer_run_plugin() {
    return MonzurMailer::instance();
}

register_activation_hook( __FILE__, function() {
    $default_options = [ 'monzurmailer_from_name' => get_bloginfo( 'name' ), 'monzurmailer_from_email' => get_option( 'admin_email' ), 'monzurmailer_footer_note' => '', 'monzurmailer_email_templates' => [], 'monzurmailer_email_logs' => [], 'monzurmailer_smtp_enable' => 0, 'monzurmailer_smtp_host' => '', 'monzurmailer_smtp_port' => 587, 'monzurmailer_smtp_username' => '', 'monzurmailer_smtp_password' => '', 'monzurmailer_smtp_encryption' => 'tls', ];
    if ( is_multisite() ) {
        foreach ( get_sites() as $site ) {
            switch_to_blog( $site->blog_id );
            foreach ( $default_options as $option => $value ) {
                if ( false === get_option( $option ) ) update_option( $option, $value );
            }
            restore_current_blog();
        }
    } else {
        foreach ( $default_options as $option => $value ) {
            if ( false === get_option( $option ) ) update_option( $option, $value );
        }
    }
});

monzurmailer_run_plugin();