const { defineConfig } = require("vite");
const legacy = require("@vitejs/plugin-legacy");
const path = require("path");
const fs = require("fs-extra");
const archiver = require("archiver");

// const preact = require('@preact/preset-vite').default;

const deploy = () => {
  const targetDir = path.resolve(__dirname, "../monitori-deployment");
  const zipFile = path.resolve(__dirname, "../monitori-deployment.zip");

  // Clean up the target directory before deploying
  if (fs.existsSync(targetDir)) {
    fs.removeSync(targetDir);
  }

  fs.mkdirSync(targetDir);

  // Copy files and folders, excluding unwanted ones
  fs.copySync(path.resolve(__dirname), targetDir, {
    filter: (src) => {
      // Excludes
      return !src.includes(".git")
        && !src.includes(".DS_Store")
        && !src.includes("deployment")
        && !src.includes("node_modules")
        && !src.includes(".gitignore")
        && !src.includes("package-lock.json")
        && !src.includes("yarn.lock")
        && !src.includes("README.md")
        && !src.includes(".git");
    },
  });

  // Create a zip archive of the deployment folder
  if (fs.existsSync(zipFile)) {
    fs.removeSync(zipFile); // Remove existing zip file
  }

  const output = fs.createWriteStream(zipFile);
  const archive = archiver("zip", { zlib: { level: 9 } });

  output.on("close", () => {
    console.log(`Deployment complete. Files copied to 'monitori-deployment' folder and zipped as 'monitori-deployment.zip'.`);
    console.log(`${archive.pointer()} total bytes written to zip.`);
  });

  archive.on("error", (err) => {
    throw err;
  });

  archive.pipe(output);
  archive.directory(targetDir, false);
  archive.finalize();
};

module.exports = defineConfig({
  // resolve: {
  //   alias: {
  //     react: 'preact/compat',
  //     'react-dom': 'preact/compat',
  //   },
  // },
  optimizeDeps: {
    optimizeDeps: {
      include: [
        "@ariakit/react-core",
        "@ariakit/core",
        "@wordpress/blocks",
        "@wordpress/element",
        "@wordpress/i18n",
      ],
    },
  },
  plugins: [
    legacy({
      targets: ["defaults", "not IE 11"], // Specify legacy browser support
    }),
  ],
  build: {
    rollupOptions: {
      onwarn(warning, warn) {
        // Suppress "Module level directives cause errors when bundled" warnings
        if (warning.code === "MODULE_LEVEL_DIRECTIVE") {
          return;
        }
        warn(warning);
      },
      external: [
        "@ariakit/react-core",
        "@ariakit/core",
        // "@wordpress/blocks",
        // "@wordpress/element",
        // "@wordpress/i18n",
        "@wordpress/editor",
      ],
      input: {
        main: path.resolve(__dirname, "src/js/main.js"),
        admin: path.resolve(__dirname, "src/js/admin.jsx"),
      },
      output: {
        globals: {
          "@wordpress/blocks": "wp.blocks",
          "@wordpress/element": "wp.element",
          "@wordpress/i18n": "wp.i18n",
        },
        entryFileNames: (chunk) => {
          return chunk.name.includes("style") || chunk.name.includes("adminStyle")
            ? "[name].css"
            : "[name].js";
        },
        assetFileNames: "[name][extname]",
        dir: path.resolve(__dirname, "dist"),
      },
    },
  },
  css: {
    preprocessorOptions: {
      scss: {},
    },
  },
  deploy,
});