import React, { createContext, useContext, useState, useEffect } from "react";

const NotificationContext = createContext();

async function addNewNotification(notificationData, nonce) {
  const formData = new FormData();
  formData.append("_wpnonce", nonce);
  formData.append("name", notificationData.name);
  formData.append("notification_type", notificationData.notificationType);
  formData.append("event_type", notificationData.eventType);
  formData.append("metric_name", notificationData.metricName);
  formData.append("change_type", notificationData.changeType);
  formData.append("metric_value", notificationData.metricValue);
  formData.append("time_period", notificationData.timePeriod);
  formData.append("compare_period", notificationData.comparePeriod);
  formData.append(
    "delivery_channels",
    JSON.stringify(notificationData.deliveryChannels)
  );

  const submitForm = await fetch(
    ajaxurl + "?action=monitori_add_notification",
    {
      method: "POST",
      body: formData,
    }
  );

  const response = await submitForm.json();

  if (response.success) {
    console.log("Notification added successfully:", response.message);
  } else {
    console.error("Error adding notification:", response.message);
  }
}

async function updateNotification(id, notificationData, nonce) {
  const formData = new FormData();
  formData.append("_wpnonce", nonce);
  formData.append("id", id);
  formData.append("name", notificationData.name);
  formData.append("notification_type", notificationData.notificationType);
  formData.append("event_type", notificationData.eventType);
  formData.append("metric_name", notificationData.metricName);
  formData.append("change_type", notificationData.changeType);
  formData.append("metric_value", notificationData.metricValue);
  formData.append("time_period", notificationData.timePeriod);
  formData.append("compare_period", notificationData.comparePeriod);
  formData.append(
    "delivery_channels",
    JSON.stringify(notificationData.deliveryChannels)
  );

  const submitForm = await fetch(
    ajaxurl + "?action=monitori_update_notification",
    {
      method: "POST",
      body: formData,
    }
  );

  const response = await submitForm.json();

  if (response.success) {
    console.log("Notification updated successfully:", response.message);
  } else {
    console.error("Error updating notification:", response.message);
  }
}

async function deleteNotification(notificationID, nonce) {
  const formData = new FormData();
  formData.append("_wpnonce", nonce);
  formData.append("id", notificationID);

  const submitForm = await fetch(
    ajaxurl + "?action=monitori_remove_notification",
    {
      method: "POST",
      body: formData,
    }
  );

  const response = await submitForm.json();

  if (response.success) {
    console.log("Notification deleted successfully:", response.message);
  } else {
    console.error("Error deleting notification:", response.message);
  }
}

export function NotificationProvider({ children, nonce }) {
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [postToEdit, setPostToEdit] = useState(null);
  const [notifications, setNotifications] = useState([]);
  const [fieldData, setFieldData] = useState({});
  const [loading, setLoading] = useState(true);

  const openModal = (notificationID = null) => {
    if (notificationID) {
      setPostToEdit(notificationID);
    }

    setIsModalOpen(true);
  };
  const closeModal = () => {
    setIsModalOpen(false);
    setPostToEdit(null);
  };

  const fetchNotifications = async () => {
    try {
      const formData = new FormData();
      formData.append("_wpnonce", nonce);

      const req = await fetch(ajaxurl + "?action=monitori_get_notifications", {
        method: "POST",
        body: formData,
      });
      const result = await req.json();

      if (!result?.success || !result?.data) {
        console.error("Failed to fetch notifications:", result);
        return;
      }
      console.log("refetching, result:", result);
      setNotifications(result.data.notifications || []);
    } catch (error) {
      console.error("Error fetching notifications:", error);
    }
  };

  const fetchFieldData = async () => {
    const formData = new FormData();
    formData.append("_wpnonce", nonce);

    const req = await fetch(ajaxurl + "?action=monitori_get_edit_fields", {
      method: "POST",
      body: formData,
    });
    const res = await req.json();

    if (!res?.success || !res?.data) {
      console.error("Failed to fetch edit fields:", res);
      return;
    }

    setFieldData(res?.data);
  };

  const getNotification = (notificationID) => {
    return notifications.find(
      (notification) => notification.id === notificationID
    );
  };

  useEffect(() => {
    if (typeof ajaxurl === "undefined") {
      console.error("ajaxurl is undefined");
      return;
    }

    if (loading) {
      const getNotificationsAndFields = async () => {
        await fetchNotifications();
        await fetchFieldData();
        setLoading(false);
      };

      getNotificationsAndFields();
    }

    if (!loading) {
      let searchParams = new URLSearchParams(window.location.search);
      const urlAction = searchParams.get("action");

      if (urlAction) {
        if (urlAction === "edit" && notifications.length > 0) {
          const notificationID = searchParams.get("id");
          const notification = getNotification(notificationID);

          if (notification) {
            openModal(notificationID);
            searchParams.delete("id");
            searchParams.delete("action");
          }
        }

        if (urlAction === "new") {
          openModal();
          searchParams.delete("action");
        }

        window.history.replaceState(
          {},
          "",
          window.location.pathname + "?" + searchParams.toString()
        );
      }
    }
  }, [ajaxurl, notifications, loading]);

  return (
    <NotificationContext.Provider
      value={{
        isModalOpen,
        openModal,
        closeModal,
        notifications,
        setNotifications,
        loading,
        setLoading,
        fetchNotifications,
        addNewNotification: (data) => addNewNotification(data, nonce),
        updateNotification: (id, data) => updateNotification(id, data, nonce),
        deleteNotification: (id) => deleteNotification(id, nonce),
        postToEdit,
        getNotification,
        fieldData,
      }}
    >
      {children}
    </NotificationContext.Provider>
  );
}

export function useNotificationContext() {
  return useContext(NotificationContext);
}
