<?php

namespace Monitori;

class Woocommerce_Metric_Integration extends \Monitori\Base_Metric_Integration
{
    public $handler = 'woocommerce';

    public function check_metric($conditions = []): bool
    {
        // Extract conditions
        $metric_name = $conditions['metric_name'] ?? '';
        $change_type = $conditions['change_type'] ?? '';
        $value = $conditions['value'] ?? 0;
        $time_period = $conditions['time_period'] ?? 'monthly';
        $compare_period = $conditions['compare_period'] ?? 'last_period';

        // Correct Method Call
        $current_metric_types = $this->get_metric_types();

        // Correctly Retrieve the Callback
        $current_callback = $current_metric_types[$metric_name]['callback'] ?? false;

        // Check if the callback exists
        if (is_callable([$this, $current_callback])) {
            return $this->{$current_callback}($change_type, $value, $time_period, $compare_period);
        }

        return false;
    }

    // Check number of WooCommerce orders based on the conditions
    private function check_number_of_orders($change_type, $value, $time_period, $compare_period)
    {
        $current_time_period = $this->get_time_period($time_period, 'current');
        $previous_time_period = $this->get_time_period($time_period, $compare_period);

        // Retrieve the number of orders
        $current_orders = $this->get_orders_count($current_time_period);
        $previous_orders = $this->get_orders_count($previous_time_period);

        // Perform the evaluation
        return $this->evaluate_change($current_orders, $previous_orders, $change_type, $value);
    }

    // private function check_number_of_products($change_type, $value, $time_period, $compare_period)
    // {
    //     $current_time_period = $this->get_time_period($time_period, 'current');
    //     $previous_time_period = $this->get_time_period($time_period, $compare_period);

    //     // Retrieve the number of orders
    //     $current_orders = $this->get_products_count($current_time_period);
    //     $previous_orders = $this->get_products_count($previous_time_period);

    //     // Perform the evaluation
    //     return $this->evaluate_change($current_orders, $previous_orders, $change_type, $value);
    // }

    // Check gross sales based on the conditions
    private function check_gross_sales($change_type, $value, $time_period, $compare_period)
    {
        // Retrieve the current gross sales for the given time period
        $current_sales = $this->get_gross_sales($time_period);

        // Retrieve the comparison data (e.g., last period)
        $previous_sales = $this->get_gross_sales($compare_period);

        // Perform the evaluation based on the change type
        return $this->evaluate_change($current_sales, $previous_sales, $change_type, $value);
    }

    private function check_net_sales($change_type, $value, $time_period, $compare_period)
    {
        // Retrieve the current gross sales for the given time period
        $current_sales = $this->get_net_sales($time_period);

        // Retrieve the comparison data (e.g., last period)
        $previous_sales = $this->get_net_sales($compare_period);

        // Perform the evaluation based on the change type
        return $this->evaluate_change($current_sales, $previous_sales, $change_type, $value);
    }

    private function get_orders_count($date_range)
    {

        // Query WooCommerce orders
        $args = [
            'limit'        => -1,             // No limit
            'date_created' => $date_range,    // Use computed date range
            'return'       => 'ids',          // Only get IDs for faster performance
        ];

        $orders = \wc_get_orders($args);

        // Return the number of found orders
        return count($orders);
    }

    // private function get_discounts($date_range)
    // {
    //     $args = [
    //         'limit'        => -1,
    //         'date_created' => $date_range,
    //         'status'       => ['wc-completed', 'wc-processing'],
    //         'return'       => 'objects',
    //     ];

    //     $orders = wc_get_orders($args);

    //     $total_discounts = array_reduce($orders, function ($total, $order) {
    //         return $total + (float) $order->get_discount_total();
    //     }, 0);

    //     return $total_discounts;
    // }

    // private function get_products_count()
    // {
    //     $args = [
    //         'status'    => 'publish',  // Only count published products
    //         'limit'     => -1,         // No limit, count all
    //         'return'    => 'ids',      // Return product IDs for better performance
    //     ];

    //     $products = wc_get_products($args);

    //     return count($products);
    // }

    private function get_time_period($time_period, $compare_period = 'current')
    {
        $date_formats = [
            'daily'   => ['-1 day', 'today'],
            'weekly'  => ['monday last week', 'sunday last week'],
            'monthly' => ['first day of last month', 'last day of last month'],
            'yearly'  => ['first day of January last year', 'last day of December last year'],
        ];

        // Handle "last_year" or adjust based on comparison
        if ($compare_period === 'last_year') {
            return match ($time_period) {
                'daily'   => gmdate('Y-m-d', strtotime('-1 year')),
                'weekly'  => gmdate('Y-m-d', strtotime('monday this week -1 year')) . '...' . gmdate('Y-m-d', strtotime('sunday this week -1 year')),
                'monthly' => gmdate('Y-m-01', strtotime('first day of this month -1 year')) . '...' . gmdate('Y-m-t', strtotime('last day of this month -1 year')),
                'yearly'  => gmdate('Y-01-01', strtotime('-1 year')) . '...' . gmdate('Y-12-31', strtotime('-1 year')),
                default   => gmdate('Y-m-01', strtotime('first day of last month')) . '...' . gmdate('Y-m-t', strtotime('last day of last month')),
            };
        }

        // Handle "last_period" or "current"
        [$start, $end] = match ($compare_period) {
            'last_period' => $date_formats[$time_period] ?? ['first day of last month', 'last day of last month'],
            'current'     => match ($time_period) {
                'daily'   => ['today', 'today'],
                'weekly'  => ['monday this week', 'today'],
                'monthly' => ['first day of this month', 'today'],
                'yearly'  => ['first day of January this year', 'today'],
                default   => ['first day of this month', 'today'],
            },
        };

        return gmdate('Y-m-d', strtotime($start)) . '...' . gmdate('Y-m-d', strtotime($end));
    }

    // Utility function to get the gross sales for a given period
    private function get_gross_sales($date_range)
    {
        $args = [
            'limit'        => -1,              // No limit, get all orders
            'date_created' => $date_range,     // Filter by date range
            'status'       => ['wc-completed', 'wc-processing'],  // Relevant statuses only
            'return'       => 'objects',       // Return full order objects
        ];

        $orders = \wc_get_orders($args);

        $total_sales = array_reduce($orders, function ($total, $order) {
            return $total + (float) $order->get_total();
        }, 0);

        return $total_sales;
    }

    private function get_net_sales($date_range)
    {
        $args = [
            'limit'        => -1,
            'date_created' => $date_range,
            'status'       => ['wc-completed', 'wc-processing'],
            'return'       => 'objects',
        ];

        $orders = \wc_get_orders($args);

        $net_sales = array_reduce($orders, function ($total, $order) {
            $order_total = (float) $order->get_total();
            $order_discounts = (float) $order->get_discount_total();
            $order_refunds = (float) $order->get_refunded_total();

            return $total + ($order_total - $order_discounts - $order_refunds);
        }, 0);

        return $net_sales;
    }

    public function get_metric_types(): array
    {
        return [
            'wc_number_of_orders' => [
                'label'     => __('WooCommerce: number of orders', 'monitori'),
                'category'  => $this->handler,
                'callback'  => 'check_number_of_orders'
            ],
            'wc_gross_sales' => [
                'label'     => __('WooCommerce: gross sales', 'monitori'),
                'category'  => $this->handler,
                'callback'  => 'check_gross_sales'
            ],
            'wc_net_sales' => [
                'label'     => __('WooCommerce: net sales', 'monitori'),
                'category'  => $this->handler,
                'callback'  => 'check_net_sales'
            ],

            // 'wc_discounts' => [
            //     'label'     => __('WooCommerce: discounts', 'monitori'),
            //     'category'  => $this->handler,
            //     'callback'  => 'get_discounts'
            // ],
            // 'wc_number_of_products' => [
            //     'label'    => __('WooCommerce: number of products in the store', 'monitori'),
            //     'category' => $this->handler,
            //     'callback' => 'check_number_of_products',
            // ],
        ];
    }
}

class Woocommerce_Event_Integration extends \Monitori\Base_Event_Integration
{
    public function get_event_types()
    {
        return [
            // Existing Events
            'woocommerce_payment_complete' => [
                'label'    => __('WooCommerce: New order created', 'monitori'),
                'category' => 'woocommerce',
                'args'     => [
                    [
                        'name' => 'order_id',
                        'type' => 'int',
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_order_status_completed' => [
                'label'         => __('WooCommerce: Order completed', 'monitori'),
                'category'      => 'woocommerce',
                'args_handler'  => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_order_status_processing' => [
                'label'    => __('WooCommerce: Order in progress', 'monitori'),
                'category' => 'woocommerce',
                'args_handler'  => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_update_order' => [
                'label'    => __('WooCommerce: Order updated', 'monitori'),
                'category' => 'woocommerce',
                'args_handler'  => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_add_to_cart' => [
                'label'         => __('WooCommerce: Product added to cart', 'monitori'),
                'category'      => 'woocommerce',
                'args'          => [
                    'cart_id' => [
                        'type'  => 'string',
                        'label' => 'Cart ID'
                    ],
                    'product_id' => [
                        'type'  => 'integer',
                        'label' => 'Product ID'
                    ],
                    'request_quantity' => [
                        'type'  => 'integer',
                        'Quantity'
                    ],
                    'variation_id' => [
                        'type'  => 'integer',
                        'label' => 'Variation ID'
                    ],
                    'variation' => [
                        'type'  => 'array',
                        'label' => 'Variation data'
                    ],
                    'cart_item_data' => [
                        'type'  => 'array',
                        'label' => 'Cart item data'
                    ]
                ],
                'args_handler'  => function ($arg_values, $arg_details) {
                    return $this->cart_args_hadler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->cart_preview_handler($arg_details);
                },
            ],
            'woocommerce_remove_cart_item' => [
                'label'    => __('WooCommerce: Product removed from the cart', 'monitori'),
                'category' => 'woocommerce',
                'args'     => [
                    'cart_id' => [
                        'type'  => 'string',
                        'label' => 'Cart ID'
                    ],
                    'product_id' => [
                        'type'  => 'integer',
                        'label' => 'Product ID'
                    ],
                    'request_quantity' => [
                        'type'  => 'integer',
                        'Quantity'
                    ],
                    'variation_id' => [
                        'type'  => 'integer',
                        'label' => 'Variation ID'
                    ],
                    'variation' => [
                        'type'  => 'array',
                        'label' => 'Variation data'
                    ],
                    'cart_item_data' => [
                        'type'  => 'array',
                        'label' => 'Cart item data'
                    ]
                ],
                'args_handler'  => function ($arg_values, $arg_details) {
                    return $this->cart_args_hadler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->cart_preview_handler($arg_details);
                },
            ],
            'woocommerce_after_cart_item_quantity_update' => [
                'label'    => __('WooCommerce: Product quantity changed', 'monitori'),
                'category' => 'woocommerce',
                // 'args'     => [
                //     'cart_id' => [
                //         'type'  => 'string',
                //         'label' => 'Cart ID'
                //     ],
                //     'product_id' => [
                //         'type'  => 'integer',
                //         'label' => 'Product ID'
                //     ],
                //     'request_quantity' => [
                //         'type'  => 'integer',
                //         'Quantity'
                //     ],
                //     'variation_id' => [
                //         'type'  => 'integer',
                //         'label' => 'Variation ID'
                //     ],
                //     'variation' => [
                //         'type'  => 'array',
                //         'label' => 'Variation data'
                //     ],
                //     'cart_item_data' => [
                //         'type'  => 'array',
                //         'label' => 'Cart item data'
                //     ]
                // ],
                // 'args_handler'  => function ($arg_values, $arg_details) {
                //     return $this->cart_args_hadler($arg_values, $arg_details);
                // }
            ],
            'woocommerce_new_order' => [
                'label'    => __('WooCommerce: New order created', 'monitori'),
                'category' => 'woocommerce',
                'args_handler'  => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_product_set_stock_status' => [
                'label'    => __('WooCommerce: Product stock status changed', 'monitori'),
                'category' => 'woocommerce',
            ],
            'woocommerce_applied_coupon' => [
                'label'    => __('WooCommerce: Coupon applied', 'monitori'),
                'category' => 'woocommerce',
                'args'     => ['coupon_code'],
            ],
            'woocommerce_order_status_changed' => [
                'label'    => __('WooCommerce: Order status changed', 'monitori'),
                'category' => 'woocommerce',
                'args_handler'  => function ($arg_values, $arg_details) {
                    $order_id = $arg_values[0];

                    return $this->order_args_handler($order_id);
                },
                'preview_handler' => function ($arg_details) {
                    return $this->order_preview_handler($arg_details);
                },
            ],
            'woocommerce_customer_save_address' => [
                'label'    => __('WooCommerce: Address updated', 'monitori'),
                'category' => 'woocommerce',
                'args'     => ['user_id', 'address_type'],
            ],
            'comment_post' => [
                'label'    => __('WooCommerce: Product review submitted', 'monitori'),
                'category' => 'woocommerce',
                'args'     => ['comment_ID', 'comment_approved', 'commentdata'],
                'is_custom_hook' => true,
                'custom_hook_init' => function () {
                    add_action('comment_post', function ($comment_ID, $comment_approved, $commentdata) {
                        if ($commentdata['comment_type'] === 'review') {
                            do_action('woocommerce_product_review_submitted', $comment_ID, $comment_approved, $commentdata);
                        }
                    }, 10, 3);
                },
            ],
        ];
    }

    public function order_args_handler($order_id, $hyperlinks = true)
    {
        if (!$order_id) return [];

        $order = wc_get_order($order_id);
        $final_args = [];

        // Basic Order Info
        if (!empty($order->get_order_number())) {
            $final_args['Order #'] = $order->get_order_number();

            if ($hyperlinks) {
                $final_args['Order #'] = sprintf(
                    '<a href="%s" target="_blank">#%s</a>',
                    admin_url('post.php?post=' . $order->get_id() . '&action=edit'),
                    $order->get_order_number()
                );
            }
        }

        if (!empty($order->get_billing_first_name()) || !empty($order->get_billing_last_name())) {
            $final_args['Customer'] = sprintf(
                '%s %s',
                $order->get_billing_first_name(),
                $order->get_billing_last_name()
            );
        }

        // Order Total
        if (!empty($order->get_total())) {
            $final_args['Total'] = $order->get_total();
        }

        // Add Products Info
        $products_list = [];

        foreach ($order->get_items() as $item_id => $item) {
            $product_name = $item->get_name();
            $quantity = $item->get_quantity();

            $products_list[] = sprintf('%s (x%d)', $product_name, $quantity);
        }

        if (!empty($products_list)) {
            $final_args['Products'] = implode(', ', $products_list);
        }

        return $final_args;
    }

    public function cart_args_hadler($arg_values, $hyperlinks = true)
    {
        $product_id = $arg_values[1];
        $quantity   = $arg_values[2];
        $final_args = [];

        $product = wc_get_product($product_id);

        if ($product) {
            $product_name = $product->get_name();

            if ($hyperlinks) {
                $product_name = sprintf(
                    '<a href="%s" target="_blank">%s</a>',
                    get_permalink($product_id),
                    $product->get_name()
                );
            }

            $final_args['Product Name'] = $product_name;
            $final_args['Quantity'] = $quantity;
        }

        return $final_args;
    }

    public function cart_preview_handler($arg_details)
    {
        if(!function_exists('wc_get_products')) {
            return false;
        }

        $get_1_product      = wc_get_products(['status' => 'publish', 'limit' => 1, 'orderby' => 'rand']);
        $random_product_id  = !empty($get_1_product) && isset($get_1_product[0]) ? $get_1_product[0]->get_id() : 0;

        if (!$random_product_id) {
            return [];
        }

        $arg_values = [
            '1234',
            $random_product_id,
            2,
            0,
            [],
            []
        ];

        return $this->cart_args_hadler($arg_values, false);
    }

    public function order_preview_handler($arg_details)
    {
        if (!function_exists('wc_get_orders')) {
            return false;
        }

        $orders = \wc_get_orders(['limit' => 1, 'orderby' => 'rand']);
        $order = !empty($orders) ? $orders[0] : null;
        $order_id = $order ? $order->get_id() : null;

        if (!$order) {
            return [
                'Order #' => '1234',
                'Customer' => 'John Doe',
            ];
        }

        return $this->order_args_handler($order_id, false);
    }
}
