<?php

namespace Monitori;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class CoreWP_Event_Integration extends \Monitori\Base_Event_Integration
{
    public $handler = 'corewp';
    public function get_event_types()
    {
        return [
            'monitori_corewp_theme_install'   => [
                'label'              => 'Core WP: Theme Install',
                'category'           => $this->handler,
                'is_custom_hook'     => true,
                'custom_hook_init'   => function () {
                    add_filter('upgrader_post_install', function ($response, $hook_extra) {

                        if (isset($hook_extra['type']) && $hook_extra['type'] === 'theme') {
                            $theme_slug = $hook_extra['theme'] ?? '';
                            $theme      = wp_get_theme($theme_slug);
                            $theme_name = $theme->get('Name');

                            do_action('monitori_corewp_theme_install', $theme_slug, $theme_name);
                        }

                        return $response;
                    }, 10, 3);
                },
                'args' => [
                    'theme_slug' => [
                        'type'  => 'string',
                        'label' => 'Theme Slug'
                    ],
                    'theme_name' => [
                        'type'  => 'string',
                        'label' => 'Theme Name'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'theme_slug' => 'twentytwentyfive',
                        'theme_name' => 'Twenty Twenty-Five'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_corewp_theme_activation'   => [
                'label'              => 'Core WP: Theme Activation',
                'category'           => $this->handler,
                'is_custom_hook'     => true,
                'custom_hook_init'   => function () {
                    add_action('after_switch_theme', function () {
                        $theme = wp_get_theme();
                        $theme_slug = $theme->get_stylesheet();
                        $theme_name = $theme->get('Name');

                        do_action('monitori_corewp_theme_activation', $theme_slug, $theme_name);
                    });
                },
                'args' => [
                    'theme_slug' => [
                        'type'  => 'string',
                        'label' => 'Theme Slug'
                    ],
                    'theme_name' => [
                        'type'  => 'string',
                        'label' => 'Theme Name'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'theme_slug' => 'twentytwentyfive',
                        'theme_name' => 'Twenty Twenty-Five'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_corewp_theme_deactivation' => [
                'label'            => 'Core WP: Theme Deactivation',
                'category'         => $this->handler,
                'is_custom_hook'   => true,
                'custom_hook_init' => function () {
                    add_action('switch_theme', function ($new_theme_slug) {
                        $old_theme = wp_get_theme();
                        $old_theme_slug = $old_theme->get_stylesheet();
                        $old_theme_name = $old_theme->get('Name');

                        do_action('monitori_corewp_theme_deactivation', $old_theme_slug, $old_theme_name, $new_theme_slug);
                    });
                },
                'args' => [
                    'old_theme_slug' => [
                        'type'  => 'string',
                        'label' => 'Old Theme Slug'
                    ],
                    'old_theme_name' => [
                        'type'  => 'string',
                        'label' => 'Old Theme Name'
                    ],
                    'new_theme_slug' => [
                        'type'  => 'string',
                        'label' => 'New Theme Slug'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'old_theme_slug' => 'twentytwentyfive',
                        'old_theme_name' => 'Twenty Twenty-Five',
                        'new_theme_slug' => 'twentytwentythree'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'update_option' => [
                'label'            => 'Core WP: Settings Updated',
                'category'         => $this->handler,
                'args'             => [
                    'option' => [
                        'type'  => 'string',
                        'label' => 'Option Name'
                    ],
                    'old_value' => [
                        'type'  => 'mixed',
                        'label' => 'Old Value'
                    ],
                    'value' => [
                        'type'  => 'mixed',
                        'label' => 'New Value'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'option'    => 'blogname',
                        'old_value' => 'Old Site Name',
                        'value'     => 'New Site Name'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'upgrader_process_complete' => [
                'label'            => 'Core WP: Upgrade Process Complete',
                'category'         => $this->handler,
                // 'args'             => [
                //     'upgrader' => [
                //         'type'  => 'object',
                //         'label' => 'Upgrader Instance'
                //     ],
                //     'hook_extra' => [
                //         'type'  => 'array',
                //         'label' => 'Hook Extra'
                //     ]
                // ],
                // 'args_handler' => function ($arg_values, $arg_details) {
                //     return $this->args_handler($arg_values, $arg_details);
                // }
            ],
            'activated_plugin' => [
                'label'            => 'Core WP: Plugin Activated',
                'category'         => $this->handler,
                'args'             => [
                    'plugin'        => [
                        'type'  => 'string',
                        'label' => 'Plugin Name'
                    ],
                    'network_wide'  => [
                        'type'  => 'boolean',
                        'label' => 'Network Wide'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'plugin'        => 'akismet/akismet.php',
                        'network_wide'  => true
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'deactivated_plugin' => [
                'label'            => 'Core WP: Plugin Deactivated',
                'category'         => $this->handler,
                'args'             => [
                    'plugin'        => [
                        'type'  => 'string',
                        'label' => 'Plugin Name'
                    ],
                    'network_wide'  => [
                        'type'  => 'boolean',
                        'label' => 'Network Wide'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'plugin'        => 'akismet/akismet.php',
                        'network_wide'  => true
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_user_created' => [
                'label'            => 'Core WP: User Created',
                'category'         => $this->handler,
                'is_custom_hook'   => true,
                'custom_hook_init' => function () {
                    add_action('user_register', function ($user_id) {
                        $user = get_user_by('id', $user_id);
                        $user_email = $user->user_email;

                        do_action('monitori_user_created', $user_id, $user_email);
                    });
                },
                'args' => [
                    'user_id' => [
                        'type'  => 'int',
                        'label' => 'User ID'
                    ],
                    'user_email' => [
                        'type'  => 'string',
                        'label' => 'User Email'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'user_id'       => 1,
                        'user_email'    => 'user@email.com'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_user_deleted' => [
                'label'            => 'Core WP: User Deleted',
                'category'         => $this->handler,
                'is_custom_hook'   => true,
                'custom_hook_init' => function () {
                    add_action('delete_user', function ($user_id) {
                        $user = get_user_by('id', $user_id);

                        if ($user) {
                            $user_email = $user->user_email;
                            do_action('monitori_user_deleted', $user_id, $user_email);
                        }
                    }, 10, 1);
                },
                'args' => [
                    'user_id' => [
                        'type'  => 'int',
                        'label' => 'User ID'
                    ],
                    'user_email' => [
                        'type'  => 'string',
                        'label' => 'User Email'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'user_id'       => 1,
                        'user_email'    => 'user@email.com'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'set_user_role' => [
                'label'            => 'Core WP: User Role Changed',
                'category'         => $this->handler,
                'args'             => [
                    'user_id' => [
                        'type'  => 'int',
                        'label' => 'User ID'
                    ],
                    'role' => [
                        'type'  => 'string',
                        'label' => 'Role'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'user_id' => 1,
                        'role'    => 'administrator'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_wpconfig_file_changed' => [
                'label'            => 'Core WP: wp-config.php File Changed',
                'category'         => $this->handler,
                'is_custom_hook'   => true,
                'custom_hook_init' => function () {

                    $event_hook = 'monitori_event_notification_daily';

                    if (!wp_next_scheduled($event_hook, [])) {
                        wp_schedule_event(time(), 'daily', $event_hook, []);
                    }

                    $instance = $this;

                    add_action($event_hook, function () use ($instance) {
                        $instance->watch_file_changes(ABSPATH . 'wp-config.php', 'monitori_wp_config_last_modified');
                    });
                },
                'args' => [
                    'current_time' => [
                        'type'  => 'string',
                        'label' => 'Current Time'
                    ],
                    'last_checked_time' => [
                        'type'  => 'string',
                        'label' => 'Last Checked Time'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'current_time'      => date_i18n('Y-m-d H:i:s', time()),
                        'last_checked_time' => date_i18n('Y-m-d H:i:s', get_option('monitori_wp_config_last_modified', 0))
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'wp_login'  => [
                'label'            => 'Core WP: User Logged In',
                'category'         => $this->handler,
                'args'             => [
                    'user_login' => [
                        'type'  => 'string',
                        'label' => 'User Login'
                    ],
                    // 'user' => [
                    //     'type'  => 'object',
                    //     'label' => 'User Object'
                    // ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'user_login' => 'admin'
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'wp_logout'  => [
                'label'            => 'Core WP: User Logged Out',
                'category'         => $this->handler,
                'args'             => [
                    'user_id' => [
                        'type'  => 'int',
                        'label' => 'User ID'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'user_id' => 1
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
            'monitori_corewp_wp_mail_failed' => [
                'label'            => 'Core WP: WP Mail Failed',
                'category'         => $this->handler,
                'is_custom_hook'   => true,
                'custom_hook_init' => function () {
                    add_action('wp_mail_failed', function ($wp_error) {
                        do_action('monitori_corewp_wp_mail_failed', $wp_error);
                    });
                },
                'args' => [
                    'wp_error' => [
                        'type'  => 'object',
                        'label' => 'WP Error'
                    ]
                ],
                'args_handler' => function ($arg_values, $arg_details) {
                    return $this->args_handler($arg_values, $arg_details);
                },
                'preview_values' => [
                    'wp_error' => new \WP_Error('test_code', 'Test error message')
                ],
                'preview_handler' => function ($arg_details) {
                    $arg_values = [
                        'wp_error' => new \WP_Error('test_code', 'Test error message')
                    ];
                    return $this->args_handler($arg_values, $arg_details);
                }
            ],
        ];
    }

    public function watch_file_changes($file_path, $db_slug)
    {
        if (file_exists($file_path)) {
            $current_time = filemtime($file_path);
            $last_checked_time = get_option($db_slug, 0);

            if (!$last_checked_time) {
                // running for the first time
                update_option($db_slug, $current_time);
                return;
            }

            if ($current_time > $last_checked_time) {
                $current_time_human = date_i18n('Y-m-d H:i:s', $current_time); // Current time
                $last_checked_time_human = $last_checked_time ? date_i18n('Y-m-d H:i:s', $last_checked_time) : 'Never';

                do_action('monitori_wpconfig_file_changed', $current_time_human, $last_checked_time_human);
                // Update the last checked time
                update_option($db_slug, $current_time);
            }
        }
    }
}
