<?php

namespace Monitori;

/**
 * The base of how the integration should look like
 * 
 * Defines all of the possible metric types, evaluations, 
 * and also integrates into existing code in the right places
 */
class Base_Metric_Integration
{
    public $metric_types;
    public $handler;

    public function check_metric($settings)
    {
        return false;
    }

    public function __construct()
    {
        add_action('monitori_metric_types', [$this, 'add_metric_types']);
        add_action('monitori_metric_integrations', [$this, 'add_integrations']);
    }

    /**
     * Callback to add current integration to the list of all available integrations
     * 
     * @param array $current_integrations Current list of integration callbacks
     * @return array Final list of integration callbacks including the current integration's callback
     * 
     * 
     */
    public function add_integrations($current_integrations)
    {
        $context = $this;

        $current_integrations[$this->handler] = function ($settings) use ($context) {
            return $context->check_metric($settings);
        };

        return $current_integrations;
    }

    /**
     * Retrieve the list of metrics for this integration
     * 
     * @return array
     */
    public function get_metric_types(): array
    {
        return $this->metric_types;
    }

    /**
     * Callback to add current metrics to the list of all available metrics
     * 
     * @param array $current_types Current list of metric types
     * @return array Final list of metric types including the current integration's metrics
     */
    public function add_metric_types(array $current_types): array
    {

        $final_metric_types = array_merge($current_types, $this->get_metric_types());

        return $final_metric_types;
    }

    /**
     * Evaluate a change in a metric
     */
    public function evaluate_change($current_value, $previous_value, $change_type, $threshold): bool
    {
        $current_value = (int) $current_value;
        $previous_value = (int) $previous_value;
        $threshold = (int) $threshold;

        $this->log("Evaluating change: Type: $change_type | Current: $current_value | Previous: $previous_value | Threshold: $threshold");

        return match ($change_type) {
            'percentage_increased' => $this->check_percentage_change([
                'current_value'  => $current_value,
                'previous_value' => $previous_value,
                'threshold'      => $threshold,
                'direction'      => 'increased',
            ]),

            'percentage_decreased' => $this->check_percentage_change([
                'current_value'  => $current_value,
                'previous_value' => $previous_value,
                'threshold'      => $threshold,
                'direction'      => 'decreased',
            ]),

            'value_equals_to' => $this->check_value_change([
                'current_value'  => $current_value,
                'previous_value' => $previous_value,
                'threshold'      => $threshold,
                'direction'      => 'exact',
            ]),

            'value_increased_by' => $this->check_value_change([
                'current_value'  => $current_value,
                'previous_value' => $previous_value,
                'threshold'      => $threshold,
                'direction'      => 'increased',
            ]),

            'value_decreased_by' => $this->check_value_change([
                'current_value'  => $current_value,
                'previous_value' => $previous_value,
                'threshold'      => $threshold,
                'direction'      => 'decreased',
            ]),

            default => $this->log("Unsupported change type: $change_type") && false,
        };
    }

    /**
     * Evaluates a percentage change between two values.
     *
     * Calculates the percentage difference from `previous_value` to `current_value`
     * and checks if it meets the specified `threshold` in the given `direction`.
     * Handles division by zero when `previous_value` is 0.
     *
     * @param array $args {
     *     @type float $current_value  Current value (default: 0).
     *     @type float $previous_value Previous value (default: 0).
     *     @type float $threshold      Minimum percentage change (default: 0).
     *     @type string $direction     'increased' or 'decreased' (default: 'increased').
     * }
     * @return bool True if the condition is met, False otherwise.
     */
    public function check_percentage_change(array $args): bool
    {
        $current_value = $args['current_value'] ?? 0;
        $previous_value = $args['previous_value'] ?? 0;
        $threshold = $args['threshold'] ?? 0;
        $direction = $args['direction'] ?? 'increased';

        // Handle division by zero
        if ($previous_value == 0) {
            // If current value > 0, consider it an infinite increase
            return $current_value > 0 && $direction === 'increased';
        }

        // Calculate percentage change
        $percentage_change = (($current_value - $previous_value) / $previous_value) * 100;

        return match ($direction) {
            'increased' => $percentage_change >= $threshold,
            'decreased' => $percentage_change <= -$threshold,
            default     => false,
        };
    }

    /**
     * Compares two values with a threshold.
     *
     * Evaluates `current_value` against `previous_value` using `threshold`
     * and `direction` ('increased', 'decreased', or 'exact').
     *
     * @param array $args {
     *     @type float $current_value  Current value (default: 0).
     *     @type float $previous_value Previous value (default: 0).
     *     @type float $threshold      Minimum value difference (default: 0).
     *     @type string $direction     Comparison type (default: 'exact').
     * }
     * @return bool True if the condition is met, False otherwise.
     */
    private function check_value_change(array $args): bool
    {
        $current_value = $args['current_value'] ?? 0;
        $previous_value = $args['previous_value'] ?? 0;
        $threshold = $args['threshold'] ?? 0;
        $direction = $args['direction'] ?? 'exact';

        return match ($direction) {
            'increased' => $current_value >= $previous_value + $threshold,
            'decreased' => $current_value <= $previous_value - $threshold,
            'exact'     => $current_value == $previous_value,
            default     => false,
        };
    }
    public function log($msg)
    {
        \monitori_console_log($msg);
    }

    static function get_metric_change_types()
    {
        return [
            'percentage_increased' => __('% increased by', 'monitori'),
            'percentage_decreased' => __('% decreased by', 'monitori'),
            'value_increased_by'   => __('Has increased by', 'monitori'),
            'value_decreased_by'   => __('Has decreased by', 'monitori'),
            'value_equals_to'      => __('Equals to', 'monitori'),
        ];
    }
}

class Base_Event_Integration
{
    public $event_types = [];
    public $handler;

    public function __construct()
    {
        add_action('monitori_event_types', [$this, 'add_event_types']);
    }

    public function get_event_types()
    {
        return $this->event_types;
    }

    public function add_event_types($current_types)
    {

        $final_event_types = array_merge($current_types, $this->get_event_types());

        return $final_event_types;
    }

    /**
     * Default way of handling arguments and preparing for display
     */
    public function args_handler($arg_values, $arg_details)
    {
        $count          = 0;
        $arg_keys       = array_keys($arg_details);
        $event_args     = [];

        foreach ($arg_values as $value) {
            $key = $arg_keys[$count];
            $arg = $arg_details[$key];

            if (isset($value) && isset($arg['label'])) {

                if ($arg['type'] === 'boolean') {
                    $value = $value ? 'Yes' : 'No';
                }

                if ($arg['type'] === 'object') {
                    $value = json_encode($value);
                }

                $event_args[$arg['label']] = $value ?? '';
            }

            $count++;
        }

        return $event_args;
    }
}
