<?php

namespace Monitori;

final class Slack_Setup
{
    static function get_slack_info()
    {
        $slack_info = get_option('monitori_slack_info');

        if (!$slack_info) {
            return null; // Option not set
        }

        $slack_info_parsed = json_decode($slack_info);

        return $slack_info_parsed;
    }

    static function prepare_message($hook_label, $args)
    {
        $subject = self::create_event_notification_subject($hook_label);

        $final_message = [
            [
                'type' => 'header',
                'text' => [
                    'type'  => 'plain_text',
                    'text'  => $subject,
                    'emoji' => true,
                ],
            ],
        ];

        if (!empty($args)) {
            foreach ($args as $key => $value) {
                $final_message[] = [
                    'type' => 'section',
                    'text' => [ // not "field"
                        'type' => 'mrkdwn',
                        'text' => '*' . $key . ':*' . "\n" . self::sanitize_for_slack($value),
                    ],
                ];
            }
        }

        return $final_message;
    }

    static function send_notification($message)
    {
        $slack_info = self::get_slack_info();

        // Check if the access token is available
        if (!$slack_info->access_token) {
            return 'Error: Access token not found.';
        }

        // Prepare the API request
        $url = 'https://slack.com/api/chat.postMessage';
        $args = [
            'body' => wp_json_encode([
                'channel' => $slack_info->connected_channel,
                'blocks'  => $message,
            ]),
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $slack_info->access_token,
            ],
            'method' => 'POST',
        ];

        $response = wp_remote_post($url, $args);

        if (is_wp_error($response)) {
            return 'Error: ' . $response->get_error_message();
        }

        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        return  $response_body['ok'] ? 'Success' : 'Failed';
    }

    private static function sanitize_for_slack($value)
    {
        // Convert <a href="url">text</a> to <url|text>
        $value = preg_replace_callback(
            '/<a\s+[^>]*href=[\'"]([^\'"]+)[\'"][^>]*>(.*?)<\/a>/i',
            function ($matches) {
                $url = html_entity_decode($matches[1]); // decode any &amp;, &#038;, etc.
                $text = trim(strip_tags($matches[2]));  // just the visible link text
                return '<' . $url . '|' . $text . '>';
            },
            $value
        );

        return $value;
    }

    static function get_slack_html($should_render = false)
    {
        $slack_info_parsed = self::get_slack_info();
        $final_escaped_string = '';

        if (!empty($slack_info_parsed->access_token)) {
            $final_escaped_string = '<p>';
            $final_escaped_string .= esc_html__('Sending messages to: ', 'monitori');
            $final_escaped_string .= esc_html('#' . $slack_info_parsed->connected_channel);
            $final_escaped_string .= ' <a href="' . esc_url(\Monitori\Admin_Edit_UI::generate_disconnect_url()) . '">';
            $final_escaped_string .= esc_html__('Disconnect', 'monitori');
            $final_escaped_string .= '</a></p>';
        } else {
            $final_escaped_string = sprintf(
                '<a href="%s" class="button">%s</a>',
                esc_url(\Monitori\Admin_Edit_UI::generate_oauth_url()),
                esc_html__('Connect Slack', 'monitori')
            );
        }

        if ($should_render) {
            echo $final_escaped_string; // Safe to echo as all content is escaped
        } else {
            return $final_escaped_string; // Safe to return as all content is escaped
        }
    }

    static function create_event_notification_subject($hook_label)
    {
        // translators: "Monitori - event triggered: {NAME_OF_THE_EVENT}"
        return sprintf(__('Event triggered: %s', 'monitori'), $hook_label);
    }

    static function get_slack_fields_api()
    {
        $slack_info_parsed = self::get_slack_info();

        if (!empty($slack_info_parsed->access_token)) {
            return [
                'type' => 'channel',
                'channel' => '#' . $slack_info_parsed->connected_channel,
                'disconnect_url' => \Monitori\Admin_Edit_UI::generate_disconnect_url('slack', true),
                'message' => __('Sending messages to:', 'monitori'),
            ];
        }

        return [
            'type' => 'connect',
            'oauth_url' => \Monitori\Admin_Edit_UI::generate_oauth_url('slack', true),
            'button_label' => __('Connect Slack', 'monitori'),
        ];
    }
}
