<?php

namespace Monitori;

final class Discord_Setup {
    static function get_discord_info() {
        $discord_info = get_option( 'monitori_discord_info' );
        if ( !$discord_info ) {
            return null;
            // Option not set
        }
        $discord_info_parsed = json_decode( $discord_info );
        if ( !isset( $discord_info_parsed->expires_at ) || !isset( $discord_info_parsed->refresh_token ) ) {
            return null;
            // Malformed data
        }
        if ( time() >= $discord_info_parsed->expires_at ) {
            $new_token = self::refresh_discord_token( $discord_info_parsed );
            if ( $new_token ) {
                return json_decode( get_option( 'monitori_discord_info' ) );
                // Return the updated info
            }
            return null;
            // Refresh failed
        }
        return $discord_info_parsed;
    }

    static function refresh_discord_token( $discord_info ) {
        $response = wp_remote_get( MONITORI_MIDDLEWARE_URL . '/api/discord/refresh?refresh_token=' . urlencode( $discord_info->refresh_token ) );
        if ( is_wp_error( $response ) ) {
            \monitori_console_log( 'Discord token refresh failed: ' . $response->get_error_message() );
            return null;
        }
        $data = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( !isset( $data['access_token'], $data['refresh_token'], $data['expires_in'] ) ) {
            \monitori_console_log( 'Discord token refresh failed: Invalid response.' );
            return null;
        }
        self::store_access_token(
            $data['access_token'],
            $discord_info->guild_id,
            $data['refresh_token'],
            time() + (int) $data['expires_in']
        );
        return $data['access_token'];
    }

    static function store_access_token(
        $access_token,
        $guild_id,
        $refresh_token,
        $expires_at
    ) {
        $result = update_option( 'monitori_discord_info', wp_json_encode( [
            'access_token'  => $access_token,
            'guild_id'      => $guild_id,
            'refresh_token' => $refresh_token,
            'expires_at'    => $expires_at,
        ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE ) );
        if ( !$result ) {
            \monitori_console_log( 'Failed to update Discord token in the database.' );
        }
        return $result;
    }

    static function disconnect() {
        delete_option( 'monitori_discord_info' );
    }

    static function send_notification( $channel_id, $message ) {
        return 'Success';
    }

    static function create_event_notification_subject( $hook_label ) {
        // translators: "Monitori - event triggered: {NAME_OF_THE_EVENT}"
        return sprintf( __( 'Event triggered: %s', 'monitori' ), $hook_label );
    }

    static function prepare_message( $hook_label, $args ) {
        $subject = self::create_event_notification_subject( $hook_label );
        $embed = [
            'title'  => $subject,
            'color'  => 5814783,
            'fields' => [],
        ];
        if ( !empty( $args ) ) {
            foreach ( $args as $key => $value ) {
                $embed['fields'][] = [
                    'name'   => $key,
                    'value'  => self::sanitize_for_discord( $value ),
                    'inline' => false,
                ];
            }
        }
        return [
            'embeds' => [$embed],
        ];
    }

    private static function sanitize_for_discord( $value ) {
        // Match and extract the href from <a> tags and return only the clean URL
        return preg_replace_callback( '/<a\\s+[^>]*href=[\'"]([^\'"]+)[\'"][^>]*>.*?<\\/a>/i', function ( $matches ) {
            return html_entity_decode( $matches[1] );
            // decode &amp; etc. to &
        }, $value );
    }

    static function get_discord_channels( $guild_id, $access_token ) {
        return [
            'channels' => [],
        ];
    }

    static function get_discord_fields_api() {
        return [
            'type'    => 'other',
            'label'   => __( 'Other', 'monitori' ),
            'message' => '',
        ];
    }

    static function get_discord_html( $is_connected = false, $should_render ) {
        return '';
    }

}
