<?php

namespace Monitori;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class Setup
{
    public $default_colors;
    public $metric_evaluator;

    function __construct()
    {
        // admin page
        new \Monitori\Admin_Dashboard_UI();

        // handling connected channels like slack
        add_action('admin_init', [$this, 'handle_connected_channels']);

        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        add_filter('wp_script_attributes', function ($attributes) {
            if ('anyf-script-js' === $attributes['id'] || 'anyf-admin-script-js' === $attributes['id']) {
                $attributes['type'] = 'module';
            }

            return $attributes;
        });

        $this->load_notifications();
    }

    public function load_notifications()
    {
        new \Monitori\Event_Manager();

        // todo: move retrieval to the manager class
        new \Monitori\Metric_Manager();
    }

    /**
     * Check if the URL has access tokens,
     * Safely store them in DB
     * Update the user!
     */
    public function handle_connected_channels()
    {
        if (isset($_GET['monitori_slack_access_token'])) {
            // Check if the connected channel parameter is present
            if (!isset($_GET['monitori_slack_connected_channel'])) {
                echo 'Something went wrong';
                return; // Exit the function early if the parameter is missing
            }

            // Sanitize the input values
            $access_token = sanitize_text_field(wp_unslash($_GET['monitori_slack_access_token']));
            $connected_channel = sanitize_text_field(wp_unslash($_GET['monitori_slack_connected_channel']));

            // Update the options in the database
            update_option('monitori_slack_info', wp_json_encode([
                'access_token'      => $access_token,
                'connected_channel' => $connected_channel,
            ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));

            // // Display an admin notice to confirm the connection was successful
            // add_action('admin_notices', function () use ($connected_channel) {
            //     echo '<div class="notice notice-success is-dismissible">
            //         <p>Slack app connected successfully! Connected channel: ' . esc_html($connected_channel) . '</p>
            //     </div>';
            // });

            // Redirect to a clean URL to remove the GET parameter from the browser address bar
            wp_safe_redirect(remove_query_arg(['monitori_slack_access_token', 'monitori_slack_connected_channel']));
            exit;
        }

        // Handle disconnection
        if (isset($_GET['monitori_disconnect_slack'])) {
            // Delete the stored options from the database
            delete_option('monitori_slack_info');

            // Display an admin notice confirming disconnection
            add_action('admin_notices', function () {
                echo '<div class="notice notice-warning is-dismissible">
                    <p>Slack app has been disconnected successfully.</p>
                </div>';
            });

            // Redirect to a clean URL to remove the GET parameter from the browser address bar
            wp_safe_redirect(remove_query_arg('monitori_disconnect_slack'));
            exit;
        }

        if (isset($_GET['monitori_discord_access_token'])) {
            // Check if the connected channel parameter is present
            if (!isset($_GET['monitori_discord_guild_id']) || !isset($_GET['monitori_discord_refresh_token'])) {
                echo 'Something went wrong';
                return; // Exit the function early if the parameter is missing
            }

            // Sanitize the input values
            $access_token = sanitize_text_field(wp_unslash($_GET['monitori_discord_access_token']));
            $refresh_token = sanitize_text_field(wp_unslash($_GET['monitori_discord_refresh_token']));
            $guild_id = sanitize_text_field(wp_unslash($_GET['monitori_discord_guild_id']));
            $expires_in = sanitize_text_field(wp_unslash($_GET['monitori_discord_expires_in']));

            // Calculate the expiration timestamp by adding `expires_in` to the current time
            $expires_at = time() + (int)$expires_in;

            \Monitori\Discord_Setup::store_access_token($access_token, $guild_id, $refresh_token, $expires_at);

            // Redirect to a clean URL to remove the GET parameter from the browser address bar
            wp_safe_redirect(remove_query_arg(['monitori_discord_access_token', 'monitori_discord_guild_id', 'monitori_discord_refresh_token']));
            exit;
        }

        if (isset($_GET['monitori_disconnect_discord'])) {

            \Monitori\Discord_Setup::disconnect();

            // Redirect to a clean URL to remove the GET parameter from the browser address bar
            wp_safe_redirect(remove_query_arg('monitori_disconnect_discord'));
            exit;
        }
    }


    /**
     * Enqueue admin scripts and styles.
     */
    public function enqueue_admin_assets()
    {
        global $pagenow;

        if ($pagenow == 'admin.php' && isset($_GET['page']) && $_GET['page'] == 'monitori-notifications') {
            wp_enqueue_script('wp-blocks');
            wp_enqueue_script('wp-element');
            wp_enqueue_style('wp-components');
        }

        wp_enqueue_style('anyf-admin-style', plugin_dir_url(dirname(__FILE__)) . 'dist/admin.css', array(), MONITORI_ASSET_VERSION);
        wp_enqueue_script('anyf-admin-script', plugin_dir_url(dirname(__FILE__)) . 'dist/admin.js', array(), MONITORI_ASSET_VERSION, true);
    }

    static function plugin_activation()
    {
        \Monitori\Database::create_notifications_table();
        
        \flush_rewrite_rules();
    }
}
