<?php

namespace Monitori;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * The base notifier class
 * Meant to be extended
 */
class Notifier
{
    public $category;
    public $delivery_channels;
    public $message;
    public $domain;
    public $hook_name;
    public $hook_label;
    public $args;

    function __construct($settings)
    {
        $this->delivery_channels    = $settings['delivery_channels'] ?? [];
        $this->category             = $settings['category'];
        $this->message              = $settings['message'] ?? '';
        $this->hook_name            = $settings['hook_name'] ?? '';

        if (!empty($settings['args'])) {
            $this->args = $this->prepare_notification_args($settings['args']);
        }

        $event_types = \Monitori\Event_Manager::get_types();
        $current_hook = !empty($event_types[$this->hook_name]) ? $event_types[$this->hook_name] : [];

        $this->hook_label = !empty($current_hook['label']) ? $current_hook['label'] : '';

        $this->domain = wp_parse_url(home_url(), PHP_URL_HOST);

        if ($this->domain === 'localhost') {
            $this->domain = 'localhost.com';
        }
    }

    public function slack_notification()
    {
        $message    = \Monitori\Slack_Setup::prepare_message($this->hook_label, $this->args);
        $response   = \Monitori\Slack_Setup::send_notification($message);

        return $response;
    }


    public function discord_notification($channel)
    {
        $message    = \Monitori\Discord_Setup::prepare_message($this->hook_label, $this->args);
        
        $channel_id = $channel['channel'] . '';
        
        $response = \Monitori\Discord_Setup::send_notification($channel_id, $message);

        return $response;
    }

    public function email_notification($recipient)
    {
        $subject    = \Monitori\Email_Setup::create_event_notification_subject($this->hook_label);
        $message    = \Monitori\Email_Setup::prepare_email_message($this->hook_label, $this->args);

        return \Monitori\Email_Setup::send_notification([
            'recipient' => $recipient,
            'subject'   => $subject,
            'message'   => $message,
            'from'      => 'monitori-no-reply@' . $this->domain
        ]);
    }

    public function send_notifications()
    {

        foreach ($this->delivery_channels as $type => $channel) {
            if ($type === 'email' && $channel['enabled']) {
                $res = $this->email_notification($channel['recipient']);
            }

            if ($type === 'slack' && $channel['enabled']) {
                $res = $this->slack_notification();
            }

            if ($type === 'discord' && $channel['enabled']) {
                $res = $this->discord_notification($channel);
            }
        }
    }

    /**
     * Prepares the notification arguments based on the provided argument values.
     *
     * This function retrieves the event types from the `Monitori\Event_Manager` class,
     * determines the current hook based on the `hook_name` property, and then maps
     * the provided argument values to the corresponding event arguments.
     *
     * @param array $arg_values The values to be used for preparing the notification arguments.
     * 
     * @return array The prepared notification arguments, where the keys are the argument labels
     *               and the values are the corresponding provided values.
     */
    public function prepare_notification_args($arg_values)
    {
        $event_types    = \Monitori\Event_Manager::get_types();
        $current_hook   = !empty($event_types[$this->hook_name]) ? $event_types[$this->hook_name] : [];
        $event_args     = [];

        if ((!empty($current_hook['args']) || !empty($current_hook['args_handler'])) && !empty($arg_values)) {

            // If a custom handler is set, use it to handle the arguments
            if (!empty($current_hook['args_handler'])) {
                $args_handler = $current_hook['args_handler'];
                $arg_values = $args_handler($arg_values, $current_hook['args']);
                return $arg_values;
            }
        }

        return $event_args;
    }
}

class Metric_Notifier extends Notifier
{
    public $metric_details;

    function __construct($settings)
    {
        $this->delivery_channels = $settings['delivery_channels'] ?? [];
        $this->category = $settings['category'];
        $this->message = $settings['message'] ?? '';
        $this->hook_name = $settings['hook_name'] ?? '';
        $this->metric_details = $settings['metric_details'] ?? '';

        $event_types = \Monitori\Metric_Manager::get_types();
        $current_hook = !empty($event_types[$this->hook_name]) ? $event_types[$this->hook_name] : [];

        $this->hook_label = !empty($current_hook['label']) ? $current_hook['label'] : '';

        $this->domain = wp_parse_url(home_url(), PHP_URL_HOST);

        if ($this->domain === 'localhost') {
            $this->domain = 'localhost.com';
        }
    }

    public function create_notification_subject()
    {
        $change_type_labels = \Monitori\Base_Metric_Integration::get_metric_change_types();
        $current_change_type_label = $change_type_labels[$this->metric_details['change_type']];
        $current_change_type_label = strtolower($current_change_type_label);

        return sprintf(
            // translators: "Monitori: {hook_label} {current_change_type_label} {value}" example: "Monitori: Number of orders decreased by 10"
            __('Monitori: %1$s %2$s %3$s', 'monitori'),
            $this->hook_label,
            $current_change_type_label,
            $this->metric_details['value']
        );
    }
}
