<?php

namespace Monitori;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Schedule the metric based notifications and notify user
 */
class Metric_Manager
{
    public $metric_notifications;
    public $time_periods = [
        'daily',
        'weekly',
        'monthly',
        // 'yearly'
    ];
    public $metrics_map = [];
    public $integration;

    public function __construct()
    {
        add_action('init', [$this, 'init']);
    }

    public function init()
    {
        $metrics_raw = \Monitori\Database::get_notifications_paginated(1, 5000, 'metric');

        $this->metric_notifications = $metrics_raw['notifications'];

        if (!empty($this->metric_notifications)) {

            // categorize events based on their time period
            foreach ($this->metric_notifications as $notification) {
                $time_period = $notification->time_period;
                $this->metrics_map[$time_period][] = $notification;
            }

            add_action('monitori_save_notification', [$this, 'monitori_update_crons'], 10, 3);

            foreach ($this->time_periods as $the_time_period) {
                $event_hook = $this->get_cron_hook($the_time_period);
                add_action($event_hook, [$this, 'handle_metric_notification']);
            }
        }
    }

    /**
     * then reschedules them based on the new data
     */
    public function monitori_update_crons($post_id)
    {
        // Check if this is an autosave or revision.
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Verify post type and user capability
        if (!current_user_can('manage_options')) {
            return;
        }

        // re-schedule them again with the new data
        foreach ($this->time_periods as $schedule_time_period) {
            // don't schedule if there are no events for it
            if (empty($this->metrics_map[$schedule_time_period])) continue;

            // Create a unique event key so the same event isn't scheduled multiple times
            $event_hook = $this->get_cron_hook($schedule_time_period);
            $event_args = [$schedule_time_period];

            // Schedule the event only if it’s not already scheduled
            if (!wp_next_scheduled($event_hook, $event_args)) {
                wp_schedule_event(time(), $schedule_time_period, $event_hook, $event_args);
            }
        }
    }

    public function handle_metric_notification($current_time_period = '')
    {
        $integration_map = [];

        $integration_map = apply_filters('monitori_metric_integrations', [], $integration_map);

        $current_metrics = !empty($this->metrics_map[$current_time_period]) ? $this->metrics_map[$current_time_period] : [];

        if (empty($current_metrics)) return false;

        foreach ($current_metrics as $metric_notification) {
            $metric_handler_name = $this->get_metric_handler($metric_notification->metric_name);
            $metric_handler_callback = !empty($integration_map[$metric_handler_name]) ? $integration_map[$metric_handler_name] : false;

            if (!is_callable($metric_handler_callback)) return false;

            $metric_details = [
                'metric_id'             => $metric_notification->id,
                'metric_name'           => $metric_notification->metric_name,
                'change_type'           => $metric_notification->change_type,
                'value'                 => $metric_notification->metric_value,
                'time_period'           => $metric_notification->time_period,
                'compare_to'            => $metric_notification->compare_period ?? 'last_period',
                'delivery_channels'     => json_decode($metric_notification->delivery_channels)
            ];
            
            // todo: figure out
            $metric_matched = $metric_handler_callback($metric_details);
            
            if ($metric_matched) {
                // Send notifications
                $this->trigger_notification($metric_details);
            }
        }
    }

    public function get_metric_handler($name)
    {
        $current_metric_type = self::get($name);

        return !empty($current_metric_type['category']) ? $current_metric_type['category'] : false;
    }

    public function trigger_notification($metric_details)
    {
        $delivery_channels = $metric_details['delivery_channels'];
        $hook_name = $metric_details['metric_name'];

        $notifier = new \Monitori\Metric_Notifier(array(
            'category'           => 'metric',
            'delivery_channels'  => $delivery_channels,
            'metric_details'     => $metric_details,
            'hook_name'          => $hook_name
        ));

        $notifier->send_notifications();
    }

    public function get_cron_hook($str_time_period)
    {
        return 'monitori_metric_notification_' . $str_time_period;
    }

    public function log($el)
    {
        \monitori_console_log($el);
    }

    /**
     * ======
     * 
     * Getting the types
     * 
     * ======
     */
    static function get_types()
    {
        $types = apply_filters('monitori_metric_types', []);

        return $types;
    }

    static function get($name)
    {
        $types = self::get_types();

        if (empty($types[$name])) return false;

        return $types[$name];
    }

    static function get_option_values()
    {
        $types = self::get_types();
        $option_values = [];

        foreach ($types as $metric_name => $details) {
            $option_values[$metric_name] = $details['label'];
        }

        return $option_values;
    }
}
