<?php

namespace Monitori;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

final class Database
{
    /**
     * Create the notifications table.
     */
    public static function create_notifications_table()
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$wpdb->prefix}monitori_notifications (
            id int(11) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            notification_type varchar(255) NOT NULL,
            event_type varchar(255) DEFAULT NULL,
            metric_name varchar(255) DEFAULT NULL,
            change_type varchar(255) DEFAULT NULL,
            metric_value decimal(10, 2) DEFAULT NULL,
            time_period varchar(255) DEFAULT NULL,
            compare_period varchar(255) DEFAULT NULL,
            delivery_channels text NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Fetch paginated notifications.
     *
     * @param int $page
     * @param int $per_page
     * @return array
     */
    public static function get_notifications_paginated($page = 1, $per_page = 5000, $type = 'all')
    {
        global $wpdb;

        $offset = ($page - 1) * $per_page;

        $query = $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}monitori_notifications ORDER BY id DESC LIMIT %d OFFSET %d",
            $per_page,
            $offset
        );

        if ($type === 'event' || $type === 'metric') {
            $query = $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}monitori_notifications WHERE notification_type = %s ORDER BY id DESC LIMIT %d OFFSET %d",
                $type,
                $per_page,
                $offset
            );
        }


        $notifications = $wpdb->get_results($query);
        $total_records = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}monitori_notifications");
        $total_pages = ceil($total_records / $per_page);
        
        return [
            'notifications' => $notifications,
            'pagination' => [
                'total_records' => $total_records,
                'total_pages' => $total_pages,
                'current_page' => $page,
                'per_page' => $per_page,
            ],
        ];
    }

    /**
     * Fetch a single notification by ID.
     *
     * @param int $id
     * @return object|null
     */
    public static function get_notification_by_id($id)
    {
        global $wpdb;

        $query = $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}monitori_notifications WHERE id = %d LIMIT 1",
            $id
        );

        return $wpdb->get_row($query);
    }

    /**
     * Add a new notification.
     *
     * @param string $name
     * @param string $notification_type
     * @param string $delivery_channels
     * @param array $settings
     * @return bool|int
     */
    public static function add_notification($name, $notification_type, $delivery_channels, $settings = [])
    {
        global $wpdb;

        $data = array_merge([
            'name' => sanitize_text_field($name),
            'notification_type' => sanitize_text_field($notification_type),
            'delivery_channels' => wp_json_encode($delivery_channels),
        ], $settings);
        
        do_action('monitori_save_notification');

        return $wpdb->insert($wpdb->prefix . 'monitori_notifications', $data);
    }

    /**
     * Update an existing notification.
     *
     * @param int $id
     * @param array $fields
     * @return bool|int
     */
    public static function update_notification($id, $name, $notification_type, $delivery_channels, $settings = [])
    {
        global $wpdb;


        $data = array_merge([
            'name' => sanitize_text_field($name),
            'notification_type' => sanitize_text_field($notification_type),
            'delivery_channels' => wp_json_encode($delivery_channels),
        ], $settings);

        do_action('monitori_save_notification');
        
        return $wpdb->update($wpdb->prefix . 'monitori_notifications', $data, ['id' => absint($id)]);
    }

    /**
     * Remove a notification by ID.
     *
     * @param int $id
     * @return bool|int
     */
    public static function remove_notification($id)
    {
        global $wpdb;

        return $wpdb->delete($wpdb->prefix . 'monitori_notifications', ['id' => absint($id)]);
    }
}
