<?php

namespace Monitori;

if ( !defined( 'ABSPATH' ) ) {
    exit;
    // Exit if accessed directly
}
class Ajax_Notifications {
    public function __construct() {
        add_action( 'wp_ajax_monitori_get_notifications', [$this, 'get_notifications'] );
        add_action( 'wp_ajax_monitori_get_notification', [$this, 'get_notification'] );
        add_action( 'wp_ajax_monitori_add_notification', [$this, 'add_notification'] );
        add_action( 'wp_ajax_monitori_update_notification', [$this, 'update_notification'] );
        add_action( 'wp_ajax_monitori_remove_notification', [$this, 'remove_notification'] );
        add_action( 'wp_ajax_monitori_get_edit_fields', [$this, 'get_edit_fields'] );
    }

    private function verify_nonce( $action ) {
        $nonce = wp_unslash( $_POST['_wpnonce'] ?? '' );
        if ( !wp_verify_nonce( $nonce, $action ) ) {
            wp_send_json_error( [
                'message' => 'Invalid nonce',
            ], 403 );
        }
    }

    public function get_notifications() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        $page = absint( $_POST['page'] ?? 1 );
        $per_page = absint( $_POST['per_page'] ?? 10 );
        $notifications = Database::get_notifications_paginated( $page, $per_page );
        wp_send_json_success( $notifications );
    }

    public function get_notification() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        $id = absint( wp_unslash( $_POST['id'] ) );
        $notification = Database::get_notification_by_id( $id );
        wp_send_json_success( $notification );
    }

    public function add_notification() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        $name = sanitize_text_field( wp_unslash( $_POST['name'] ?? '' ) );
        $notification_type = sanitize_text_field( wp_unslash( $_POST['notification_type'] ?? 'event' ) );
        $event_type = sanitize_text_field( wp_unslash( $_POST['event_type'] ?? 'event' ) );
        $change_type = sanitize_text_field( wp_unslash( $_POST['change_type'] ?? '' ) );
        $metric_value = sanitize_text_field( wp_unslash( $_POST['metric_value'] ?? '' ) );
        $metric_name = sanitize_text_field( wp_unslash( $_POST['metric_name'] ?? '' ) );
        $time_period = sanitize_text_field( wp_unslash( $_POST['time_period'] ?? 'daily' ) );
        $compare_period = sanitize_text_field( wp_unslash( $_POST['compare_period'] ?? 'last_period' ) );
        $delivery_channels = $this->sanitize_delivery_channels( wp_unslash( $_POST['delivery_channels'] ?? '' ) );
        $settings = [
            'event_type'     => $event_type,
            'change_type'    => $change_type,
            'metric_name'    => $metric_name,
            'metric_value'   => $metric_value,
            'time_period'    => $time_period,
            'compare_period' => $compare_period,
        ];
        if ( empty( $name ) ) {
            wp_send_json_error( [
                'message' => 'Name is required',
            ], 400 );
        }
        $result = Database::add_notification(
            $name,
            $notification_type,
            $delivery_channels,
            $settings
        );
        if ( !$result ) {
            wp_send_json_error( [
                'message' => 'Failed to add notification',
            ], 500 );
        }
        wp_send_json_success( [
            'message' => 'Notification added successfully',
        ] );
    }

    public function update_notification() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        $id = absint( wp_unslash( $_POST['id'] ?? 0 ) );
        $name = sanitize_text_field( wp_unslash( $_POST['name'] ?? '' ) );
        $notification_type = sanitize_text_field( wp_unslash( $_POST['notification_type'] ?? 'event' ) );
        $event_type = sanitize_text_field( wp_unslash( $_POST['event_type'] ?? 'event' ) );
        $change_type = sanitize_text_field( wp_unslash( $_POST['change_type'] ?? '' ) );
        $metric_value = sanitize_text_field( wp_unslash( $_POST['metric_value'] ?? '' ) );
        $metric_name = sanitize_text_field( wp_unslash( $_POST['metric_name'] ?? '' ) );
        $time_period = sanitize_text_field( wp_unslash( $_POST['time_period'] ?? 'daily' ) );
        $compare_period = sanitize_text_field( wp_unslash( $_POST['compare_period'] ?? 'last_period' ) );
        $delivery_channels = $this->sanitize_delivery_channels( wp_unslash( $_POST['delivery_channels'] ?? '' ) );
        $settings = [
            'event_type'     => $event_type,
            'change_type'    => $change_type,
            'metric_name'    => $metric_name,
            'metric_value'   => $metric_value,
            'time_period'    => $time_period,
            'compare_period' => $compare_period,
        ];
        if ( empty( $id ) ) {
            wp_send_json_error( [
                'message' => 'ID is required',
            ], 400 );
        }
        $result = Database::update_notification(
            $id,
            $name,
            $notification_type,
            $delivery_channels,
            $settings
        );
        if ( !$result ) {
            wp_send_json_error( [
                'message' => 'Failed to edit notification',
            ], 500 );
        }
        wp_send_json_success( [
            'message' => 'Notification updated successfully',
        ] );
    }

    public function remove_notification() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        $id = absint( wp_unslash( $_POST['id'] ) );
        if ( empty( $id ) ) {
            wp_send_json_error( [
                'message' => 'ID is required',
            ], 400 );
        }
        $result = Database::remove_notification( $id );
        if ( !$result ) {
            wp_send_json_error( [
                'message' => 'Failed to delete notification',
            ], 500 );
        }
        wp_send_json_success( [
            'message' => 'Notification deleted successfully',
        ] );
    }

    public function get_edit_fields() {
        $this->verify_nonce( 'monitori_notifications_nonce' );
        if ( !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [
                'message' => 'Unauthorized',
            ], 403 );
        }
        try {
            // Fetch options dynamically
            $event_types = \Monitori\Event_Manager::get_option_values();
            $metric_types = \Monitori\Metric_Manager::get_option_values();
            $metric_change_types = \Monitori\Base_Metric_Integration::get_metric_change_types();
            $event_preview_details = \Monitori\Event_Manager::get_preview_details();
            // Predefined options
            $notification_types = [
                'event'  => __( 'Event', 'monitori' ),
                'metric' => __( 'Metric', 'monitori' ),
            ];
            $time_periods = [
                'daily'   => __( 'Daily', 'monitori' ),
                'weekly'  => __( 'Weekly', 'monitori' ),
                'monthly' => __( 'Monthly', 'monitori' ),
            ];
            $compare_periods = [
                'last_period' => __( 'Last period', 'monitori' ),
                'last_year'   => __( 'Last year', 'monitori' ),
            ];
            $delivery_channels = [
                'email' => [
                    'label' => __( 'Email', 'monitori' ),
                ],
                'slack' => [
                    'label'  => __( 'Slack', 'monitori' ),
                    'fields' => \Monitori\Slack_Setup::get_slack_fields_api(),
                ],
            ];
            // Return fields
            wp_send_json_success( [
                'notification_types' => $notification_types,
                'event_types'        => $event_types,
                'metric_types'       => $metric_types,
                'change_types'       => $metric_change_types,
                'time_periods'       => $time_periods,
                'compare_periods'    => $compare_periods,
                'delivery_channels'  => $delivery_channels,
                'preview_details'    => [
                    'events' => $event_preview_details ?? [],
                ],
            ] );
        } catch ( \Exception $e ) {
            wp_send_json_error( [
                'message' => 'Failed to fetch fields',
                'error'   => $e->getMessage(),
            ], 500 );
        }
    }

    /**
     * Sanitize delivery channels by decoding JSON and sanitizing values
     * @since 0.1.3
     * 
     * return array - sanitized delivery channels
     */
    public function sanitize_delivery_channels( $delivery_channels ) {
        $delivery_channels = json_decode( $delivery_channels, true );
        // Validate decoded JSON
        if ( json_last_error() !== JSON_ERROR_NONE || !is_array( $delivery_channels ) ) {
            wp_send_json_error( [
                'message' => 'Invalid delivery channels format',
            ], 400 );
        }
        // Additional sanitization for delivery channels
        array_walk_recursive( $delivery_channels, function ( &$value ) {
            $value = sanitize_text_field( $value );
        } );
        return $delivery_channels;
    }

}
