<?php

namespace MonieswitchWP\WooCommerce;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;


class PaymentButton {
  
    public $button_id;
  
    public $amount;
    
    public $reference;
    
    public $payment_shortcode_arrays = [];
  
    public function __construct()
    {
        add_shortcode('monieswitch', [$this, 'monieswitch_payment_button']);
        
        if(!is_admin()) {
            add_action('wp_footer', [$this, 'add_popup_footer']);
        }
    
        add_action('wp_ajax_monieswitch_custom_payment_handler', [$this, 'monieswitch_custom_payment_handler']);
        add_action('wp_ajax_nopriv_monieswitch_custom_payment_handler', [$this, 'monieswitch_custom_payment_handler']);
    }
    
    public function monieswitch_custom_payment_handler() {
        // Initialize result array
        $result = array(
            'status' => false,
            'message' => ''
        );
        
        // Verify button_id exists and sanitize it
        if (!isset($_POST['button_id'])) {
            $result['message'] = esc_html__('Button ID is missing', 'monieswitch');
            echo wp_json_encode($result);
            wp_die();
        }
        
        $button_id = sanitize_text_field(wp_unslash($_POST['button_id']));
        
        // Verify nonce exists and validate it
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'monieswitch-woocommerce-payment-'.$button_id)) {
            $result['message'] = esc_html__('Payment initialization failed','monieswitch');
            echo wp_json_encode($result);
            wp_die();
        }
        
        // Verify email exists and validate it
        if (!isset($_POST['email']) || !filter_var(sanitize_email(wp_unslash($_POST['email'])), FILTER_VALIDATE_EMAIL)) {
            $result['message'] = esc_html__('Email address not valid', 'monieswitch');
            echo wp_json_encode($result);
            wp_die();
        }
        
        // Process only the data we need instead of the entire $_POST array
        $sanitized_data = array();
        $allowed_fields = array('button_id', 'email', 'nonce');
        
        foreach ($allowed_fields as $field) {
            if (isset($_POST[$field])) {
                $sanitized_data[$field] = sanitize_text_field(wp_unslash($_POST[$field]));
            }
        }
        
        // If phone number is provided, sanitize it
        if (isset($_POST['phone'])) {
            $sanitized_data['phone'] = sanitize_text_field(wp_unslash($_POST['phone']));
        }
        
        $result['status'] = true;
        $result['message'] = esc_html__('Payment initialization successful', 'monieswitch');
        $result['data'] = $sanitized_data;
    
        echo wp_json_encode($result);
        wp_die();
    }
    
    public function monieswitch_payment_button($atts, $content) {
        $monieswitch_settings = get_option('woocommerce_monieswitch_woocommerce_settings');

        //set up parameters
        extract(shortcode_atts(array(
            'type' => 'primary',
            'amount' => '',
            'currency' => get_woocommerce_currency(),
            'channels'  => json_encode(explode(',', $monieswitch_settings['channels'])), // BANK, CARD
            'bearer'  => $monieswitch_settings['bearer'],
            'subaccount'  => $monieswitch_settings['subaccount']
        ), $atts));
        
        if(empty($content)) {
          $content = esc_html__('Pay Now', 'monieswitch');
        }
    
        $monieswitch_settings = get_option('woocommerce_monieswitch_woocommerce_settings');
    
        if(empty($monieswitch_settings['public_key'])) {
            if(current_user_can( 'manage_options' )) {
                return wp_kses('<span style="color: red;">' . esc_html__('Public key is missing, kindly add your public key', 'monieswitch') . '</span>', array('span' => array('style' => array())));
            } else {
              return '';
            }
        }
    
        if(empty($amount)) {
            if(current_user_can( 'manage_options' )) {
                return wp_kses('<span style="color: red;">' . esc_html__('No amount found, kindly specify the amount', 'monieswitch') . '</span>', array('span' => array('style' => array())));
            } else {
                return '';
            }
        }
        
        if(empty($reference)) {
            $reference = 'MS_'.$this->generate_transaction_ref();
        }
        
        ob_start();
        
        $button_id = $this->generate_transaction_ref();
        
        $this->button_id = $button_id;
        $this->amount = $amount;
        $this->reference = $reference;
        
        $this->payment_shortcode_arrays[] = ['button_id' => $button_id, 'amount' => $amount, 'reference' => $reference];
    
        wp_enqueue_script( 'jquery' );
    
        wp_enqueue_script( 'monieswitch', esc_url('https://collect.monieswitch.com/pay.js'), array( 'jquery' ), MONIESWITCH_WP_VERSION_NUMBER, false );
    
        wp_enqueue_script( 'monieswitch-custom', plugins_url( 'assets/js/monieswitch.js', MONIESWITCH_WP_SYSTEM_FILE_PATH ), [ 'jquery', 'monieswitch' ], MONIESWITCH_WP_VERSION_NUMBER, true );
    
        wp_enqueue_style( 'monieswitch-custom', plugins_url( 'assets/css/monieswitch.css', MONIESWITCH_WP_SYSTEM_FILE_PATH ), [], MONIESWITCH_WP_VERSION_NUMBER);

        $monieswitch_params = [
            'ajax_url'  => admin_url('admin-ajax.php'),
            'publicKey' => $monieswitch_settings['public_key'],
            'currency' => $currency,
            'subaccountId' => $subaccount,
            'bearer' => $bearer,
            'channels' => $channels
        ];

        wp_localize_script( 'monieswitch-custom', 'wc_monieswitch_params', $monieswitch_params );

        ?>
        <button id="monieswitch_custom_payment_button-<?php echo esc_attr($button_id) ?>" type="button" class="monieswitch-custom-payment-button <?php echo esc_attr($type) ?>" data-custom-popup="<?php echo esc_attr('monieswitch_'.$button_id); ?>" style="display: none"><?php echo esc_html($content); ?></button>
        <?php
        return ob_get_clean();
    }
    
    public function add_popup_footer() {
      ob_start();
      
      foreach($this->payment_shortcode_arrays as $key => $value) {
          $button_id = $value['button_id'];
          $amount = $value['amount'];
          $reference = $value['reference'];
          $formatted_amount = $this->format_amount($amount);
    
          $nonce = wp_create_nonce('monieswitch-woocommerce-payment-'.$button_id);
      ?>
      <section class="monieswitch_custom_payment_container <?php echo esc_attr($button_id); ?>" id="<?php echo esc_attr('monieswitch_'.$button_id); ?>" style="display: none">
        <div class="monieswitch-monieswitch-container">
          <button class="monieswitch-close-button" type="button">X</button>
          <div class="monieswitch-title">
            <h4><?php 
              /* translators: %s: Formatted amount with currency symbol */
              printf(esc_html__('Pay with Monieswitch - %s', 'monieswitch'), wp_kses_post($formatted_amount)); 
            ?></h4>
          </div>
          <hr />
          <form action="" id="form-<?php echo esc_attr($button_id); ?>" method="post" class="monieswitch-payment-form" data-form-reference="<?php echo esc_attr($reference); ?>">
              <div class="input-group">
                <input name="monieswitch-email" type="email" required placeholder="<?php echo esc_attr__('Enter Your Email', 'monieswitch') ?>" class="input-control" value="" />
              </div>
              <div class="input-group">
                <input name="monieswitch-phone-number" type="tel" placeholder="<?php echo esc_attr__('Enter Your Phone Number (e.g 0812345678)', 'monieswitch') ?>" class="input-control" />
              </div>
              <div class="input-group">
                <input type="submit" value="<?php echo esc_attr__('Proceed', 'monieswitch') ?>" class="input-submit monieswitch-submit" />
              </div>
              <input name="monieswitch-nonce" type="hidden" value="<?php echo esc_attr($nonce) ?>" />
              <input name="monieswitch-button-id" type="hidden" value="<?php echo esc_attr($button_id) ?>" />
              <input name="monieswitch-amount" type="hidden" value="<?php echo esc_attr($amount) ?>" />
            </form>
          <div class="notices-alert error"></div>
          <div class="notices-alert success"></div>
        </div>
      </section>
      <?php
      }
      
      echo wp_kses_post(ob_get_clean());
    }
    
    public function format_amount($amount) {
      return wp_kses('&#x20A6;', array()) . number_format( intval($amount), 2, '.', ',');
    }
    
    public function generate_transaction_ref($length = 3) {
        $random = "";
        $char_list = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890";
        $max = strlen($char_list) - 1;
        
        for($i = 0; $i < $length; $i++) {
            $random .= $char_list[wp_rand(0, $max)];
        }
        $prefix = "PB_";
        return $prefix. $random . '_'.time();
    }
    
    /**
     * @return PaymentButton|null
     */
    public static function get_instance()
    {
        static $instance = null;
        
        if (is_null($instance)) {
            $instance = new self();
        }
        
        return $instance;
    }
    
    
}