<?php

namespace MonieswitchWP\WooCommerce;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

use WpOrg\Requests\Exception;

class Payment extends \WC_Payment_Gateway {
    
    public $public_key;

    public $secret_key;

    public $customer_phone;

    public $bearer;

    public $subaccount;

    public $channels;

    public $payment_page;

    public $msg;

    public $logger;

    public $context;
    
    public function __construct()
    {
        $this->id                 = 'monieswitch_woocommerce';
        $this->icon               = apply_filters('woocommerce_monieswitch_woocommerce_payment_icon', '');
        $this->has_fields         = true;
        $this->enabled            = 'no';
        $this->method_title       = __( 'Monieswitch', 'monieswitch' );
        $this->method_description = __( 'Monieswitch Payment Gateway provide merchants with the tools and services needed to accept online payments from local and international customers using Mastercard, Visa, Verve Cards and Bank Accounts', 'monieswitch' );
    
        // Load the settings.
        $this->init_form_fields();
        $this->init_settings();
    
        // Define user set variables
        $this->title            = $this->get_option( 'title' );
        $this->description      = $this->get_option( 'description' );
        $this->public_key        = $this->get_option('public_key');
        $this->secret_key        = $this->get_option('secret_key');
        $this->customer_phone   = $this->get_option('customer_phone');
        $this->bearer   = $this->get_option('bearer');
        $this->subaccount   = $this->get_option('subaccount');
        $this->payment_page     = $this->get_option( 'payment_page' );
        $this->channels     = $this->get_option( 'channels' );

        $this->logger = wc_get_logger();
        $this->context = ['source' => $this->id];

        add_action('woocommerce_update_options_payment_gateways_'.$this->id, [$this, 'process_admin_options']);
    
        add_action( 'woocommerce_receipt_' . $this->id, array( $this, 'receipt_page' ) );
    
        add_action( 'wp_enqueue_scripts', [ $this, 'payment_scripts' ] );
        
        add_action( 'woocommerce_api_monieswitch_woocommerce_gateway', array( $this, 'verify_monieswitch_woocommerce_transaction' ) );

        add_action('admin_enqueue_scripts', [$this, 'custom_monieswitch_woocommerce_admin_script']);

        // Check if the gateway can be used.
        if ( ! $this->is_valid_for_use() ) {
            $this->enabled = false;
        }
    }
    
    /**
     * Check if this gateway is enabled and available in the user's country.
     */
    public function is_valid_for_use() {
        
        if ( ! in_array( get_woocommerce_currency(), apply_filters( 'woocommerce_monieswitch_supported_currencies', array( 'NGN', 'USD') ) ) ) {
            
            /* translators: %s: URL to WooCommerce currency settings */
            $this->msg = sprintf( __( 'Monieswitch does not support your store currency. Kindly set it to either NGN (&#8358) or USD ($) <a href="%s">here</a>', 'monieswitch' ), admin_url( 'admin.php?page=wc-settings&tab=general' ) );
            
            return false;
            
        }
        
        return true;
        
    }
    
    public function verify_monieswitch_woocommerce_transaction() {
        // Verify nonce if present
        if (isset($_REQUEST['nonce']) && isset($_REQUEST['monieswitch_reference'])) {
            $monieswitch_reference = sanitize_text_field(wp_unslash($_REQUEST['monieswitch_reference']));
            $nonce = sanitize_text_field(wp_unslash($_REQUEST['nonce']));
            
            // Verify the nonce
            if (!wp_verify_nonce($nonce, $monieswitch_reference)) {
                // Invalid nonce
                wp_die(esc_html__('Security check failed', 'monieswitch'), esc_html__('Security Error', 'monieswitch'), array('response' => 403));
            }
        } elseif (isset($_REQUEST['monieswitch_reference'])) {
            // For backward compatibility or direct API callbacks
            $monieswitch_reference = sanitize_text_field(wp_unslash($_REQUEST['monieswitch_reference']));
        } else {
            $monieswitch_reference = false;
        }
    
        @ob_clean();
        
        if($monieswitch_reference) {

            $order_details = explode( '_', $monieswitch_reference );

            $order_id = (int) $order_details[0];

            $order = wc_get_order($order_id);

            if(!empty($order_id)) {
                if ( in_array( $order->get_status(), array( 'processing', 'completed', 'on-hold' ) ) ) {
                    wp_redirect( $this->get_return_url( $order ) );
                    exit;
                }

                $order->payment_complete( $monieswitch_reference );
                /* translators: %s: Transaction reference */
                $order->add_order_note( sprintf( esc_html__( 'Payment via Monieswitch was successful (Transaction Reference: %s)', 'monieswitch' ), esc_html($monieswitch_reference) ) );

                if ( $this->is_autocomplete_order_enabled( $order ) ) {
                    $order->update_status( 'completed' );
                }
            }

            wp_redirect( $this->get_return_url( $order ) );
            exit;
          
        }
    }
    
    protected function is_autocomplete_order_enabled( $order ) {
        $autocomplete_order = false;
        
        $payment_method = $order->get_payment_method();
        
        $monieswitch_settings = get_option('woocommerce_' . $payment_method . '_settings');
        
        if ( isset( $monieswitch_settings['autocomplete_order'] ) && 'yes' === $monieswitch_settings['autocomplete_order'] ) {
            $autocomplete_order = true;
        }
        
        return $autocomplete_order;
    }
    
    /**
     * Check if Monieswitch merchant details is filled.
     */
    public function admin_notices() {
        
        if ( $this->enabled == 'no' ) {
            return;
        }
        
        // Check required fields.
        if (empty($this->public_key) ) {
            /* translators: %s: URL to Monieswitch settings page */
            $error_html = '<div class="error"><p>' . sprintf( wp_kses( __( 'Please enter your Monieswitch merchant details <a href="%s">here</a> to be able to use the Monieswitch for WooCommerce.', 'monieswitch' ), array('a' => array('href' => array())) ), esc_url(admin_url( 'admin.php?page=wc-settings&tab=checkout&section=monieswitch_woocommerce' )) ) . '</p></div>';
            echo wp_kses( $error_html, array(
                'div' => array('class' => array()),
                'p' => array(),
                'a' => array('href' => array())
            ) );
            return;
        }
        
    }
    
    /**
     * Check if Monieswitch payment gateway is enabled.
     *
     * @return bool
     */
    public function is_available() {
        
        if ( 'yes' == $this->enabled ) {
            
            if (empty($this->public_key) ) {
                return false;
            }
            return true;
        }
        
        return false;
        
    }
    
    /**
     * Admin Panel Options.
     */
    public function admin_options() {
        
        ?>
        
        <h2><?php esc_html_e( 'Monieswitch', 'monieswitch' ); ?>
            <?php
                if ( function_exists( 'wc_back_link' ) ) {
                    wc_back_link( esc_html__( 'Return to payments', 'monieswitch' ), esc_url(admin_url( 'admin.php?page=wc-settings&tab=checkout' )) );
                }
            ?>
        </h2>
        
        <?php
        
        if ( $this->is_valid_for_use() ) {
            
            echo wp_kses('<table class="form-table">', array('table' => array('class' => array())));
            $this->generate_settings_html();
            echo wp_kses('</table>', array('table' => array()));
            
        } else {
            ?>
            <div class="inline error"><p><strong><?php esc_html_e( 'Monieswitch Payment Gateway Disabled', 'monieswitch' ); ?></strong>: <?php echo wp_kses_post($this->msg); ?></p></div>
            
            <?php
        }
        
    }
    
    public function init_form_fields() {
        $this->form_fields = apply_filters('wc_monieswitch_payment_form_fields', [
            'enabled' => [
                'title'   => esc_html__( 'Enable/Disable', 'monieswitch' ),
                'type'    => 'checkbox',
                'label'   => esc_html__( 'Enable Monieswitch', 'monieswitch' ),
                'default' => 'no'
            ],
            'title' => [
                'title'       => esc_html__( 'Title', 'monieswitch' ),
                'type'        => 'text',
                'description' => esc_html__( 'This controls the title for the payment method the customer sees during checkout.', 'monieswitch' ),
                'default'     => esc_html__( 'Pay with Monieswitch', 'monieswitch' ),
                'desc_tip'    => true,
            ],
            'description' =>    [
                'title'       => esc_html__( 'Description', 'monieswitch' ),
                'type'        => 'textarea',
                'description' => esc_html__( 'This controls the payment method description which the user sees during checkout.', 'monieswitch' ),
                'default'     => esc_html__( 'Make payment using Monieswitch', 'monieswitch' ),
                'desc_tip'    => true,
            ],
            'payment_page' => [
                'title'         => esc_html__( 'Payment Option', 'monieswitch' ),
                'type'          => 'select',
                'description'   => esc_html__( 'Popup shows the payment popup on the page while Redirect will redirect the customer to an external Monieswitch page to make payment.', 'monieswitch' ),
                'options'       => [
                    'client'     => esc_html__('Client Side', 'monieswitch'),
                    'server'     => esc_html__('Server Side', 'monieswitch'),
                    'external'   => esc_html__('External Link', 'monieswitch'),
                ],
                'default'       => 'popup',
                'desc_tip'      => true,
            ],
            'public_key' => [
                'title'       => esc_html__( 'Public Key', 'monieswitch' ),
                'type'        => 'text',
                'description' => esc_html__( 'Enter your Public Key here.', 'monieswitch' ),
                'desc_tip'    => true,
            ],
            'secret_key' => [
                'title'       => esc_html__( 'Secret Key', 'monieswitch' ),
                'type'        => 'password',
                'description' => esc_html__( 'Enter your Secret Key here.', 'monieswitch' ),
                'desc_tip'    => true,
            ],
            'autocomplete_order' => [
                'title'         => esc_html__( 'Autocomplete Order', 'monieswitch' ),
                'type'          => 'checkbox',
                'label'         => esc_html__( 'If enabled, the order will be marked as complete after successful payment', 'monieswitch' ),
                'default'       => 'no',
                'desc_tip'      => true,
            ],
            'channels' => [
                'title'       => esc_html__( 'Channels', 'monieswitch' ),
                'type'          => 'select',
                'options'       => [
                    ''     => esc_html__('Select one', 'monieswitch'),
                    'BANK'     => esc_html__('Bank Transfer', 'monieswitch'),
                ],
                'default'       => 'BANK',
                'desc_tip'      => true,
                'required'      => true,
            ],
            'customer_phone' => [
                'title'       => esc_html__( 'Customer\'s Phone', 'monieswitch' ),
                'type'        => 'checkbox',
                'label'       => esc_html__( 'Send the Customer\'s Phone.', 'monieswitch' ),
                'default'       => 'no',
            ],
            'bearer' => [
                'title'       => esc_html__( 'Bearer', 'monieswitch' ),
                'type'          => 'select',
                'options'       => [
                    ''          => esc_html__('Select one', 'monieswitch'),
                    'account'     => esc_html__('Account', 'monieswitch'),
                    'subaccount'     => esc_html__('Subaccount', 'monieswitch'),
                    'customer'     => esc_html__('Customer', 'monieswitch'),
                ],
                'default'       => 'account',
                'desc_tip'      => true,
                'required'      => true,
            ],
            'subaccount' => [
                'title'       => esc_html__( 'Subaccount', 'monieswitch' ),
                'type'        => 'tel',
                'desc_tip'    => true,
            ],
        ]);
    }
    
    
    public function process_payment($order_id)
    {
        $order = wc_get_order( $order_id );
    
        return array(
            'result'   => 'success',
            'redirect' => $order->get_checkout_payment_url( true ),
        );
    }

    public function custom_monieswitch_woocommerce_admin_script($hook)
    {
        // Verify we're on the right page and section
        if ('woocommerce_page_wc-settings' === $hook && isset($_GET['section']) && sanitize_text_field(wp_unslash($_GET['section'])) === $this->id) {
            // Note: WooCommerce settings page already has its own nonce verification
            // This is an admin page that's already protected by WooCommerce
            wp_enqueue_script(
                'monieswitch-admin', plugins_url( 'assets/js/monieswitch-admin.js',  MONIESWITCH_WP_SYSTEM_FILE_PATH ), ['jquery'], MONIESWITCH_WP_VERSION_NUMBER, true
            );
        }

    }
    
    
    public function payment_scripts() {
        if ( ! is_checkout_pay_page() ) {
            return;
        }
    
        if ( $this->enabled == 'no' ) {
            return;
        }

        // Validate, unslash, and sanitize $_GET['key']
        if ( isset( $_GET['key'] ) ) {
            $order_key = sanitize_text_field( wp_unslash( $_GET['key'] ) );
            // Note: We don't need to urldecode after sanitize_text_field as it already handles this
        } else {
            // Handle missing key gracefully
            return;
        }
        $order_id  = absint( get_query_var( 'order-pay' ) );
    
        $order = wc_get_order( $order_id );
    
        $payment_method = method_exists( $order, 'get_payment_method' ) ? $order->get_payment_method() : $order->payment_method;
    
        if ( $this->id !== $payment_method ) {
            $this->logger->error('Monieswitch payment method is not valid', $this->context);
            return;
        }
        
        if(empty($this->public_key)) {
            $this->logger->error('Public key is missing', $this->context);
            return;
        }

        if(empty($this->secret_key) && $this->payment_page == 'external') {
            $this->logger->error('Secret key is missing: External payment option requires using secret key', $this->context);
            return;
        }

        try {
            $txnref = $order_id . '_' . time();
            
            wp_enqueue_script( 'jquery' );

            wp_enqueue_script( 'monieswitch', esc_url('https://collect.monieswitch.com/pay.js'), array('jquery'), MONIESWITCH_WP_VERSION_NUMBER, false );

            wp_enqueue_script( 'monieswitch-custom', plugins_url( 'assets/js/monieswitch.js',  MONIESWITCH_WP_SYSTEM_FILE_PATH ), array( 'jquery', 'monieswitch' ), MONIESWITCH_WP_VERSION_NUMBER, true );

            $monieswitch_params = [];

            if ( is_checkout_pay_page() && get_query_var( 'order-pay' ) ) {

                $email = method_exists( $order, 'get_billing_email' ) ? $order->get_billing_email() : $order->billing_email;

                $amount = (float) $order->get_total();

                $the_order_id  = method_exists( $order, 'get_id' ) ? $order->get_id() : '';
                $the_order_key = method_exists( $order, 'get_order_key' ) ? $order->get_order_key() : $order->order_key;

                $billing_phone = method_exists( $order, 'get_billing_phone' ) ? $order->get_billing_phone() : $order->billing_phone;

                if ( $the_order_id == $order_id && $the_order_key == $order_key ) {

                    $monieswitch_params['email']    = $email;
                    $monieswitch_params['amount']   = $amount;
                    $monieswitch_params['reference']   = $txnref;
                }

                if($this->customer_phone !== 'no') {
                    $monieswitch_params['phoneNumber']   = $billing_phone;
                }

                if(!empty($this->subaccount)) {
                    $monieswitch_params['subaccountId']   = $this->subaccount;
                }

                if(!empty($this->bearer)) {
                    $monieswitch_params['bearer']   = $this->bearer;
                }

                if(!empty($this->channels)) {
                    $monieswitch_params['channels']   = explode(',', $this->channels);
                }

                $monieswitch_params['currency']   = $order->get_currency();

                update_post_meta( $order_id, '_monieswitch_woocommerce_txn_ref', $txnref );
            }

            $monieswitch_params['publicKey'] = $this->public_key;
            $monieswitch_params['payment_page'] = $this->payment_page;
            $monieswitch_params['redirectURL'] = esc_url(add_query_arg(['action' => 'monieswitch_woocommerce_callback', 'nonce' => wp_create_nonce($txnref)], admin_url( 'admin-ajax.php' )));

            // payment options
            if($monieswitch_params['payment_page'] == 'external' || $monieswitch_params['payment_page'] == 'server') {

                $checkout_initialize = API::get_instance()->generate_checkout_code($monieswitch_params);

                if(isset($checkout_initialize['status']) && $checkout_initialize['status']) {
                    $monieswitch_params['order_id'] = $order_id;
                    $monieswitch_params['data'] = $checkout_initialize['data'];

                    wp_localize_script( 'monieswitch-custom', 'wc_monieswitch_params', $monieswitch_params );
                } else {
                    throw new \Exception( $checkout_initialize['message'] );
                }

            } else {
                wp_localize_script( 'monieswitch-custom', 'wc_monieswitch_params', $monieswitch_params );
            }

        } catch (\Exception $e) {
            $this->logger->error($e->getMessage(), $this->context);
        }
        
    }
    
    public function append_button_footer() {
    
    }
    
    /**
     * Displays the payment page.
     *
     * @param $order_id
     */
    public function receipt_page( $order_id ) {
        
        $order = wc_get_order( $order_id );
        
        echo '<div id="monieswitch-form">';
        
        echo '<p>' . esc_html__( 'Thank you for your order, please click the button below to pay with Monieswitch.', 'monieswitch' ) . '</p>';
        
        // Create the form HTML with proper escaping for text but not for the HTML structure
        // This approach bypasses wp_kses() for the HTML structure while still safely escaping dynamic content
        $error_message = esc_html__('There was an error initializing this payment, please try again.', 'monieswitch');
        $pay_now_text = esc_html__( 'Pay Now', 'monieswitch' );
        $form_action = esc_url(WC()->api_request_url('Monieswitch_Woocommerce_Gateway'));
        
        // Output the HTML directly without wp_kses filtering
        echo <<<HTML
        <div id="monieswitch_woocommerce_form">
            <p class="monieswitch_error_message" style="display: none;margin:0;color:red;font-size: 14px;">{$error_message}</p>
            <form id="order_review" method="post" action="{$form_action}"></form>
            <button class="button" id="monieswitch-payment-button" style="display: none;">{$pay_now_text}</button>
        </div>
        HTML;
        
        echo wp_kses('</div>', array('div' => array()));
        
    }
    
    
    /**
     * @return Payment|null
     */
    public static function get_instance()
    {
        static $instance = null;
        
        if (is_null($instance)) {
            $instance = new self();
        }
        
        return $instance;
    }
}