<?php

namespace MonieswitchWP\WooCommerce;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;


class API {

    public $id = 'monieswitch_woocommerce';

    public $backend_api_url = 'https://nini.monieswitch.com';

    public $headers;

    public $secret_keys;

    public $is_autocomplete_order_enabled;


    public function __construct()
    {
        $monieswitch_settings = get_option('woocommerce_monieswitch_woocommerce_settings');

        $this->secret_keys = $monieswitch_settings['secret_key'];

        $this->is_autocomplete_order_enabled = isset( $monieswitch_settings['autocomplete_order'] ) && 'yes' === $monieswitch_settings['autocomplete_order'];

        $this->headers = [
            'Authorization' => sprintf('Bearer %s', $this->secret_keys),
            'Content-Type'  => 'application/json',
        ];

        add_action( 'wp_ajax_monieswitch_woocommerce_callback', array( $this, 'monieswitch_woocommerce_callback' ) );
    }

    public function monieswitch_woocommerce_callback()
    {
        $logger = wc_get_logger();
        $context = ['source' => $this->id];

        if(isset($_REQUEST['reference'])) {
            $monieswitch_reference = sanitize_text_field( wp_unslash($_REQUEST['reference']) );
        } else {
            $monieswitch_reference = false;
        }

        if($monieswitch_reference) {
            $order_details = explode( '_', $monieswitch_reference );

            $order_id = (int) $order_details[0];

            $order = wc_get_order($order_id);

            $failed_url = add_query_arg(['monieswitch-status' => 'failed'], $order->get_checkout_payment_url(true));

            if(!isset($_REQUEST['nonce'])) {
                $logger->error('Unable to authenticate callback URL: Either reference or security nonce is missing', $context);
                wp_redirect($failed_url);
                wp_die();
            }

            if(!isset($_REQUEST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['nonce'])), $monieswitch_reference)) {
                $logger->error('Security nonce verification failed', $context);
                wp_redirect($failed_url);
                wp_die();
            }

            try {

                $checkout_initialize = $this->verify_payment_transaction_by_reference($monieswitch_reference);

                if(isset($checkout_initialize['status']) && $checkout_initialize['status']) {

                    if ( in_array( $order->get_status(), array( 'processing', 'completed', 'on-hold' ) ) ) {
                        wp_redirect( $order->get_checkout_order_received_url() );
                        wp_die();
                    }

                    $order->payment_complete( $monieswitch_reference );
                    /* translators: %s: Transaction reference */
                    $order->add_order_note( sprintf( esc_html__( 'Payment via Monieswitch was successful (Transaction Reference: %s)', 'monieswitch' ), esc_html($monieswitch_reference) ) );

                    if ( $this->is_autocomplete_order_enabled ) {
                        $order->update_status( 'completed' );
                    }

                    wp_redirect( $order->get_checkout_order_received_url() );
                    wp_die();


                } else {
                    throw new \Exception( $checkout_initialize['message'] );
                }

            } catch (\Exception $e) {
                $logger->error($e->getMessage(), $context);
                wp_redirect($failed_url);
                wp_die();
            }
        }


    }

    public function generate_checkout_code($data)
    {
        $response = wp_remote_post($this->backend_api_url.'/checkout/initialize', [
            'method'    => 'POST',
            'headers'   => $this->headers,
            'body'      => json_encode($data),
            'timeout'   => 20,
        ]);

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            return ['success' => false, 'message' => $error_message];
        }

        $response_body = wp_remote_retrieve_body($response);
        return json_decode($response_body, true);
    }


    public function verify_payment_transaction_by_reference($reference)
    {
        $url = add_query_arg(['reference' => $reference], $this->backend_api_url.'/checkout/status');
        $response = wp_remote_post($url, [
            'method'    => 'GET',
            'headers'   => $this->headers,
        ]);

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            return ['success' => false, 'message' => $error_message];
        }

        $response_body = wp_remote_retrieve_body($response);
        return json_decode($response_body, true);
    }

    /**
     * @return API
     */
    public static function get_instance()
    {
        static $instance = null;

        if (is_null($instance)) {
            $instance = new self();
        }

        return $instance;
    }
}