<?php
/**
 * Plugin Name: Mon Petit Avis
 * Description: Easily collect and manage certified customer reviews for your WooCommerce store.
 * Version: 1.0.0
 * Author: NovaDigital
 * Author URI: https://mon-petit-avis.fr
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: mon-petit-avis
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * WC requires at least: 7.0
 * WC tested up to: 9.0
 */

if (!defined('ABSPATH')) exit;

final class MonPetAv_Woo {
    const OPTION_KEY = 'monpetav_settings';
    const NONCE_KEY  = 'monpetav_nonce';
    const MENU_SLUG  = 'monpetitavis-settings';
    const CAP        = 'manage_woocommerce';

    /** Base API Mon Petit Avis — à adapter si besoin. */
    const API_BASE   = 'https://api.mon-petit-avis.fr/v1';
    const META_REMOTE_CREATED = '_monpetav_remote_created';

    private static $instance = null;

    public static function instance() {
        if (self::$instance === null) self::$instance = new self();
        return self::$instance;
    }

    private function __construct() {
        // Déclarer la compatibilité HPOS (WooCommerce Custom Order Tables)
        add_action('before_woocommerce_init', [$this, 'declare_compatibility']);

        // WooCommerce requis
        add_action('admin_init', [$this, 'check_dependencies']);

        // Admin
        add_action('admin_menu', [$this, 'add_menu']);
        add_action('admin_notices', [$this, 'display_admin_notices']);
        add_action('admin_post_monpetav_save_settings', [$this, 'handle_save_settings']);
        add_action('admin_post_monpetav_import_products', [$this, 'handle_import_products']);

        // Hooks produits
        add_action('woocommerce_new_product', [$this, 'create_product_by_id'], 10, 1);
        add_action('woocommerce_update_product', [$this, 'sync_product_by_id'], 10, 1);
        add_action('before_delete_post', [$this, 'maybe_mark_product_deleted'], 10, 1);

        // Hooks commandes - Plusieurs hooks pour maximiser les chances
        add_action('woocommerce_checkout_order_processed', [$this, 'send_order_event'], 10, 3);
        add_action('woocommerce_thankyou', [$this, 'send_order_event_by_id'], 10, 1);
        add_action('woocommerce_order_status_completed', [$this, 'send_order_event_by_id'], 10, 1);
        add_action('woocommerce_order_status_processing', [$this, 'send_order_event_by_id'], 10, 1);

        // Option par défaut
        add_action('plugins_loaded', function () {
            $opt = get_option(self::OPTION_KEY);
            if (!$opt) {
                update_option(self::OPTION_KEY, [
                    'api_key'   => '',
                    'sync_on_update' => 'yes',
                ]);
            }
        });
    }

    /* -------------------- Compatibilité WooCommerce -------------------- */
    public function declare_compatibility() {
        if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        }
    }

    /* -------------------- Dépendances -------------------- */
    public function check_dependencies() {
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', function () {
                printf(
                    '<div class="notice notice-error"><p><strong>%s</strong> %s</p></div>',
                    esc_html__('Mon Petit Avis', 'mon-petit-avis'),
                    esc_html__('nécessite WooCommerce. Activez WooCommerce pour utiliser l\'extension.', 'mon-petit-avis')
                );
            });
        }
    }

    public function display_admin_notices() {
        $notices = get_transient('monpetav_admin_notices');
        if ($notices && is_array($notices)) {
            foreach ($notices as $notice) {
                printf('<div class="notice notice-error is-dismissible"><p>%s</p></div>', wp_kses_post($notice));
            }
            delete_transient('monpetav_admin_notices');
        }
    }

    /* -------------------- Admin UI -------------------- */
    public function add_menu() {
        add_submenu_page(
            'woocommerce',
            __('Mon Petit Avis', 'mon-petit-avis'),
            __('Mon Petit Avis', 'mon-petit-avis'),
            self::CAP,
            self::MENU_SLUG,
            [$this, 'render_settings_page']
        );
    }

    private function get_settings() {
        $opt = get_option(self::OPTION_KEY, []);
        if (!isset($opt['sync_on_update'])) {
            $opt['sync_on_update'] = 'yes';
        }
        return $opt;
    }

    public function render_settings_page() {
        if (!current_user_can(self::CAP)) return;

        $s = $this->get_settings();
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Mon Petit Avis – Réglages', 'mon-petit-avis'); ?></h1>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field(self::NONCE_KEY); ?>
                <input type="hidden" name="action" value="monpetav_save_settings" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="monpetav_api_key"><?php esc_html_e('Clé API entreprise', 'mon-petit-avis'); ?></label></th>
                        <td>
                            <input type="password" id="monpetav_api_key" name="api_key" class="regular-text"
                                   value="<?php echo esc_attr($s['api_key'] ?? ''); ?>" placeholder="sk_live_..." />
                            <p class="description">
                                <a href="https://app.mon-petit-avis.fr/settings/company" target="_blank">
                                    <?php esc_html_e('Récupérer votre clé API', 'mon-petit-avis'); ?>
                                </a>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Sync automatique à la mise à jour', 'mon-petit-avis'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="sync_on_update" value="yes" <?php checked(($s['sync_on_update'] ?? 'yes'), 'yes'); ?> />
                                <?php esc_html_e('Synchroniser les produits à chaque création/mise à jour.', 'mon-petit-avis'); ?>
                            </label>
                        </td>
                    </tr>
                </table>

                <?php submit_button(esc_html__('Enregistrer', 'mon-petit-avis')); ?>
            </form>

            <hr/>

            <h2><?php esc_html_e('Importer tous les produits WooCommerce', 'mon-petit-avis'); ?></h2>
            <p><?php esc_html_e('Cliquez pour envoyer tous les produits existants vers Mon Petit Avis.', 'mon-petit-avis'); ?></p>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field(self::NONCE_KEY); ?>
                <input type="hidden" name="action" value="monpetav_import_products" />
                <?php submit_button(esc_html__('Importer maintenant', 'mon-petit-avis'), 'primary', 'submit', false); ?>
            </form>
        </div>
        <?php
    }

    public function handle_save_settings() {
        if (!current_user_can(self::CAP)) wp_die('Unauthorized');
        check_admin_referer(self::NONCE_KEY);

        $api_key  = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $sync_upd = (!empty($_POST['sync_on_update']) && $_POST['sync_on_update'] === 'yes') ? 'yes' : 'no';

        update_option(self::OPTION_KEY, [
            'api_key'   => $api_key,
            'sync_on_update' => $sync_upd,
        ]);

        wp_redirect(add_query_arg(['page' => self::MENU_SLUG, 'updated' => 'true'], admin_url('admin.php')));
        exit;
    }

    public function handle_import_products() {
        if (!current_user_can(self::CAP)) wp_die('Unauthorized');
        check_admin_referer(self::NONCE_KEY);

        $count = $this->bulk_import_products();

        wp_redirect(add_query_arg([
            'page' => self::MENU_SLUG,
            'imported' => (int)$count
        ], admin_url('admin.php')));
        exit;
    }

    /* -------------------- Sync Produits -------------------- */

    /**
     * Crée un produit dans Mon Petit Avis via POST.
     */
    public function create_product_by_id($product_id) {
        $this->log('info', sprintf('create_product_by_id called for product #%d', $product_id));

        $settings = $this->get_settings();
        if (($settings['sync_on_update'] ?? 'yes') !== 'yes') {
            $this->log('info', sprintf('Product #%d creation skipped: sync disabled', $product_id));
            return;
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            $this->log('warning', sprintf('Product #%d creation skipped: product not found', $product_id));
            return;
        }

        if ($this->should_delay_creation($product)) {
            $this->mark_remote_creation_state($product_id, 'pending');
            $this->log('info', sprintf(
                'Product #%d creation delayed: product not ready (status=%s, name=%s)',
                $product_id,
                $product->get_status(),
                $product->get_name()
            ));
            return;
        }

        $payload = $this->build_product_payload($product);
        $response = $this->post_json('/company/products/', $payload);

        if (is_wp_error($response)) {
            $status = $this->get_error_status($response);

            if (in_array($status, [409, 422], true)) {
                $this->mark_remote_creation_state($product_id, 'yes');
                $this->log('info', sprintf('Product #%d already exists remotely (status %d); switching to update', $product_id, $status));
                $this->sync_product_by_id($product_id);
                return;
            }

            $this->mark_remote_creation_state($product_id, 'no');
            $this->log('warning', sprintf('Product #%d creation failed: %s', $product_id, $response->get_error_message()));
            return;
        }

        $this->mark_remote_creation_state($product_id, 'yes');
        $this->log('info', sprintf('Product #%d created successfully', $product_id));
    }

    /**
     * Met à jour un produit dans Mon Petit Avis via PATCH.
     */
    public function sync_product_by_id($product_id) {
        $settings = $this->get_settings();
        if (($settings['sync_on_update'] ?? 'yes') !== 'yes') return;

        $product = wc_get_product($product_id);
        if (!$product) {
            $this->log('warning', sprintf('Product #%d update skipped: product not found', $product_id));
            return;
        }

        $payload = $this->build_product_payload($product);

        $state = $this->get_remote_creation_state($product_id);
        $remote_created = ($state === 'yes');

        if (!$remote_created) {
            if ($this->should_delay_creation($product)) {
                $this->mark_remote_creation_state($product_id, 'pending');
                $this->log('info', sprintf('Product #%d creation still pending (status: %s)', $product_id, $product->get_status()));
                return;
            }

            $create_response = $this->post_json('/company/products/', $payload);

            if (is_wp_error($create_response)) {
                $status = $this->get_error_status($create_response);

                if (in_array($status, [409, 422], true)) {
                    $this->mark_remote_creation_state($product_id, 'yes');
                    $this->log('info', sprintf('Product #%d already exists remotely (status %d); proceeding with update', $product_id, $status));
                    $remote_created = true;
                } else {
                    $this->mark_remote_creation_state($product_id, 'no');
                    $this->log('warning', sprintf('Product #%d creation via sync failed: %s', $product_id, $create_response->get_error_message()));
                    return;
                }
            } else {
                $this->mark_remote_creation_state($product_id, 'yes');
                $this->log('info', sprintf('Product #%d created via sync', $product_id));
                return;
            }
        }

        // Utiliser PATCH avec SKU dans l'URL pour les mises à jour
        $sku = $payload['sku'];
        $update_response = $this->patch_json('/company/products/' . urlencode($sku) . '/', $payload);

        if (is_wp_error($update_response)) {
            $data = $update_response->get_error_data();
            $status = (is_array($data) && isset($data['status'])) ? (int)$data['status'] : 0;

            if ($status === 404) {
                $this->log('warning', sprintf('Product #%d update returned 404, retrying creation', $product_id));
                $retry_response = $this->post_json('/company/products/', $payload);

                if (!is_wp_error($retry_response)) {
                    $this->mark_remote_creation_state($product_id, 'yes');
                    $this->log('info', sprintf('Product #%d recreated after 404', $product_id));
                    return;
                }
            }

            $this->log('warning', sprintf('Product #%d update failed: %s', $product_id, $update_response->get_error_message()));
        }
    }

    /**
     * Marque un produit supprimé (si le SaaS gère la suppression).
     */
    public function maybe_mark_product_deleted($post_id) {
        if (get_post_type($post_id) !== 'product') return;
        // Optionnel: notifie la suppression
        $product = wc_get_product($post_id);
        if (!$product) return;

        $payload = $this->build_product_payload($product, 'archived');
        $sku = $payload['sku'];
        $this->patch_json('/company/products/' . urlencode($sku) . '/', $payload);
        $this->clear_remote_creation_state($post_id);
    }

    private function build_product_payload(WC_Product $product, $forced_status = null) {
        $sku = $product->get_sku();
        if ($sku === '') {
            $sku = 'product-' . $product->get_id();
        }

        $payload = [
            'name'   => $product->get_name(),
            'sku'    => $sku,
            'status' => $forced_status ?: $this->map_product_status($product),
        ];

        // Gestion de l'image
        $image_id = $product->get_image_id();
        if ($image_id) {
            $image_path = get_attached_file($image_id);
            if ($image_path && file_exists($image_path)) {
                $payload['image'] = $image_path; // Chemin vers le fichier
            }
        }

        return $payload;
    }

    private function map_product_status(WC_Product $product) {
        $status = $product->get_status();

        switch ($status) {
            case 'publish':
            case 'private':
                return 'active';
            case 'trash':
                return 'archived';
            case 'draft':
            case 'pending':
            default:
                return 'draft';
        }
    }

    private function bulk_import_products() {
        if (!class_exists('WC_Product_Query')) return 0;
        $page     = 1;
        $per_page = 100;
        $imported = 0;

        do {
            $query = new WC_Product_Query([
                'limit'   => $per_page,
                'page'    => $page,
                'status'  => ['publish','private'],
                'orderby' => 'ID',
                'order'   => 'ASC',
                'return'  => 'objects',
            ]);
            $products = $query->get_products();

            foreach ($products as $product) {
                $payload = $this->build_product_payload($product);
                // Pour l'import en masse, on utilise POST pour créer
                $res = $this->post_json('/company/products/', $payload);
                if (!is_wp_error($res)) {
                    $imported++;
                    $this->mark_remote_creation_state($product->get_id(), 'yes');
                }
            }
            $page++;
        } while (!empty($products) && count($products) === $per_page);

        return $imported;
    }

    /* -------------------- Evénements Commandes -------------------- */

    /**
     * Envoie l'événement "commande passée" à Mon Petit Avis.
     * @param int        $order_id
     * @param array      $posted_data
     * @param WC_Order   $order
     */
    public function send_order_event($order_id, $posted_data, $order) {
        if (!$order instanceof WC_Order) $order = wc_get_order($order_id);
        if (!$order) return;

        $customer_email = $order->get_billing_email();
        $customer_firstname = $order->get_billing_first_name();
        $customer_lastname = $order->get_billing_last_name();

        if (empty($customer_email) || empty($customer_firstname) || empty($customer_lastname)) {
            return;
        }

        $success_count = 0;
        $error_count = 0;

        // Créer une commande Mon Petit Avis pour chaque produit
        foreach ($order->get_items() as $item_id => $item) {
            $product = $item->get_product();
            if (!$product) {
                $error_count++;
                continue;
            }

            $product_id = $product->get_id();
            $quantity = $item->get_quantity();

            // Récupérer le SKU du produit
            $product_sku = $product->get_sku();
            if (empty($product_sku)) {
                $product_sku = 'product-' . $product_id;
            }

            // Créer une commande pour chaque quantité (si qty > 1, créer plusieurs commandes)
            for ($i = 0; $i < $quantity; $i++) {
                $payload = [
                    'product_id' => $product_sku, // Utiliser le SKU au lieu de l'ID numérique
                    'buyer_email' => $customer_email,
                    'buyer_firstname' => $customer_firstname,
                    'buyer_lastname' => $customer_lastname,
                ];

                $this->post_json('/order/', $payload);
            }
        }
    }

    /**
     * Version simplifiée qui prend seulement l'order_id
     */
    public function send_order_event_by_id($order_id) {
        $order = wc_get_order($order_id);
        $this->send_order_event($order_id, [], $order);
    }

    /* -------------------- HTTP Helper -------------------- */

    private function patch_json($path, $data) {
        return $this->send_request($path, $data, 'PATCH');
    }

    private function post_json($path, $data) {
        return $this->send_request($path, $data, 'POST');
    }

    private function send_request($path, $data, $method = 'POST') {
        $settings = $this->get_settings();
        $api_key  = isset($settings['api_key']) ? $settings['api_key'] : '';
        $base     = rtrim(self::API_BASE, '/');

        if (empty($api_key)) {
            $this->admin_log_notice(
                __('Clé API manquante : rendez-vous dans WooCommerce > Mon Petit Avis pour la renseigner.', 'mon-petit-avis')
            );
            return new WP_Error('mpa_no_api_key', __('Clé API manquante', 'mon-petit-avis'));
        }

        $url = $base . $path;

        // Vérifier si on a un fichier à envoyer
        $has_file = false;
        if (isset($data['image']) && is_string($data['image']) && file_exists($data['image'])) {
            $has_file = true;
        }

        if ($has_file) {
            // Envoi multipart/form-data pour les fichiers
            $boundary = wp_generate_uuid4();
            $body = $this->build_multipart_body($data, $boundary);

            $args = [
                'timeout' => 30,
                'headers' => [
                    'Content-Type'  => 'multipart/form-data; boundary=' . $boundary,
                    'X-API-Key'     => $api_key,
                    'User-Agent'    => 'MonPetitAvis-WooCommerce/1.0.0; ' . get_bloginfo('url'),
                ],
                'body' => $body,
            ];
        } else {
            // Envoi JSON classique
            $args = [
                'timeout' => 20,
                'headers' => [
                    'Content-Type'  => 'application/json',
                    'Accept'        => 'application/json',
                    'X-API-Key'     => $api_key,
                    'User-Agent'    => 'MonPetitAvis-WooCommerce/1.0.0; ' . get_bloginfo('url'),
                ],
                'body'    => wp_json_encode($data),
            ];
        }

        // Choisir la méthode HTTP appropriée
        if ($method === 'PATCH') {
            $args['method'] = 'PATCH';
            $resp = wp_remote_request($url, $args);
        } else {
            $resp = wp_remote_post($url, $args);
        }

        if (is_wp_error($resp)) {
            $this->admin_log_notice(
                sprintf(
                    /* translators: %s: error message */
                    __('Mon Petit Avis : erreur réseau : %s', 'mon-petit-avis'),
                    esc_html($resp->get_error_message())
                )
            );
            return $resp;
        }

        $code = wp_remote_retrieve_response_code($resp);
        $body = wp_remote_retrieve_body($resp);

        if ($code < 200 || $code >= 300) {
            $this->admin_log_notice(
                sprintf(
                    /* translators: 1: HTTP status code 2: response body */
                    __('Mon Petit Avis : erreur API (%1$s) : %2$s', 'mon-petit-avis'),
                    $code,
                    esc_html($body)
                )
            );
            return new WP_Error('mpa_bad_status', __('Statut HTTP invalide', 'mon-petit-avis'), ['status' => $code, 'body' => $body]);
        }

        return $resp;
    }

    private function get_remote_creation_state($product_id) {
        $value = get_post_meta($product_id, self::META_REMOTE_CREATED, true);
        return is_string($value) ? $value : '';
    }

    private function mark_remote_creation_state($product_id, $state) {
        update_post_meta($product_id, self::META_REMOTE_CREATED, $state);
    }

    private function clear_remote_creation_state($product_id) {
        delete_post_meta($product_id, self::META_REMOTE_CREATED);
    }

    private function should_delay_creation(WC_Product $product) {
        $status = $product->get_status();
        $name   = trim($product->get_name());

        if ($status === 'auto-draft') {
            return true;
        }

        if ($name === '' || strtoupper($name) === 'AUTO-DRAFT') {
            return true;
        }

        return false;
    }

    private function get_error_status($error) {
        if (!is_wp_error($error)) {
            return 0;
        }

        $data = $error->get_error_data();

        if (is_array($data) && isset($data['status'])) {
            return (int)$data['status'];
        }

        if (is_int($data)) {
            return $data;
        }

        return 0;
    }

    private function log($level, $message) {
        if (function_exists('wc_get_logger')) {
            wc_get_logger()->log($level, $message, ['source' => 'mon-petit-avis']);
        } else {
            error_log(sprintf('[Mon Petit Avis][%s] %s', strtoupper($level), $message));
        }
    }

    private function build_multipart_body($data, $boundary) {
        $body = '';

        foreach ($data as $key => $value) {
            if ($key === 'image' && is_string($value) && file_exists($value)) {
                // Fichier image
                $file_content = file_get_contents($value);
                $filename = basename($value);
                $mime_type = wp_get_image_mime($value) ?: 'application/octet-stream';

                $body .= "--{$boundary}\r\n";
                $body .= "Content-Disposition: form-data; name=\"image\"; filename=\"{$filename}\"\r\n";
                $body .= "Content-Type: {$mime_type}\r\n\r\n";
                $body .= $file_content . "\r\n";
            } else {
                // Champ texte
                $body .= "--{$boundary}\r\n";
                $body .= "Content-Disposition: form-data; name=\"{$key}\"\r\n\r\n";
                $body .= $value . "\r\n";
            }
        }

        $body .= "--{$boundary}--\r\n";

        return $body;
    }

    private function admin_log_notice($msg) {
        // Stocke les erreurs dans un transient pour affichage sur la prochaine requête
        $notices = get_transient('monpetav_admin_notices') ?: [];
        $notices[] = $msg;
        set_transient('monpetav_admin_notices', $notices, 60);
    }
}

MonPetAv_Woo::instance();
