<?php
namespace ModernPress\Widgets;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class ModernPress_Slider_Widget extends \Elementor\Widget_Base
{


    /**
     * Register required scripts and styles for the slider
     */
    public function register_slider_assets()
    {
        // Register Swiper JS
        wp_register_script(
            'swiper',
            MODERNPRESS_PLUGIN_URL . 'assets/vendor/swiper/swiper-bundle.min.js',
            ['jquery'],
            '10.3.1',
            true
        );

        // Register Swiper CSS
        wp_register_style(
            'swiper',
            MODERNPRESS_PLUGIN_URL . 'assets/vendor/swiper/swiper-bundle.min.css',
            [],
            '10.3.1'
        );

        // Register custom slider script that handles both frontend and editor
        wp_register_script(
            'modernpress-slider',
            MODERNPRESS_PLUGIN_URL . 'assets/js/modernpress-slider.js',
            ['jquery', 'swiper', 'elementor-frontend'],
            MODERNPRESS_VERSION,
            true
        );
    }

    public function get_name()
    {
        return 'modernpress_slider_widget';
    }

    public function get_title()
    {
        return esc_html__('ModernPress Slider', 'modernpress-addons');
    }

    public function get_icon()
    {
        return 'eicon-slider-device';
    }

    public function get_categories()
    {
        return ['modernpress'];
    }

    public function get_keywords()
    {
        return ['slider', 'carousel', 'image', 'gallery', 'modernpress'];
    }

    public function get_script_depends()
    {
        return ['swiper', 'modernpress-slider'];
    }

    public function get_style_depends()
    {
        return ['swiper', 'modernpress-slider-styles'];
    }

    protected function register_controls()
    {
        // Content Tab - Slider Items
        $this->start_controls_section(
            'section_slides',
            [
                'label' => esc_html__('Slides', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => esc_html__('Layout', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__('Classic Slider', 'modernpress-addons'),
                    'layout-2' => esc_html__('Modern Slider', 'modernpress-addons'),
                    // 'layout-3' => esc_html__('Creative Slider', 'modernpress-addons'),
                ],
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Slide Title', 'modernpress-addons'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'description',
            [
                'label' => esc_html__('Description', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => esc_html__('Slide Description', 'modernpress-addons'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'button_text',
            [
                'label' => esc_html__('Button Text', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Learn More', 'modernpress-addons'),
            ]
        );

        $repeater->add_control(
            'button_link',
            [
                'label' => esc_html__('Button Link', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => 'https://your-link.com',
                'default' => [
                    'url' => '#',
                    'is_external' => false,
                    'nofollow' => false,
                ],
            ]
        );

        $this->add_control(
            'slides',
            [
                'label' => esc_html__('Slides', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'title' => esc_html__('Slide #1', 'modernpress-addons'),
                        'description' => esc_html__('Click edit button to change this text.', 'modernpress-addons'),
                    ],
                    [
                        'title' => esc_html__('Slide #2', 'modernpress-addons'),
                        'description' => esc_html__('Click edit button to change this text.', 'modernpress-addons'),
                    ],
                ],
                'title_field' => '{{{ title }}}',
            ]
        );

        $this->end_controls_section();

        // Slider Settings
        $this->start_controls_section(
            'section_slider_settings',
            [
                'label' => esc_html__('Slider Settings', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'navigation',
            [
                'label' => esc_html__('Navigation', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'pagination',
            [
                'label' => esc_html__('Pagination', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'pagination_type',
            [
                'label' => esc_html__('Pagination Type', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'bullets',
                'options' => [
                    'bullets' => esc_html__('Bullets', 'modernpress-addons'),
                    'fraction' => esc_html__('Fraction', 'modernpress-addons'),
                    'progressbar' => esc_html__('Progress Bar', 'modernpress-addons'),
                ],
                'condition' => [
                    'pagination' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label' => esc_html__('Autoplay Speed', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 5000,
                'condition' => [
                    'autoplay' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Infinite Loop', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'effect',
            [
                'label' => esc_html__('Transition Effect', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'slide',
                'options' => [
                    'slide' => esc_html__('Slide', 'modernpress-addons'),
                    'fade' => esc_html__('Fade', 'modernpress-addons'),
                    'cube' => esc_html__('Cube', 'modernpress-addons'),
                    'coverflow' => esc_html__('Coverflow', 'modernpress-addons'),
                    'flip' => esc_html__('Flip', 'modernpress-addons'),
                ],
            ]
        );

        $this->add_control(
            'speed',
            [
                'label' => esc_html__('Transition Speed', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 500,
            ]
        );
        $this->add_control(
            'responsive_heading',
            [
                'label' => esc_html__('Responsive Settings', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        
        $this->add_control(
            'slides_to_show',
            [
                'label' => esc_html__('Slides Per View (Desktop)', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 10,
                'step' => 1,
                'default' => 1,
            ]
        );
        
        $this->add_control(
            'slides_to_show_tablet',
            [
                'label' => esc_html__('Slides Per View (Tablet)', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 8,
                'step' => 1,
                'default' => 1,
            ]
        );
        
        $this->add_control(
            'slides_to_show_mobile',
            [
                'label' => esc_html__('Slides Per View (Mobile)', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 1,
                'max' => 4,
                'step' => 1,
                'default' => 1,
            ]
        );
        
        $this->add_control(
            'center_slides',
            [
                'label' => esc_html__('Center Slides', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => '',
                'description' => esc_html__('Enable to center the active slide', 'modernpress-addons'),
            ]
        );
        
        $this->add_control(
            'slides_spacing',
            [
                'label' => esc_html__('Space Between Slides', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0,
                'max' => 100,
                'step' => 5,
                'default' => 30,
            ]
        );
        $this->add_control(
            'vertical_slider_heading',
            [
                'label' => esc_html__('Vertical Slider Settings', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'layout' => 'layout-2',
                ],
            ]
        );
        
        $this->add_control(
            'mousewheel',
            [
                'label' => esc_html__('Enable Mousewheel', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
                'condition' => [
                    'layout' => 'layout-2',
                ],
            ]
        );
        
        $this->add_responsive_control(
            'vertical_slider_height',
            [
                'label' => esc_html__('Slider Height', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh', '%'],
                'range' => [
                    'px' => [
                        'min' => 200,
                        'max' => 1200,
                        'step' => 10,
                    ],
                    'vh' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 5,
                    ],
                ],
                'default' => [
                    'unit' => 'vh',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-slider-layout-2' => 'height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'layout' => 'layout-2',
                ],
            ]
        );
        $this->end_controls_section();

        // Style Tab - Slider
        $this->start_controls_section(
            'section_style_slider',
            [
                'label' => esc_html__('Slider', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'slider_height',
            [
                'label' => esc_html__('Height', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 200,
                        'max' => 1200,
                    ],
                    'vh' => [
                        'min' => 20,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-slider' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'slider_border_radius',
            [
                'label' => esc_html__('Border Radius', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-slider' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab - Navigation
        $this->start_controls_section(
            'section_style_navigation',
            [
                'label' => esc_html__('Navigation', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'navigation' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'arrows_color',
            [
                'label' => esc_html__('Arrows Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#0D3FB0',
                'selectors' => [
                    '{{WRAPPER}} .swiper-button-next path, {{WRAPPER}} .swiper-button-prev path' => 'stroke: {{VALUE}};',
                    '{{WRAPPER}} .swiper-button-next, {{WRAPPER}} .swiper-button-prev' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'arrows_size',
            [
                'label' => esc_html__('Arrows Size', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 80,
                    ],
                ],
                'default' => [
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .swiper-button-next svg, {{WRAPPER}} .swiper-button-prev svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .swiper-button-next, {{WRAPPER}} .swiper-button-prev' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'arrows_background_color',
            [
                'label' => esc_html__('Arrows Background', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#FFFFFF',
                'selectors' => [
                    '{{WRAPPER}} .swiper-button-next rect, {{WRAPPER}} .swiper-button-prev rect' => 'fill: {{VALUE}};',
                    '{{WRAPPER}} .swiper-button-next, {{WRAPPER}} .swiper-button-prev' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab - Pagination
        $this->start_controls_section(
            'section_style_pagination',
            [
                'label' => esc_html__('Pagination', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'pagination' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'pagination_active_color',
            [
                'label' => esc_html__('Pagination Active Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#0D3FB0',
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullet-active' => 'background-color: {{VALUE}} !important; border-color: {{VALUE}} !important;',
                ],
                'condition' => [
                    'pagination' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'pagination_inactive_color',
            [
                'label' => esc_html__('Pagination Inactive Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#cccccc',
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullet:not(.swiper-pagination-bullet-active)' => 'background-color: {{VALUE}} !important;',
                ],
                'condition' => [
                    'pagination' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab - Content
        $this->start_controls_section(
            'section_style_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .slide-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .slide-title',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__('Description Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .slide-description' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .slide-description',
            ]
        );

        // Button style
        $this->add_control(
            'button_heading',
            [
                'label' => esc_html__('Button', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'button_color',
            [
                'label' => esc_html__('Button Text Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .slide-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background',
            [
                'label' => esc_html__('Button Background', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .slide-button' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'selector' => '{{WRAPPER}} .slide-button',
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        // Enqueue the styles for this slider instance
        // Add slider wrapper with unique ID for targeting
        $slider_id = 'modernpress-slider-' . $this->get_id();
        // modernpress_slider_enqueue_styles($slider_id, $settings);
        echo '<div id="' . esc_attr($slider_id) . '" class="modernpress-slider-wrapper" data-slider-id="' . esc_attr($this->get_id()) . '" data-settings="' . esc_attr(wp_json_encode($settings)) . '">';

        // Get the layout file path
        $layout_file = MODERNPRESS_PLUGIN_DIR . 'includes/widgets/includes/slider-widget/' . $settings['layout'] . '.php';

        if (file_exists($layout_file)) {
            include $layout_file;
        } else {
            echo esc_html__('Layout file not found.', 'modernpress-addons');
        }

        echo '</div>'; // Close slider wrapper

        // Enqueue required assets
        wp_enqueue_style('swiper');
        wp_enqueue_script('swiper');
        wp_enqueue_script('modernpress-slider');
    }
}