<?php
namespace ModernPress\Widgets;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class ModernPress_Posts_Widget extends \Elementor\Widget_Base
{

    public function get_name()
    {
        return 'modernpress_posts_widget';
    }

    public function get_title()
    {
        return esc_html__('ModernPress Posts', 'modernpress-addons');
    }

    public function get_icon()
    {
        return 'eicon-post-list';
    }

    public function get_categories()
    {
        return ['modernpress'];
    }

    public function get_keywords()
    {
        return ['posts', 'blog', 'grid', 'slider', 'modernpress'];
    }

    public function get_script_depends()
    {
        return ['swiper'];
    }

    public function get_style_depends()
    {
        return ['swiper'];
    }

    protected function register_controls()
    {
        // Query Section
        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_html__('Query Settings', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        // Category Selection
        $categories = get_categories(['hide_empty' => true]);
        $category_options = [];
        foreach ($categories as $category) {
            $category_options[$category->term_id] = $category->name;
        }

        $this->add_control(
            'selected_categories',
            [
                'label' => esc_html__('Select Categories', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $category_options,
                'label_block' => true,
            ]
        );

        // Posts Per Page
        $this->add_control(
            'posts_per_page',
            [
                'label' => esc_html__('Posts Per Page', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 3,
                'min' => 1,
                'max' => 50,
            ]
        );

        // Order By
        $this->add_control(
            'orderby',
            [
                'label' => esc_html__('Order By', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'date',
                'options' => [
                    'date' => esc_html__('Date', 'modernpress-addons'),
                    'title' => esc_html__('Title', 'modernpress-addons'),
                    'modified' => esc_html__('Last Modified', 'modernpress-addons'),
                    'rand' => esc_html__('Random', 'modernpress-addons'),
                ],
            ]
        );

        // Order
        $this->add_control(
            'order',
            [
                'label' => esc_html__('Order', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'DESC',
                'options' => [
                    'DESC' => esc_html__('Descending', 'modernpress-addons'),
                    'ASC' => esc_html__('Ascending', 'modernpress-addons'),
                ],
            ]
        );

        $this->end_controls_section();

        // Layout Settings
        $this->start_controls_section(
            'section_layout',
            [
                'label' => esc_html__('Layout Settings', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        // Layout Selection
        $this->add_control(
            'layout',
            [
                'label' => esc_html__('Layout', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'modernpress-addons'),
                    'layout-2' => esc_html__('Layout 2', 'modernpress-addons'),
                    'layout-3' => esc_html__('Layout 3', 'modernpress-addons'),
                    'layout-4' => esc_html__('Layout 4', 'modernpress-addons'),
                    'layout-5' => esc_html__('Layout 5', 'modernpress-addons'),
                    'layout-6' => esc_html__('Layout 6', 'modernpress-addons'),
                    'layout-7' => esc_html__('Layout 7', 'modernpress-addons'),
                    'layout-8' => esc_html__('Layout 8', 'modernpress-addons'),
                    'layout-9' => esc_html__('Layout 9', 'modernpress-addons'),
                    'layout-10' => esc_html__('Layout 10', 'modernpress-addons'),
                    'layout-11' => esc_html__('Layout 11', 'modernpress-addons'),
                ],
            ]
        );

        // Enable Slider
        $this->add_control(
            'enable_slider',
            [
                'label' => esc_html__('Enable Slider', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'modernpress-addons'),
                'label_off' => esc_html__('No', 'modernpress-addons'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->end_controls_section();

        // Slider Settings
        $this->start_controls_section(
            'section_slider',
            [
                'label' => esc_html__('Slider Settings', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'enable_slider' => 'yes',
                ],
            ]
        );

        // Slides Per View
        $this->add_control(
            'slides_per_view',
            [
                'label' => esc_html__('Slides Per View', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '3',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                ],
            ]
        );

        // Slides to Scroll
        $this->add_control(
            'slides_to_scroll',
            [
                'label' => esc_html__('Slides to Scroll', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                ],
            ]
        );

        // Autoplay
        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Autoplay Speed
        $this->add_control(
            'autoplay_speed',
            [
                'label' => esc_html__('Autoplay Speed', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 3000,
                'condition' => [
                    'autoplay' => 'yes',
                ],
            ]
        );

        // Infinite Loop
        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Infinite Loop', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Navigation
        $this->add_control(
            'navigation',
            [
                'label' => esc_html__('Navigation', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        // Pagination
        $this->add_control(
            'pagination',
            [
                'label' => esc_html__('Pagination', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();
        // Style Section
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style Settings', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        // Title Color
        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-title a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-post-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        // Title Typography
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => esc_html__('Title Typography', 'modernpress-addons'),
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-title a',
                    '{{WRAPPER}} .modernpress-post-title ',
                ],
            ]
        );

        // Description Color
        $this->add_control(
            'description_color',
            [
                'label' => esc_html__('Description Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-excerpt' => 'color: {{VALUE}};',
                ],
            ]
        );

        // Description Typography
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'label' => esc_html__('Description Typography', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-post-excerpt',
            ]
        );
        // meta Color
        $this->add_control(
            'meta_color',
            [
                'label' => esc_html__('Meta Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-author-details span' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'meta_typography',
                'label' => esc_html__('Meta Typography', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-post-author-details span',
            ]
        );
        // tag Color
        $this->add_control(
            'tag_color',
            [
                'label' => esc_html__('Tag Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-categories a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-post-categories .category-tag' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'tag_hover_color',
            [
                'label' => esc_html__('Tag Hover Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-categories a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-post-categories .category-tag:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        // Description Typography
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'tag_typography',
                'label' => esc_html__('Tag Typography', 'modernpress-addons'),
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-categories a',
                    '{{WRAPPER}} .modernpress-post-categories .category-tag',
                ],
            ]
        );
        $this->add_control(
            'tag_background_color',
            [
                'label' => esc_html__('Tag Background Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-categories a' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-post-categories .category-tag' => 'background-color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'tag_hover_background_color',
            [
                'label' => esc_html__('Tag Hover Background Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-categories a:hover' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .modernpress-post-categories .category-tag:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        // link Color
        $this->add_control(
            'link_color',
            [
                'label' => esc_html__('Link Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post-read-more a' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'link_typography',
                'label' => esc_html__('Link Typography', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-post-read-more a',
            ]
        );

        // Background Color
        $this->add_control(
            'background_color',
            [
                'label' => esc_html__('Background Color', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        // Padding
        $this->add_responsive_control(
            'padding',
            [
                'label' => esc_html__('Padding', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Margin
        $this->add_responsive_control(
            'margin',
            [
                'label' => esc_html__('Margin', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // Border
        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'label' => esc_html__('Border', 'modernpress-addons'),
                'selector' => '{{WRAPPER}} .modernpress-post',
            ]
        );

        // Border Radius
        $this->add_responsive_control(
            'border_radius',
            [
                'label' => esc_html__('Border Radius', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .modernpress-post' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $is_slider = $settings['enable_slider'] === 'yes';

        // Query arguments
        $query_args = [
            'post_type' => 'post',
            'posts_per_page' => $settings['posts_per_page'],
            'orderby' => $settings['orderby'],
            'order' => $settings['order'],
        ];

        // Add category filter if categories are selected
        if (!empty($settings['selected_categories'])) {
            $query_args['category__in'] = $settings['selected_categories'];
        }

        $query = new \WP_Query($query_args);

        // Slider wrapper start
        if ($is_slider) {
            echo '<div class="modernpress-posts-slider swiper">';
            echo '<div class="swiper-wrapper">';
        } else {
            // Create specific grid class for each layout
            $layout_number = str_replace('layout-', '', $settings['layout']);
            echo '<div class="modernpress-posts-grid modernpress-posts-grid-' . esc_attr($layout_number) . '">';
        }

        // Loop through posts
        if ($query->have_posts()) {
            $post_counter = 0;
            while ($query->have_posts()) {
                $query->the_post();
                $post_counter++;

                if ($is_slider) {
                    echo '<div class="swiper-slide">';
                }

                // Get the layout file path
                $layout_file = MODERNPRESS_PLUGIN_DIR . 'includes/widgets/includes/posts-widget/' . $settings['layout'] . '.php';

                if (file_exists($layout_file)) {
                    // Pass post data to layout
                    $post_settings = [
                        'post_id' => get_the_ID(),
                        'title' => get_the_title(),
                        'date' => get_the_date(),
                        'categories' => get_the_category(),
                        'excerpt' => get_the_excerpt(),
                        'image' => get_the_post_thumbnail_url() ?: \Elementor\Utils::get_placeholder_image_src(),
                        'permalink' => get_permalink(),
                        'post_counter' => $post_counter // Add this line
                    ];

                    // Set featured flag for first post in layout-10
                    if ($settings['layout'] === 'layout-10') {
                        $featured = ($post_counter === 1);
                    }
                    // Pass the counter to the layout
                    $post_settings['post_counter'] = $post_counter;
                    // Include the layout file with local settings
                    include $layout_file;
                }

                if ($is_slider) {
                    echo '</div>'; // End swiper-slide
                }
            }
            wp_reset_postdata();
        }

        // Slider wrapper end
        if ($is_slider) {
            echo '</div>'; // End swiper-wrapper

            // Add navigation
            if ($settings['navigation'] === 'yes') {
                echo '<div class="swiper-button-next"></div>';
                echo '<div class="swiper-button-prev"></div>';
            }

            // Add pagination
            if ($settings['pagination'] === 'yes') {
                echo '<div class="swiper-pagination"></div>';
            }

            echo '</div>'; // End modernpress-posts-slider

            // Add Swiper initialization script
            ?>

            <?php
        } else {
            echo '</div>'; // End modernpress-posts-grid
        }

        wp_enqueue_style(
            'swiper',
            plugin_dir_url(__FILE__) . 'assets/css/swiper-bundle.min.css',
            array(),
            '10.3.1'
        );

        wp_enqueue_script(
            'swiper',
            plugin_dir_url(__FILE__) . 'assets/js/swiper-bundle.min.js',
            array('jquery'),
            '10.3.1',
            true
        );
    }
}