<?php
namespace ModernPress\Widgets;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class ModernPress_Client_Logos_Widget extends \Elementor\Widget_Base
{


    public function get_name()
    {
        return 'modernpress_client_logos_widget';
    }

    public function get_title()
    {
        return esc_html__('ModernPress Client Logos', 'modernpress-addons');
    }

    public function get_icon()
    {
        return 'eicon-logo';
    }

    public function get_categories()
    {
        return ['modernpress'];
    }

    public function get_keywords()
    {
        return ['logo', 'client', 'brand', 'modernpress'];
    }
    public function get_style_depends()
    {
        return ['modernpress-client-logos', 'modernpress-client-logos-layout-2', 'modernpress-client-logos-layout-3', 'modernpress-client-logos-layout-4', 'modernpress-client-logos-layout-5'];
    }

    protected function add_inline_style($custom_css, $layout = 'layout-1')
    {
        // This is a helper method to add inline styles
        $custom_css = trim($custom_css);
        if (empty($custom_css)) {
            return;
        }

        $custom_css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $custom_css);
        $custom_css = str_replace(["\r\n", "\r", "\n", "\t", '  ', '    ', '    '], '', $custom_css);

        $widget_id = esc_attr($this->get_id()); // Escape the widget ID

        // Determine which stylesheet to attach the inline style to based on layout
        $stylesheet_handle = 'modernpress-client-logos';
        if ($layout === 'layout-2') {
            $stylesheet_handle = 'modernpress-client-logos-layout-2';
        } elseif ($layout === 'layout-3') {
            $stylesheet_handle = 'modernpress-client-logos-layout-3';
        } elseif ($layout === 'layout-4') {
            $stylesheet_handle = 'modernpress-client-logos-layout-4';
        } elseif ($layout === 'layout-5') {
            $stylesheet_handle = 'modernpress-client-logos-layout-5';
        }

        // Properly escape both variables when used in the output
        wp_add_inline_style($stylesheet_handle, "#elementor-element-" . esc_attr($widget_id) . " {" . esc_html($custom_css) . "}");
    }


    protected function register_controls()
    {
        // Content Tab
        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Content', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'layout',
            [
                'label' => esc_html__('Layout', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'layout-1',
                'options' => [
                    'layout-1' => esc_html__('Simple Grid', 'modernpress-addons'),
                    'layout-2' => esc_html__('Strip Layout', 'modernpress-addons'),
                    'layout-3' => esc_html__('Bordered Grid', 'modernpress-addons'),
                    'layout-4' => esc_html__('Spaced Grid', 'modernpress-addons'),
                    'layout-5' => esc_html__('Alternating Background', 'modernpress-addons'),
                ],
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'logo',
            [
                'label' => esc_html__('Logo', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'company_name',
            [
                'label' => esc_html__('Company Name', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Company Name', 'modernpress-addons'),
                'placeholder' => esc_html__('Enter company name', 'modernpress-addons'),
            ]
        );

        $repeater->add_control(
            'link',
            [
                'label' => esc_html__('Website URL', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::URL,
                'placeholder' => esc_html__('https://your-link.com', 'modernpress-addons'),
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
            ]
        );

        $repeater->add_control(
            'grayscale',
            [
                'label' => esc_html__('Grayscale Effect', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'modernpress-addons'),
                'label_off' => esc_html__('No', 'modernpress-addons'),
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'logos',
            [
                'label' => esc_html__('Client Logos', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'company_name' => esc_html__('Company #1', 'modernpress-addons'),
                    ],
                    [
                        'company_name' => esc_html__('Company #2', 'modernpress-addons'),
                    ],
                ],
                'title_field' => '{{{ company_name }}}',
            ]
        );

        // Replace the simple columns control with responsive control
        $this->add_responsive_control(
            'columns',
            [
                'label' => esc_html__('Columns', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '5',
                'tablet_default' => '3',
                'mobile_default' => '2',
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                    '5' => '5',
                    '6' => '6',
                ],
                'selectors' => [
                    '{{WRAPPER}} .logos-grid' => '--desktop-columns: {{VALUE}};',
                    '{{WRAPPER}} .logos-strip' => '--desktop-columns: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'modernpress-addons'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'logo_width',
            [
                'label' => esc_html__('Logo Width', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 50,
                        'max' => 500,
                        'step' => 10,
                    ],
                    '%' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 5,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 150,
                ],
                'tablet_default' => [
                    'unit' => 'px',
                    'size' => 120,
                ],
                'mobile_default' => [
                    'unit' => 'px',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .client-logo img' => 'max-width: {{SIZE}}{{UNIT}}; width: 100%;',
                ],
            ]
        );

        $this->add_responsive_control(
            'logo_height',
            [
                'label' => esc_html__('Logo Height', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 20,
                        'max' => 300,
                        'step' => 10,
                    ],
                    'em' => [
                        'min' => 1,
                        'max' => 20,
                        'step' => 0.5,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .client-logo img' => 'max-height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'logo_hover_effect',
            [
                'label' => esc_html__('Hover Effect', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'none',
                'options' => [
                    'none' => esc_html__('None', 'modernpress-addons'),
                    'scale' => esc_html__('Scale', 'modernpress-addons'),
                    'opacity' => esc_html__('Opacity', 'modernpress-addons'),
                    'color' => esc_html__('Color', 'modernpress-addons'),
                ],
            ]
        );

        $this->add_control(
            'logo_padding',
            [
                'label' => esc_html__('Logo Padding', 'modernpress-addons'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .logo-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '20',
                    'right' => '20',
                    'bottom' => '20',
                    'left' => '20',
                    'unit' => 'px',
                    'isLinked' => true,
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        // Add responsive column classes based on settings
        // Ensure column settings exist to avoid undefined array key warnings
        $desktop_columns = isset($settings['columns']) ? $settings['columns'] : '4';
        $tablet_columns = isset($settings['columns_tablet']) ? $settings['columns_tablet'] : '3';
        $mobile_columns = isset($settings['columns_mobile']) ? $settings['columns_mobile'] : '2';

        $this->add_render_attribute('logos-container', 'class', 'modernpress-client-logos');
        $this->add_render_attribute('logos-container', 'class', 'modernpress-client-logos-' . $settings['layout']);
        $this->add_render_attribute('logos-container', 'data-desktop-columns', $desktop_columns);
        $this->add_render_attribute('logos-container', 'data-tablet-columns', $tablet_columns);
        $this->add_render_attribute('logos-container', 'data-mobile-columns', $mobile_columns);

        // Get the layout file path
        $layout_file = MODERNPRESS_PLUGIN_DIR . 'includes/widgets/includes/client-logos-widget/' . $settings['layout'] . '.php';

        if (file_exists($layout_file)) {
            include $layout_file;
        } else {
            echo esc_html__('Layout file not found.', 'modernpress-addons');
        }

        function register_client_logos_styles()
        {
            wp_register_style(
                'modernpress-client-logos',
                plugins_url('assets/css/client-logos.css', __FILE__),
                [],
                '1.0.0'
            );
        }
        add_action('wp_enqueue_scripts', 'register_client_logos_styles');
        add_action('elementor/frontend/after_enqueue_styles', 'register_client_logos_styles');


    }
}